#!/usr/bin/env python3
"""
Cost Calculator for Restaurant Chatbot
Calculates estimated costs based on usage patterns
"""

# OpenAI Pricing (as of October 2025)
GPT4O_MINI_INPUT_PRICE = 0.150 / 1_000_000  # per token
GPT4O_MINI_OUTPUT_PRICE = 0.600 / 1_000_000  # per token
WHISPER_PRICE = 0.006  # per minute

# Average token usage (based on actual logs)
AVG_INPUT_TOKENS = 13_700  # System Prompt + User Message + History
AVG_OUTPUT_TOKENS = 150    # AI Reply
AVG_TOTAL_TOKENS = AVG_INPUT_TOKENS + AVG_OUTPUT_TOKENS

# Average voice message duration
AVG_VOICE_DURATION_MINUTES = 0.5  # 30 seconds


def calculate_text_message_cost():
    """Calculate cost for a single text message"""
    input_cost = AVG_INPUT_TOKENS * GPT4O_MINI_INPUT_PRICE
    output_cost = AVG_OUTPUT_TOKENS * GPT4O_MINI_OUTPUT_PRICE
    total_cost = input_cost + output_cost
    return total_cost


def calculate_voice_message_cost():
    """Calculate cost for a single voice message"""
    transcription_cost = AVG_VOICE_DURATION_MINUTES * WHISPER_PRICE
    text_processing_cost = calculate_text_message_cost()
    total_cost = transcription_cost + text_processing_cost
    return total_cost


def calculate_monthly_cost(orders_per_day, messages_per_order=12, voice_percentage=0):
    """
    Calculate monthly cost based on usage
    
    Args:
        orders_per_day: Number of orders per day
        messages_per_order: Average messages per order (default: 12)
        voice_percentage: Percentage of voice messages (0-100)
    
    Returns:
        dict with cost breakdown
    """
    # Calculate total messages
    messages_per_day = orders_per_day * messages_per_order
    messages_per_month = messages_per_day * 30
    
    # Calculate voice vs text split
    voice_messages = int(messages_per_month * (voice_percentage / 100))
    text_messages = messages_per_month - voice_messages
    
    # Calculate costs
    text_cost = text_messages * calculate_text_message_cost()
    voice_cost = voice_messages * calculate_voice_message_cost()
    total_cost = text_cost + voice_cost
    
    return {
        'orders_per_day': orders_per_day,
        'messages_per_day': messages_per_day,
        'messages_per_month': messages_per_month,
        'text_messages': text_messages,
        'voice_messages': voice_messages,
        'text_cost': text_cost,
        'voice_cost': voice_cost,
        'total_cost': total_cost,
        'cost_per_order': total_cost / (orders_per_day * 30)
    }


def print_cost_breakdown(result):
    """Print formatted cost breakdown"""
    print("\n" + "="*60)
    print("📊 تقدير التكلفة الشهرية")
    print("="*60)
    print(f"\n📈 الاستخدام:")
    print(f"  • الطلبات يومياً: {result['orders_per_day']}")
    print(f"  • الرسائل يومياً: {result['messages_per_day']:,}")
    print(f"  • الرسائل شهرياً: {result['messages_per_month']:,}")
    print(f"  • رسائل نصية: {result['text_messages']:,}")
    print(f"  • رسائل صوتية: {result['voice_messages']:,}")
    
    print(f"\n💰 التكلفة:")
    print(f"  • تكلفة الرسائل النصية: ${result['text_cost']:.2f}")
    print(f"  • تكلفة الرسائل الصوتية: ${result['voice_cost']:.2f}")
    print(f"  • التكلفة الإجمالية: ${result['total_cost']:.2f}")
    print(f"  • تكلفة الطلب الواحد: ${result['cost_per_order']:.4f}")
    print("="*60 + "\n")


def compare_scenarios():
    """Compare different restaurant sizes"""
    scenarios = [
        ("مطعم صغير", 50, 0),
        ("مطعم صغير (30% صوتي)", 50, 30),
        ("مطعم متوسط", 150, 0),
        ("مطعم متوسط (30% صوتي)", 150, 30),
        ("مطعم كبير", 300, 0),
        ("مطعم كبير (30% صوتي)", 300, 30),
    ]
    
    print("\n" + "="*80)
    print("📊 مقارنة السيناريوهات المختلفة")
    print("="*80)
    print(f"\n{'السيناريو':<30} {'الطلبات/يوم':<15} {'الرسائل/شهر':<15} {'التكلفة الشهرية':<15}")
    print("-"*80)
    
    for name, orders, voice_pct in scenarios:
        result = calculate_monthly_cost(orders, voice_percentage=voice_pct)
        print(f"{name:<30} {orders:<15} {result['messages_per_month']:<15,} ${result['total_cost']:<14.2f}")
    
    print("="*80 + "\n")


def calculate_roi():
    """Calculate ROI compared to human agent"""
    print("\n" + "="*60)
    print("💰 العائد على الاستثمار (ROI)")
    print("="*60)
    
    # Human agent costs
    agent_salary = 1000  # monthly
    agent_orders_per_day = 100
    
    # Bot costs (medium restaurant)
    bot_result = calculate_monthly_cost(150, voice_percentage=30)
    bot_cost = bot_result['total_cost']
    bot_orders_per_day = 150
    
    savings = agent_salary - bot_cost
    savings_percentage = (savings / agent_salary) * 100
    
    print(f"\n👤 موظف خدمة عملاء:")
    print(f"  • الراتب الشهري: ${agent_salary:.2f}")
    print(f"  • الطلبات المعالجة يومياً: {agent_orders_per_day}")
    print(f"  • ساعات العمل: 8 ساعات/يوم")
    
    print(f"\n🤖 البوت:")
    print(f"  • التكلفة الشهرية: ${bot_cost:.2f}")
    print(f"  • الطلبات المعالجة يومياً: {bot_orders_per_day}")
    print(f"  • ساعات العمل: 24/7")
    
    print(f"\n💵 التوفير:")
    print(f"  • التوفير الشهري: ${savings:.2f}")
    print(f"  • نسبة التوفير: {savings_percentage:.1f}%")
    print(f"  • التوفير السنوي: ${savings * 12:.2f}")
    print("="*60 + "\n")


def interactive_calculator():
    """Interactive cost calculator"""
    print("\n" + "="*60)
    print("🧮 حاسبة التكلفة التفاعلية")
    print("="*60)
    
    try:
        orders = int(input("\n📊 كم عدد الطلبات يومياً؟ "))
        messages = int(input("💬 كم متوسط الرسائل لكل طلب؟ (افتراضي: 12) ") or "12")
        voice_pct = int(input("🎤 كم نسبة الرسائل الصوتية؟ (0-100) ") or "0")
        
        result = calculate_monthly_cost(orders, messages, voice_pct)
        print_cost_breakdown(result)
        
        # Show yearly projection
        yearly_cost = result['total_cost'] * 12
        print(f"📅 التكلفة السنوية المتوقعة: ${yearly_cost:.2f}")
        
    except ValueError:
        print("❌ خطأ: يرجى إدخال أرقام صحيحة")
    except KeyboardInterrupt:
        print("\n\n👋 تم الإلغاء")


def main():
    """Main function"""
    print("\n" + "="*60)
    print("💰 حاسبة تكلفة Restaurant Chatbot")
    print("="*60)
    
    # Show single message costs
    text_cost = calculate_text_message_cost()
    voice_cost = calculate_voice_message_cost()
    
    print(f"\n📝 تكلفة الرسالة الواحدة:")
    print(f"  • رسالة نصية: ${text_cost:.6f} (~${text_cost:.3f})")
    print(f"  • رسالة صوتية: ${voice_cost:.6f} (~${voice_cost:.3f})")
    
    # Compare scenarios
    compare_scenarios()
    
    # Calculate ROI
    calculate_roi()
    
    # Interactive calculator
    while True:
        choice = input("\n🔢 هل تريد حساب تكلفة مخصصة؟ (y/n): ").lower()
        if choice == 'y':
            interactive_calculator()
        else:
            break
    
    print("\n✅ شكراً لاستخدام حاسبة التكلفة!\n")


if __name__ == "__main__":
    main()

