"""
Audit REAL order query issues (not false positives)
Focus on actual database column mismatches
"""
import os
import re
from pathlib import Path

def print_section(title):
    """Print section header"""
    print("\n" + "="*80)
    print(f"  {title}")
    print("="*80)

def check_file_for_real_issues(file_path):
    """Check file for REAL database column issues"""
    issues = []
    
    try:
        with open(file_path, 'r', encoding='utf-8') as f:
            content = f.read()
            lines = content.split('\n')
        
        # Only check if file uses orders table
        if "table('orders')" not in content and 'table("orders")' not in content:
            return issues
        
        # REAL ISSUE 1: .select() with 'order_number' on orders table
        # Pattern: table('orders').select(...'order_number'...)
        pattern = r"table\(['\"]orders['\"]\)[^\n]*\.select\(['\"]([^'\"]+)['\"]\)"
        for match in re.finditer(pattern, content, re.MULTILINE):
            columns_str = match.group(1)
            if 'order_number' in columns_str:
                line_num = content[:match.start()].count('\n') + 1
                issues.append({
                    'line': line_num,
                    'type': 'DATABASE_COLUMN',
                    'severity': 'HIGH',
                    'issue': "Using 'order_number' in .select() but column is 'order_id'",
                    'fix': "Change 'order_number' to 'order_id'",
                    'context': lines[line_num-1].strip()[:100]
                })
        
        # REAL ISSUE 2: .eq('order_number', ...) on orders table
        pattern = r"table\(['\"]orders['\"]\)[^\n]*\.eq\(['\"]order_number['\"]\s*,"
        for match in re.finditer(pattern, content, re.MULTILINE):
            line_num = content[:match.start()].count('\n') + 1
            issues.append({
                'line': line_num,
                'type': 'DATABASE_COLUMN',
                'severity': 'HIGH',
                'issue': "Using .eq('order_number', ...) but column is 'order_id'",
                'fix': "Change to .eq('order_id', ...)",
                'context': lines[line_num-1].strip()[:100]
            })
        
        # REAL ISSUE 3: Accessing order['order_number'] after querying orders table
        # This is trickier - need to find patterns like:
        # order_response = supabase.table('orders')...
        # order = order_response.data[0]
        # order['order_number']  ← This is the issue
        
        # Find all places where we query orders table and store in variable
        order_query_pattern = r"(\w+)\s*=\s*[^\n]*table\(['\"]orders['\"]\)[^\n]*\.execute\(\)"
        order_vars = []
        for match in re.finditer(order_query_pattern, content):
            var_name = match.group(1)
            order_vars.append(var_name)
        
        # Now find accesses to order['order_number'] or order.get('order_number')
        for var_name in order_vars:
            # Check for dict access
            pattern = rf"{var_name}(?:\.data\[0\]|\[0\])\[['\"]order_number['\"]\]"
            for match in re.finditer(pattern, content):
                line_num = content[:match.start()].count('\n') + 1
                issues.append({
                    'line': line_num,
                    'type': 'DATABASE_COLUMN',
                    'severity': 'HIGH',
                    'issue': f"Accessing order['order_number'] but column is 'order_id'",
                    'fix': "Change to order['order_id']",
                    'context': lines[line_num-1].strip()[:100]
                })
            
            # Check for .get() access
            pattern = rf"{var_name}(?:\.data\[0\]|\[0\])\.get\(['\"]order_number['\"]\s*[,\)]"
            for match in re.finditer(pattern, content):
                line_num = content[:match.start()].count('\n') + 1
                issues.append({
                    'line': line_num,
                    'type': 'DATABASE_COLUMN',
                    'severity': 'HIGH',
                    'issue': f"Accessing order.get('order_number') but column is 'order_id'",
                    'fix': "Change to order.get('order_id')",
                    'context': lines[line_num-1].strip()[:100]
                })
        
        # REAL ISSUE 4: .select() with 'total' instead of 'total_omr'
        pattern = r"table\(['\"]orders['\"]\)[^\n]*\.select\(['\"]([^'\"]+)['\"]\)"
        for match in re.finditer(pattern, content, re.MULTILINE):
            columns_str = match.group(1)
            # Check if 'total' is used but not 'total_omr'
            if re.search(r'\btotal\b', columns_str) and 'total_omr' not in columns_str:
                line_num = content[:match.start()].count('\n') + 1
                issues.append({
                    'line': line_num,
                    'type': 'DATABASE_COLUMN',
                    'severity': 'HIGH',
                    'issue': "Using 'total' in .select() but column is 'total_omr'",
                    'fix': "Change 'total' to 'total_omr'",
                    'context': lines[line_num-1].strip()[:100]
                })
        
        # REAL ISSUE 5: Accessing order['total'] after querying orders
        for var_name in order_vars:
            # Check for dict access to 'total'
            pattern = rf"{var_name}(?:\.data\[0\]|\[0\])\[['\"]total['\"]\]"
            for match in re.finditer(pattern, content):
                line_num = content[:match.start()].count('\n') + 1
                # Make sure it's not 'total_omr'
                if 'total_omr' not in lines[line_num-1]:
                    issues.append({
                        'line': line_num,
                        'type': 'DATABASE_COLUMN',
                        'severity': 'HIGH',
                        'issue': f"Accessing order['total'] but column is 'total_omr'",
                        'fix': "Change to order['total_omr']",
                        'context': lines[line_num-1].strip()[:100]
                    })
        
    except Exception as e:
        print(f"   ❌ Error reading {file_path}: {e}")
    
    return issues

def audit_real_issues():
    """Audit for REAL database column issues only"""
    
    print_section("🔍 AUDITING FOR REAL DATABASE COLUMN ISSUES")
    print("\nℹ️  This audit focuses on actual database column mismatches")
    print("   (Not internal variables like session.get('items'))")
    
    # Files to check
    files_to_check = [
        'app/routes/webhook.py',
        'app/services/session_manager.py',
        'app/services/order_manager.py',
        'app/state_machine/order_state_manager.py',
    ]
    
    all_issues = {}
    total_issues = 0
    
    for file_path in files_to_check:
        path = Path(file_path)
        if not path.exists():
            continue
        
        print(f"\n📄 Checking {file_path}...")
        
        issues = check_file_for_real_issues(path)
        if issues:
            all_issues[str(path)] = issues
            total_issues += len(issues)
            print(f"   ⚠️  Found {len(issues)} REAL issue(s)")
        else:
            print(f"   ✅ No issues found")
    
    # Detailed report
    if all_issues:
        print_section(f"📋 DETAILED REPORT ({total_issues} REAL issues)")
        
        for file_path, issues in all_issues.items():
            print(f"\n📄 {file_path}")
            print("-" * 80)
            
            for issue in issues:
                print(f"\n   Line {issue['line']}: [{issue['severity']}] {issue['type']}")
                print(f"   ❌ Issue: {issue['issue']}")
                print(f"   ✅ Fix:   {issue['fix']}")
                print(f"   📝 Context: {issue['context']}")
    else:
        print_section("✅ NO REAL ISSUES FOUND")
        print("\n🎉 All database queries are using correct column names!")
    
    # Summary
    print_section("📊 SUMMARY")
    print(f"\n   Files checked: {len(files_to_check)}")
    print(f"   Files with REAL issues: {len(all_issues)}")
    print(f"   Total REAL issues: {total_issues}")
    
    if total_issues > 0:
        print(f"\n   ⚠️  Action required: Fix {total_issues} database column issue(s)")
    else:
        print(f"\n   ✅ All database queries are correct!")

if __name__ == "__main__":
    audit_real_issues()

