"""
Audit all order-related queries in the codebase
Check for incorrect column names
"""
import os
import re
from pathlib import Path

# Correct column names from schema
ORDERS_COLUMNS = {
    'id', 'order_id', 'chat_id', 'customer_id', 'status',
    'subtotal_omr', 'delivery_fee_omr', 'discount_omr', 'tax_omr', 'total_omr',
    'delivery_address', 'delivery_latitude', 'delivery_longitude', 'delivery_location_name',
    'address_type', 'location_lat', 'location_lng', 'location_link',
    'payment_method', 'notes', 'created_at', 'updated_at'
}

ORDER_ITEMS_COLUMNS = {
    'id', 'order_id', 'item_code', 'item_name', 'qty',
    'unit_price_omr', 'subtotal_omr'
}

# Common incorrect column names
INCORRECT_NAMES = {
    'order_number': 'order_id',  # ❌ order_number → ✅ order_id
    'total': 'total_omr',         # ❌ total → ✅ total_omr
    'items': 'order_items',       # ❌ items → ✅ order_items (separate table)
    'subtotal': 'subtotal_omr',   # ❌ subtotal → ✅ subtotal_omr
    'delivery_fee': 'delivery_fee_omr',  # ❌ delivery_fee → ✅ delivery_fee_omr
    'quantity': 'qty',            # ❌ quantity → ✅ qty
    'price': 'unit_price_omr',    # ❌ price → ✅ unit_price_omr
}

def print_section(title):
    """Print section header"""
    print("\n" + "="*80)
    print(f"  {title}")
    print("="*80)

def check_file(file_path):
    """Check a Python file for incorrect column names"""
    issues = []
    
    try:
        with open(file_path, 'r', encoding='utf-8') as f:
            content = f.read()
            lines = content.split('\n')
        
        # Check for table('orders') queries
        if "table('orders')" in content or 'table("orders")' in content:
            # Check for .select() with incorrect columns
            select_pattern = r"\.select\(['\"]([^'\"]+)['\"]\)"
            for match in re.finditer(select_pattern, content):
                columns_str = match.group(1)
                columns = [c.strip() for c in columns_str.split(',')]
                
                for col in columns:
                    # Remove table prefixes and wildcards
                    col_clean = col.split('.')[-1].strip()
                    if col_clean == '*':
                        continue
                    
                    # Check if it's an incorrect name
                    if col_clean in INCORRECT_NAMES:
                        line_num = content[:match.start()].count('\n') + 1
                        issues.append({
                            'line': line_num,
                            'type': 'select',
                            'incorrect': col_clean,
                            'correct': INCORRECT_NAMES[col_clean],
                            'context': lines[line_num-1].strip()
                        })
            
            # Check for .eq(), .get(), dictionary access with incorrect names
            for incorrect, correct in INCORRECT_NAMES.items():
                # Check .eq('column_name')
                eq_pattern = rf"\.eq\(['\"]({incorrect})['\"]\s*,"
                for match in re.finditer(eq_pattern, content):
                    line_num = content[:match.start()].count('\n') + 1
                    issues.append({
                        'line': line_num,
                        'type': 'eq',
                        'incorrect': incorrect,
                        'correct': correct,
                        'context': lines[line_num-1].strip()
                    })
                
                # Check dictionary access ['column_name']
                dict_pattern = rf"\[['\"]({incorrect})['\"]\]"
                for match in re.finditer(dict_pattern, content):
                    line_num = content[:match.start()].count('\n') + 1
                    # Skip if it's in a comment
                    if not lines[line_num-1].strip().startswith('#'):
                        issues.append({
                            'line': line_num,
                            'type': 'dict_access',
                            'incorrect': incorrect,
                            'correct': correct,
                            'context': lines[line_num-1].strip()
                        })
                
                # Check .get('column_name')
                get_pattern = rf"\.get\(['\"]({incorrect})['\"]\s*[,\)]"
                for match in re.finditer(get_pattern, content):
                    line_num = content[:match.start()].count('\n') + 1
                    issues.append({
                        'line': line_num,
                        'type': 'get',
                        'incorrect': incorrect,
                        'correct': correct,
                        'context': lines[line_num-1].strip()
                    })
        
    except Exception as e:
        print(f"   ❌ Error reading {file_path}: {e}")
    
    return issues

def audit_codebase():
    """Audit the entire codebase for order query issues"""
    
    print_section("🔍 AUDITING ORDER QUERIES IN CODEBASE")
    
    # Directories to check
    directories = [
        'app/routes',
        'app/services',
        'app/handlers',
        'app/state_machine',
        'scripts'
    ]
    
    all_issues = {}
    total_issues = 0
    
    for directory in directories:
        dir_path = Path(directory)
        if not dir_path.exists():
            continue
        
        print(f"\n📁 Checking {directory}/...")
        
        # Find all Python files
        py_files = list(dir_path.rglob('*.py'))
        
        for py_file in py_files:
            issues = check_file(py_file)
            if issues:
                all_issues[str(py_file)] = issues
                total_issues += len(issues)
                print(f"   ⚠️  {py_file.name}: {len(issues)} issue(s)")
    
    # Print detailed report
    if all_issues:
        print_section(f"📋 DETAILED REPORT ({total_issues} issues found)")
        
        for file_path, issues in all_issues.items():
            print(f"\n📄 {file_path}")
            print("-" * 80)
            
            for issue in issues:
                print(f"\n   Line {issue['line']}: {issue['type']}")
                print(f"   ❌ Incorrect: {issue['incorrect']}")
                print(f"   ✅ Correct:   {issue['correct']}")
                print(f"   📝 Context:   {issue['context'][:70]}...")
    else:
        print_section("✅ NO ISSUES FOUND")
        print("\n🎉 All order queries are using correct column names!")
    
    # Summary
    print_section("📊 SUMMARY")
    print(f"\n   Files checked: {sum(len(list(Path(d).rglob('*.py'))) for d in directories if Path(d).exists())}")
    print(f"   Files with issues: {len(all_issues)}")
    print(f"   Total issues: {total_issues}")
    
    if total_issues > 0:
        print(f"\n   ⚠️  Action required: Fix {total_issues} issue(s)")
        print("\n   Common fixes:")
        for incorrect, correct in INCORRECT_NAMES.items():
            print(f"      • {incorrect:20s} → {correct}")
    else:
        print(f"\n   ✅ All queries are correct!")

if __name__ == "__main__":
    audit_codebase()

