#!/usr/bin/env python3
"""
Analyze System Prompt size and suggest optimizations
"""

import os
import sys
from pathlib import Path
from dotenv import load_dotenv
from supabase import create_client
import tiktoken

# Load environment variables
load_dotenv()


def count_tokens(text: str) -> int:
    """Count tokens using tiktoken"""
    encoding = tiktoken.encoding_for_model("gpt-4o-mini")
    return len(encoding.encode(text))


def analyze_prompt_sections():
    """Analyze all prompt sections"""
    # Initialize Supabase client
    supabase = create_client(
        os.getenv('SUPABASE_URL'),
        os.getenv('SUPABASE_SERVICE_KEY')
    )
    
    # Get all active sections
    result = supabase.table('prompt_sections') \
        .select('section_key, section_name, content, priority') \
        .eq('active', True) \
        .order('priority') \
        .execute()
    
    sections = result.data
    
    print("\n" + "="*80)
    print("📊 تحليل حجم System Prompt")
    print("="*80)
    
    # Analyze each section
    section_analysis = []
    total_tokens = 0
    
    for section in sections:
        content = section['content']
        tokens = count_tokens(content)
        total_tokens += tokens
        
        section_analysis.append({
            'key': section['section_key'],
            'name': section['section_name'],
            'priority': section['priority'],
            'tokens': tokens,
            'chars': len(content)
        })
    
    # Sort by tokens (descending)
    section_analysis.sort(key=lambda x: x['tokens'], reverse=True)
    
    # Print summary
    print(f"\n📈 الملخص:")
    print(f"  • إجمالي الأقسام: {len(sections)}")
    print(f"  • إجمالي Tokens: {total_tokens:,}")
    print(f"  • إجمالي الأحرف: {sum(s['chars'] for s in section_analysis):,}")
    print(f"  • متوسط Tokens/قسم: {total_tokens // len(sections):,}")
    
    # Print detailed breakdown
    print(f"\n📋 تفصيل الأقسام (مرتبة حسب الحجم):")
    print("-"*80)
    print(f"{'القسم':<40} {'Tokens':>10} {'النسبة':>10} {'الأولوية':>10}")
    print("-"*80)
    
    for section in section_analysis:
        percentage = (section['tokens'] / total_tokens) * 100
        print(f"{section['name']:<40} {section['tokens']:>10,} {percentage:>9.1f}% {section['priority']:>10}")
    
    print("-"*80)
    print(f"{'المجموع':<40} {total_tokens:>10,} {'100.0%':>10}")
    print("="*80)
    
    # Identify optimization opportunities
    print(f"\n💡 فرص التحسين:")
    print("-"*80)
    
    # Large sections (> 800 tokens)
    large_sections = [s for s in section_analysis if s['tokens'] > 800]
    if large_sections:
        print(f"\n1️⃣ أقسام كبيرة (> 800 tokens):")
        for section in large_sections:
            print(f"   • {section['name']}: {section['tokens']:,} tokens")
            print(f"     → يمكن تقليلها بنسبة 30-40%")
    
    # Similar sections that can be merged
    print(f"\n2️⃣ أقسام يمكن دمجها:")
    intent_sections = [s for s in section_analysis if 'intent_' in s['key']]
    if len(intent_sections) > 10:
        print(f"   • لديك {len(intent_sections)} intent sections")
        print(f"     → يمكن دمج المتشابهة منها")
    
    # Calculate potential savings
    print(f"\n3️⃣ التوفير المتوقع:")
    
    # Scenario 1: Optimize large sections
    large_tokens = sum(s['tokens'] for s in large_sections)
    savings_1 = large_tokens * 0.35  # 35% reduction
    
    # Scenario 2: Merge similar sections
    savings_2 = len(intent_sections) * 100 if len(intent_sections) > 10 else 0
    
    # Scenario 3: Remove examples
    savings_3 = total_tokens * 0.15  # 15% from removing examples
    
    total_savings = savings_1 + savings_2 + savings_3
    new_total = total_tokens - total_savings
    
    print(f"   • تحسين الأقسام الكبيرة: -{savings_1:,.0f} tokens")
    print(f"   • دمج الأقسام المتشابهة: -{savings_2:,.0f} tokens")
    print(f"   • إزالة الأمثلة المكررة: -{savings_3:,.0f} tokens")
    print(f"   • التوفير الإجمالي: -{total_savings:,.0f} tokens ({(total_savings/total_tokens)*100:.1f}%)")
    print(f"   • الحجم الجديد: {new_total:,.0f} tokens")
    
    # Calculate cost savings
    current_cost = total_tokens * 0.150 / 1_000_000
    new_cost = new_total * 0.150 / 1_000_000
    cost_savings = current_cost - new_cost
    
    print(f"\n💰 توفير التكلفة:")
    print(f"   • التكلفة الحالية: ${current_cost:.6f}/رسالة")
    print(f"   • التكلفة بعد التحسين: ${new_cost:.6f}/رسالة")
    print(f"   • التوفير: ${cost_savings:.6f}/رسالة ({(cost_savings/current_cost)*100:.1f}%)")
    
    # Monthly savings (for medium restaurant)
    messages_per_month = 54_000
    monthly_savings = cost_savings * messages_per_month
    print(f"   • التوفير الشهري (مطعم متوسط): ${monthly_savings:.2f}")
    print(f"   • التوفير السنوي: ${monthly_savings * 12:.2f}")
    
    print("\n" + "="*80)
    
    # Recommendations
    print(f"\n🎯 التوصيات:")
    print("-"*80)
    print("1. ابدأ بتحسين الأقسام الكبيرة (> 800 tokens)")
    print("2. اختصر الأمثلة واستخدم أمثلة أقصر")
    print("3. ادمج الأقسام المتشابهة")
    print("4. استخدم لغة أكثر إيجازاً")
    print("5. احذف التكرار والمعلومات غير الضرورية")
    print("\n⚠️  تذكر: اختبر جيداً بعد كل تحسين!")
    print("="*80 + "\n")
    
    return section_analysis, total_tokens


def compare_with_target():
    """Compare current size with target"""
    _, current_tokens = analyze_prompt_sections()
    target_tokens = 6000
    
    print(f"\n📊 المقارنة مع الهدف:")
    print("-"*80)
    print(f"الحالي: {current_tokens:,} tokens")
    print(f"الهدف: {target_tokens:,} tokens")
    print(f"الفرق: {current_tokens - target_tokens:,} tokens")
    print(f"نسبة التقليل المطلوبة: {((current_tokens - target_tokens) / current_tokens) * 100:.1f}%")
    print("="*80 + "\n")


if __name__ == "__main__":
    try:
        analyze_prompt_sections()
        # compare_with_target()
    except Exception as e:
        print(f"\n❌ خطأ: {e}")
        import traceback
        traceback.print_exc()

