#!/usr/bin/env python3
# -*- coding: utf-8 -*-
"""
استخراج رسائل المستخدمين من اللوقات
لتحليل أنماط الاستخدام والكلمات الشائعة
"""

import json
import re
from collections import Counter
from datetime import datetime
from pathlib import Path
from typing import Dict, List


class MessageExtractor:
    """مستخرج رسائل المستخدمين"""
    
    def __init__(self, log_file: str = 'logs/app.log'):
        self.log_file = log_file
        self.messages = []
        self.intents = Counter()
        self.words = Counter()
        self.dialects = Counter()
    
    def extract(self, limit: int = 1000) -> Dict:
        """استخراج الرسائل"""
        
        print(f"📊 بدء استخراج رسائل المستخدمين من: {self.log_file}")
        print(f"📝 الحد الأقصى: {limit} رسالة")
        print("="*60)
        
        if not Path(self.log_file).exists():
            print(f"❌ الملف غير موجود: {self.log_file}")
            return self._create_empty_report()
        
        # قراءة اللوقات
        with open(self.log_file, 'r', encoding='utf-8') as f:
            lines = f.readlines()
        
        print(f"📄 عدد الأسطر: {len(lines)}")
        
        # استخراج الرسائل
        for line in lines:
            if len(self.messages) >= limit:
                break
            
            message_data = self._extract_message(line)
            if message_data:
                self.messages.append(message_data)
                
                # تحليل الرسالة
                self._analyze_message(message_data)
        
        print(f"✅ تم استخراج {len(self.messages)} رسالة")
        
        # إنشاء التقرير
        report = self._create_report()
        
        # حفظ التقرير
        self._save_report(report)
        
        # طباعة الملخص
        self._print_summary(report)
        
        return report
    
    def _extract_message(self, line: str) -> Dict:
        """استخراج رسالة من سطر"""
        
        # البحث عن رسائل المستخدمين
        patterns = [
            r'Processing message[:\s]+["\']([^"\']+)["\']',
            r'Received message[:\s]+["\']([^"\']+)["\']',
            r'User message[:\s]+["\']([^"\']+)["\']',
            r'Message text[:\s]+["\']([^"\']+)["\']',
        ]
        
        for pattern in patterns:
            match = re.search(pattern, line, re.IGNORECASE)
            if match:
                message_text = match.group(1).strip()
                
                # استخراج النية إن وجدت
                intent = self._extract_intent(line)
                
                # استخراج الوقت
                timestamp = self._extract_timestamp(line)
                
                return {
                    'text': message_text,
                    'intent': intent,
                    'timestamp': timestamp,
                    'length': len(message_text),
                    'word_count': len(message_text.split())
                }
        
        return None
    
    def _extract_intent(self, line: str) -> str:
        """استخراج النية من السطر"""
        
        patterns = [
            r'Intent[:\s]+([a-z_]+)',
            r'Classified as[:\s]+([a-z_]+)',
            r'Detected intent[:\s]+([a-z_]+)',
        ]
        
        for pattern in patterns:
            match = re.search(pattern, line, re.IGNORECASE)
            if match:
                return match.group(1)
        
        return 'unknown'
    
    def _extract_timestamp(self, line: str) -> str:
        """استخراج الوقت من السطر"""
        
        # محاولة استخراج timestamp
        match = re.search(r'(\d{4}-\d{2}-\d{2}[T\s]\d{2}:\d{2}:\d{2})', line)
        if match:
            return match.group(1)
        
        return datetime.now().isoformat()
    
    def _analyze_message(self, message_data: Dict):
        """تحليل رسالة"""
        
        text = message_data['text'].lower()
        
        # تحليل النية
        if message_data['intent']:
            self.intents[message_data['intent']] += 1
        
        # تحليل الكلمات
        words = text.split()
        for word in words:
            if len(word) >= 3:  # تجاهل الكلمات القصيرة جداً
                self.words[word] += 1
        
        # كشف اللهجة
        dialect = self._detect_dialect(text)
        if dialect:
            self.dialects[dialect] += 1
    
    def _detect_dialect(self, text: str) -> str:
        """كشف اللهجة من النص"""
        
        dialect_keywords = {
            'خليجي': ['ابغي', 'ابي', 'ودي', 'عطني', 'شحالة', 'بجم', 'خلي'],
            'مصري': ['عايز', 'عاوز', 'هاتلي', 'جيبلي', 'بكام', 'فراخ', 'عيش'],
            'شامي': ['بدي', 'بدنا', 'جبلي', 'عطيني', 'قديش', 'أديش'],
            'عراقي': ['اريد', 'نريد', 'اكو', 'ماكو', 'شلون', 'شنو'],
            'مغربي': ['نحب', 'نبغي', 'نزيد', 'بغيت', 'شحال', 'واش']
        }
        
        for dialect, keywords in dialect_keywords.items():
            for keyword in keywords:
                if keyword in text:
                    return dialect
        
        return 'فصحى'
    
    def _create_report(self) -> Dict:
        """إنشاء تقرير شامل"""
        
        report = {
            'timestamp': datetime.now().isoformat(),
            'log_file': self.log_file,
            'total_messages': len(self.messages),
            'statistics': {
                'avg_message_length': sum(m['length'] for m in self.messages) / len(self.messages) if self.messages else 0,
                'avg_word_count': sum(m['word_count'] for m in self.messages) / len(self.messages) if self.messages else 0,
                'unique_words': len(self.words),
                'unique_intents': len(self.intents)
            },
            'top_intents': [
                {'intent': intent, 'count': count}
                for intent, count in self.intents.most_common(10)
            ],
            'top_words': [
                {'word': word, 'count': count}
                for word, count in self.words.most_common(50)
            ],
            'dialect_distribution': [
                {'dialect': dialect, 'count': count, 'percentage': round((count / len(self.messages)) * 100, 2)}
                for dialect, count in self.dialects.most_common()
            ],
            'sample_messages': self.messages[:20]  # أول 20 رسالة كعينة
        }
        
        return report
    
    def _create_empty_report(self) -> Dict:
        """إنشاء تقرير فارغ"""
        return {
            'timestamp': datetime.now().isoformat(),
            'log_file': self.log_file,
            'total_messages': 0,
            'statistics': {},
            'top_intents': [],
            'top_words': [],
            'dialect_distribution': [],
            'sample_messages': []
        }
    
    def _save_report(self, report: Dict):
        """حفظ التقرير"""
        
        # إنشاء مجلد التحليل
        Path('analysis').mkdir(exist_ok=True)
        
        # حفظ JSON
        report_file = 'analysis/user_messages_report.json'
        with open(report_file, 'w', encoding='utf-8') as f:
            json.dump(report, f, ensure_ascii=False, indent=2)
        
        print(f"\n✅ تم حفظ التقرير: {report_file}")
        
        # حفظ الرسائل في ملف نصي
        messages_file = 'analysis/user_messages.txt'
        with open(messages_file, 'w', encoding='utf-8') as f:
            f.write(f"رسائل المستخدمين ({len(self.messages)} رسالة)\n")
            f.write("="*60 + "\n\n")
            
            for i, msg in enumerate(self.messages, 1):
                f.write(f"{i:4d}. [{msg['intent']:15s}] {msg['text']}\n")
        
        print(f"✅ تم حفظ الرسائل: {messages_file}")
        
        # حفظ أكثر الكلمات شيوعاً
        words_file = 'analysis/top_words.txt'
        with open(words_file, 'w', encoding='utf-8') as f:
            f.write("أكثر 50 كلمة شيوعاً\n")
            f.write("="*60 + "\n\n")
            
            for word, count in self.words.most_common(50):
                f.write(f"{count:4d}x  {word}\n")
        
        print(f"✅ تم حفظ الكلمات الشائعة: {words_file}")
    
    def _print_summary(self, report: Dict):
        """طباعة ملخص التقرير"""
        
        print("\n" + "="*60)
        print("📊 ملخص رسائل المستخدمين")
        print("="*60)
        
        if report['total_messages'] == 0:
            print("\n❌ لم يتم العثور على رسائل")
            return
        
        stats = report['statistics']
        
        print(f"\n📈 الإحصائيات:")
        print(f"   إجمالي الرسائل:        {report['total_messages']}")
        print(f"   متوسط طول الرسالة:     {stats['avg_message_length']:.1f} حرف")
        print(f"   متوسط عدد الكلمات:     {stats['avg_word_count']:.1f} كلمة")
        print(f"   عدد الكلمات الفريدة:   {stats['unique_words']}")
        print(f"   عدد النوايا الفريدة:   {stats['unique_intents']}")
        
        print("\n🎯 أكثر 10 نوايا شيوعاً:")
        print("-"*60)
        for item in report['top_intents']:
            percentage = (item['count'] / report['total_messages']) * 100
            print(f"   {item['intent']:20s}: {item['count']:4d} ({percentage:5.1f}%)")
        
        print("\n🗣️ توزيع اللهجات:")
        print("-"*60)
        for item in report['dialect_distribution']:
            print(f"   {item['dialect']:10s}: {item['count']:4d} ({item['percentage']:5.1f}%)")
        
        print("\n📝 أكثر 20 كلمة شيوعاً:")
        print("-"*60)
        for i, item in enumerate(report['top_words'][:20], 1):
            print(f"   {i:2d}. {item['word']:15s}: {item['count']:4d}x")
        
        print("\n💬 عينة من الرسائل (أول 10):")
        print("-"*60)
        for i, msg in enumerate(report['sample_messages'][:10], 1):
            print(f"   {i:2d}. [{msg['intent']:12s}] {msg['text']}")


def main():
    """الدالة الرئيسية"""
    
    print("🚀 مستخرج رسائل المستخدمين - نظام Chatbot المطعم")
    print("="*60)
    
    # إنشاء المستخرج
    extractor = MessageExtractor('logs/app.log')
    
    # استخراج الرسائل
    report = extractor.extract(limit=1000)
    
    print("\n✅ اكتمل الاستخراج!")
    print(f"📄 راجع التقرير الكامل في: analysis/user_messages_report.json")


if __name__ == '__main__':
    main()

