#!/usr/bin/env python3
# -*- coding: utf-8 -*-
"""
حساب المقاييس الأساسية (Baseline Metrics)
لقياس الأداء الحالي قبل التحسينات
"""

import json
import re
from datetime import datetime, timedelta
from pathlib import Path
from collections import Counter
from typing import Dict, List


class BaselineCalculator:
    """حاسب المقاييس الأساسية"""
    
    def __init__(self, log_file: str = 'logs/app.log'):
        self.log_file = log_file
        self.metrics = {
            'understanding_accuracy': 0,    # دقة الفهم
            'search_accuracy': 0,           # دقة البحث
            'completion_rate': 0,           # معدل الإتمام
            'avg_response_time': 0,         # متوسط وقت الاستجابة
            'user_satisfaction': 0,         # رضا المستخدم
            'total_messages': 0,
            'successful_messages': 0,
            'failed_messages': 0,
            'total_searches': 0,
            'successful_searches': 0,
            'failed_searches': 0,
            'orders_started': 0,
            'orders_completed': 0,
            'orders_cancelled': 0
        }
    
    def calculate(self) -> Dict:
        """حساب جميع المقاييس"""
        
        print(f"📊 بدء حساب المقاييس الأساسية من: {self.log_file}")
        print("="*60)
        
        if not Path(self.log_file).exists():
            print(f"❌ الملف غير موجود: {self.log_file}")
            return self._create_empty_metrics()
        
        # قراءة اللوقات
        with open(self.log_file, 'r', encoding='utf-8') as f:
            lines = f.readlines()
        
        print(f"📄 عدد الأسطر: {len(lines)}")
        
        # تحليل اللوقات
        self._analyze_logs(lines)
        
        # حساب المقاييس النهائية
        self._calculate_final_metrics()
        
        # إنشاء التقرير
        report = self._create_report()
        
        # حفظ التقرير
        self._save_report(report)
        
        # طباعة الملخص
        self._print_summary(report)
        
        return report
    
    def _analyze_logs(self, lines: List[str]):
        """تحليل اللوقات"""
        
        response_times = []
        
        for line in lines:
            line_lower = line.lower()
            
            # عدد الرسائل
            if 'processing message' in line_lower or 'received message' in line_lower:
                self.metrics['total_messages'] += 1
            
            # الرسائل الناجحة
            if any(keyword in line_lower for keyword in ['intent detected', 'classified as', 'تم التصنيف']):
                self.metrics['successful_messages'] += 1
            
            # الرسائل الفاشلة
            if any(keyword in line_lower for keyword in ['لم أفهم', 'not understood', 'unknown intent']):
                self.metrics['failed_messages'] += 1
            
            # عمليات البحث
            if 'searching for' in line_lower or 'faiss search' in line_lower:
                self.metrics['total_searches'] += 1
            
            # البحث الناجح
            if 'found' in line_lower and 'results' in line_lower and 'no results' not in line_lower:
                self.metrics['successful_searches'] += 1
            
            # البحث الفاشل
            if 'no results found' in line_lower or 'لم أجد' in line_lower:
                self.metrics['failed_searches'] += 1
            
            # الطلبات المبدوءة
            if 'order started' in line_lower or 'بدء الطلب' in line_lower:
                self.metrics['orders_started'] += 1
            
            # الطلبات المكتملة
            if 'order completed' in line_lower or 'تم تأكيد الطلب' in line_lower:
                self.metrics['orders_completed'] += 1
            
            # الطلبات الملغاة
            if 'order cancelled' in line_lower or 'تم إلغاء الطلب' in line_lower:
                self.metrics['orders_cancelled'] += 1
            
            # وقت الاستجابة
            time_match = re.search(r'response time[:\s]+([0-9.]+)', line_lower)
            if time_match:
                response_times.append(float(time_match.group(1)))
        
        # حساب متوسط وقت الاستجابة
        if response_times:
            self.metrics['avg_response_time'] = sum(response_times) / len(response_times)
    
    def _calculate_final_metrics(self):
        """حساب المقاييس النهائية"""
        
        # دقة الفهم
        if self.metrics['total_messages'] > 0:
            self.metrics['understanding_accuracy'] = (
                self.metrics['successful_messages'] / self.metrics['total_messages']
            ) * 100
        
        # دقة البحث
        if self.metrics['total_searches'] > 0:
            self.metrics['search_accuracy'] = (
                self.metrics['successful_searches'] / self.metrics['total_searches']
            ) * 100
        
        # معدل الإتمام
        if self.metrics['orders_started'] > 0:
            self.metrics['completion_rate'] = (
                self.metrics['orders_completed'] / self.metrics['orders_started']
            ) * 100
        
        # رضا المستخدم (تقدير بناءً على معدل الإتمام ودقة الفهم)
        self.metrics['user_satisfaction'] = (
            self.metrics['understanding_accuracy'] * 0.4 +
            self.metrics['completion_rate'] * 0.6
        )
    
    def _create_report(self) -> Dict:
        """إنشاء تقرير شامل"""
        
        report = {
            'timestamp': datetime.now().isoformat(),
            'log_file': self.log_file,
            'baseline_metrics': {
                'understanding_accuracy': round(self.metrics['understanding_accuracy'], 2),
                'search_accuracy': round(self.metrics['search_accuracy'], 2),
                'completion_rate': round(self.metrics['completion_rate'], 2),
                'avg_response_time': round(self.metrics['avg_response_time'], 3),
                'user_satisfaction': round(self.metrics['user_satisfaction'], 2)
            },
            'detailed_stats': {
                'messages': {
                    'total': self.metrics['total_messages'],
                    'successful': self.metrics['successful_messages'],
                    'failed': self.metrics['failed_messages']
                },
                'searches': {
                    'total': self.metrics['total_searches'],
                    'successful': self.metrics['successful_searches'],
                    'failed': self.metrics['failed_searches']
                },
                'orders': {
                    'started': self.metrics['orders_started'],
                    'completed': self.metrics['orders_completed'],
                    'cancelled': self.metrics['orders_cancelled']
                }
            },
            'targets': {
                'understanding_accuracy': 95.0,
                'search_accuracy': 90.0,
                'completion_rate': 85.0,
                'avg_response_time': 2.0,
                'user_satisfaction': 80.0
            },
            'gaps': self._calculate_gaps()
        }
        
        return report
    
    def _calculate_gaps(self) -> Dict:
        """حساب الفجوات بين الحالي والمستهدف"""
        
        targets = {
            'understanding_accuracy': 95.0,
            'search_accuracy': 90.0,
            'completion_rate': 85.0,
            'avg_response_time': 2.0,
            'user_satisfaction': 80.0
        }
        
        gaps = {}
        
        for metric, target in targets.items():
            current = self.metrics[metric]
            
            if metric == 'avg_response_time':
                # أقل أفضل
                gap = current - target
                status = '✅' if current <= target else '❌'
            else:
                # أكثر أفضل
                gap = target - current
                status = '✅' if current >= target else '❌'
            
            gaps[metric] = {
                'current': round(current, 2),
                'target': target,
                'gap': round(gap, 2),
                'status': status
            }
        
        return gaps
    
    def _create_empty_metrics(self) -> Dict:
        """إنشاء مقاييس فارغة"""
        return {
            'timestamp': datetime.now().isoformat(),
            'log_file': self.log_file,
            'baseline_metrics': {},
            'detailed_stats': {},
            'targets': {},
            'gaps': {}
        }
    
    def _save_report(self, report: Dict):
        """حفظ التقرير"""
        
        # إنشاء مجلد المقاييس
        Path('metrics').mkdir(exist_ok=True)
        
        # حفظ JSON
        report_file = 'metrics/baseline.json'
        with open(report_file, 'w', encoding='utf-8') as f:
            json.dump(report, f, ensure_ascii=False, indent=2)
        
        print(f"\n✅ تم حفظ التقرير: {report_file}")
    
    def _print_summary(self, report: Dict):
        """طباعة ملخص التقرير"""
        
        print("\n" + "="*60)
        print("📊 المقاييس الأساسية (Baseline Metrics)")
        print("="*60)
        
        if not report['baseline_metrics']:
            print("\n❌ لا توجد بيانات كافية لحساب المقاييس")
            return
        
        print("\n📈 المقاييس الرئيسية:")
        print("-"*60)
        
        for metric, data in report['gaps'].items():
            metric_name = self._get_metric_name(metric)
            unit = '%' if metric != 'avg_response_time' else 's'
            
            print(f"\n{data['status']} {metric_name}:")
            print(f"   الحالي:    {data['current']}{unit}")
            print(f"   المستهدف:  {data['target']}{unit}")
            print(f"   الفجوة:    {abs(data['gap'])}{unit}")
        
        print("\n" + "="*60)
        print("📊 إحصائيات تفصيلية:")
        print("="*60)
        
        stats = report['detailed_stats']
        
        print(f"\n💬 الرسائل:")
        print(f"   الإجمالي:  {stats['messages']['total']}")
        print(f"   الناجحة:   {stats['messages']['successful']}")
        print(f"   الفاشلة:   {stats['messages']['failed']}")
        
        print(f"\n🔍 عمليات البحث:")
        print(f"   الإجمالي:  {stats['searches']['total']}")
        print(f"   الناجحة:   {stats['searches']['successful']}")
        print(f"   الفاشلة:   {stats['searches']['failed']}")
        
        print(f"\n🛒 الطلبات:")
        print(f"   المبدوءة:  {stats['orders']['started']}")
        print(f"   المكتملة:  {stats['orders']['completed']}")
        print(f"   الملغاة:    {stats['orders']['cancelled']}")
        
        print("\n" + "="*60)
        print("💡 التوصيات:")
        print("="*60)
        
        # توصيات بناءً على الفجوات
        for metric, data in report['gaps'].items():
            if data['status'] == '❌':
                metric_name = self._get_metric_name(metric)
                print(f"\n❌ {metric_name}: يحتاج تحسين ({abs(data['gap'])} نقطة)")
                
                if metric == 'understanding_accuracy':
                    print("   - حسّن تصنيف النوايا")
                    print("   - أضف كلمات مفتاحية جديدة")
                    print("   - حسّن معالجة اللهجات")
                
                elif metric == 'search_accuracy':
                    print("   - حسّن بيانات التدريب")
                    print("   - أضف aliases للأصناف")
                    print("   - حسّن دالة البحث")
                
                elif metric == 'completion_rate':
                    print("   - حسّن تجربة المستخدم")
                    print("   - قلل خطوات الطلب")
                    print("   - أضف رسائل توضيحية")
                
                elif metric == 'avg_response_time':
                    print("   - حسّن أداء قاعدة البيانات")
                    print("   - استخدم caching")
                    print("   - قلل استدعاءات API")
    
    def _get_metric_name(self, metric: str) -> str:
        """الحصول على اسم المقياس بالعربية"""
        names = {
            'understanding_accuracy': 'دقة الفهم',
            'search_accuracy': 'دقة البحث',
            'completion_rate': 'معدل الإتمام',
            'avg_response_time': 'متوسط وقت الاستجابة',
            'user_satisfaction': 'رضا المستخدم'
        }
        return names.get(metric, metric)


def main():
    """الدالة الرئيسية"""
    
    print("🚀 حاسب المقاييس الأساسية - نظام Chatbot المطعم")
    print("="*60)
    
    # إنشاء الحاسب
    calculator = BaselineCalculator('logs/app.log')
    
    # حساب المقاييس
    report = calculator.calculate()
    
    print("\n✅ اكتمل الحساب!")
    print(f"📄 راجع التقرير الكامل في: metrics/baseline.json")


if __name__ == '__main__':
    main()

