#!/usr/bin/env python3
# -*- coding: utf-8 -*-
"""
تحليل الأخطاء من ملفات اللوقات
يستخرج ويصنف جميع الأخطاء لتحديد نقاط الضعف
"""

import json
import re
from collections import Counter
from datetime import datetime
from pathlib import Path
from typing import Dict, List


class ErrorAnalyzer:
    """محلل الأخطاء من اللوقات"""
    
    def __init__(self, log_file: str = 'logs/app.log'):
        self.log_file = log_file
        self.errors = {
            'search_failures': [],      # فشل البحث
            'intent_errors': [],        # خطأ في تصنيف النية
            'dialect_issues': [],       # مشاكل اللهجات
            'spelling_errors': [],      # أخطاء إملائية
            'context_failures': [],     # فشل فهم السياق
            'item_not_found': [],       # لم يتم العثور على الصنف
            'unknown_errors': []        # أخطاء غير معروفة
        }
    
    def analyze(self) -> Dict:
        """تحليل ملف اللوقات"""
        
        print(f"📊 بدء تحليل الأخطاء من: {self.log_file}")
        print("="*60)
        
        if not Path(self.log_file).exists():
            print(f"❌ الملف غير موجود: {self.log_file}")
            return self._create_empty_report()
        
        # قراءة اللوقات
        with open(self.log_file, 'r', encoding='utf-8') as f:
            lines = f.readlines()
        
        print(f"📄 عدد الأسطر: {len(lines)}")
        
        # تحليل كل سطر
        for line in lines:
            self._analyze_line(line)
        
        # إنشاء التقرير
        report = self._create_report()
        
        # حفظ التقرير
        self._save_report(report)
        
        # طباعة الملخص
        self._print_summary(report)
        
        return report
    
    def _analyze_line(self, line: str):
        """تحليل سطر واحد من اللوقات"""
        
        line_lower = line.lower()
        
        # فشل البحث
        if any(keyword in line_lower for keyword in ['no results found', 'لم أجد', 'لا توجد نتائج']):
            self.errors['search_failures'].append(self._extract_query(line))
        
        # خطأ في تصنيف النية
        elif 'intent misclassified' in line_lower or 'wrong intent' in line_lower:
            self.errors['intent_errors'].append(self._extract_message(line))
        
        # لم يتم العثور على الصنف
        elif 'item not found' in line_lower or 'لم أجد الصنف' in line_lower:
            self.errors['item_not_found'].append(self._extract_query(line))
        
        # مشاكل اللهجات
        elif 'dialect' in line_lower or 'لهجة' in line_lower:
            self.errors['dialect_issues'].append(self._extract_message(line))
        
        # أخطاء إملائية
        elif 'spelling' in line_lower or 'إملائي' in line_lower:
            self.errors['spelling_errors'].append(self._extract_message(line))
        
        # فشل فهم السياق
        elif 'context' in line_lower or 'سياق' in line_lower:
            self.errors['context_failures'].append(self._extract_message(line))
        
        # أخطاء عامة
        elif any(keyword in line_lower for keyword in ['error', 'exception', 'failed', 'خطأ', 'فشل']):
            self.errors['unknown_errors'].append(line.strip())
    
    def _extract_query(self, line: str) -> str:
        """استخراج الاستعلام من السطر"""
        # محاولة استخراج النص بين علامات الاقتباس
        match = re.search(r'["\']([^"\']+)["\']', line)
        if match:
            return match.group(1)
        
        # محاولة استخراج النص بعد "query:" أو "message:"
        match = re.search(r'(?:query|message|text):\s*(.+?)(?:\s*\||$)', line, re.IGNORECASE)
        if match:
            return match.group(1).strip()
        
        return line.strip()
    
    def _extract_message(self, line: str) -> str:
        """استخراج الرسالة من السطر"""
        return self._extract_query(line)
    
    def _create_report(self) -> Dict:
        """إنشاء تقرير شامل"""
        
        # حساب الإحصائيات
        total_errors = sum(len(v) for v in self.errors.values())
        
        # أكثر الاستعلامات الفاشلة شيوعاً
        all_failed_queries = (
            self.errors['search_failures'] + 
            self.errors['item_not_found']
        )
        top_failed_queries = Counter(all_failed_queries).most_common(20)
        
        # تصنيف الأخطاء
        error_breakdown = {k: len(v) for k, v in self.errors.items()}
        
        # حساب النسب المئوية
        error_percentages = {}
        if total_errors > 0:
            for category, count in error_breakdown.items():
                error_percentages[category] = round((count / total_errors) * 100, 2)
        
        report = {
            'timestamp': datetime.now().isoformat(),
            'log_file': self.log_file,
            'total_errors': total_errors,
            'error_breakdown': error_breakdown,
            'error_percentages': error_percentages,
            'top_failed_queries': [
                {'query': query, 'count': count} 
                for query, count in top_failed_queries
            ],
            'priority_areas': self._identify_priority_areas(error_breakdown),
            'detailed_errors': self.errors
        }
        
        return report
    
    def _identify_priority_areas(self, error_breakdown: Dict) -> List[Dict]:
        """تحديد مجالات الأولوية"""
        
        priority_areas = []
        
        for category, count in error_breakdown.items():
            if count == 0:
                continue
            
            # تحديد الأولوية
            if count > 50:
                priority = 'عالية جداً'
                emoji = '🔴'
            elif count > 20:
                priority = 'عالية'
                emoji = '🟠'
            elif count > 10:
                priority = 'متوسطة'
                emoji = '🟡'
            else:
                priority = 'منخفضة'
                emoji = '🟢'
            
            priority_areas.append({
                'category': category,
                'count': count,
                'priority': priority,
                'emoji': emoji
            })
        
        # ترتيب حسب العدد
        priority_areas.sort(key=lambda x: x['count'], reverse=True)
        
        return priority_areas
    
    def _create_empty_report(self) -> Dict:
        """إنشاء تقرير فارغ"""
        return {
            'timestamp': datetime.now().isoformat(),
            'log_file': self.log_file,
            'total_errors': 0,
            'error_breakdown': {},
            'error_percentages': {},
            'top_failed_queries': [],
            'priority_areas': [],
            'detailed_errors': {}
        }
    
    def _save_report(self, report: Dict):
        """حفظ التقرير"""
        
        # إنشاء مجلد التقارير
        Path('analysis').mkdir(exist_ok=True)
        
        # حفظ JSON
        report_file = 'analysis/error_report.json'
        with open(report_file, 'w', encoding='utf-8') as f:
            json.dump(report, f, ensure_ascii=False, indent=2)
        
        print(f"\n✅ تم حفظ التقرير: {report_file}")
        
        # حفظ أكثر الاستعلامات الفاشلة
        if report['top_failed_queries']:
            queries_file = 'analysis/top_failed_queries.txt'
            with open(queries_file, 'w', encoding='utf-8') as f:
                f.write("أكثر 20 استعلام فاشل\n")
                f.write("="*60 + "\n\n")
                for item in report['top_failed_queries']:
                    f.write(f"{item['count']:3d}x  {item['query']}\n")
            
            print(f"✅ تم حفظ الاستعلامات الفاشلة: {queries_file}")
    
    def _print_summary(self, report: Dict):
        """طباعة ملخص التقرير"""
        
        print("\n" + "="*60)
        print("📊 ملخص تحليل الأخطاء")
        print("="*60)
        
        print(f"\n📈 إجمالي الأخطاء: {report['total_errors']}")
        
        if report['total_errors'] == 0:
            print("\n✅ لا توجد أخطاء! النظام يعمل بشكل ممتاز!")
            return
        
        print("\n📋 تصنيف الأخطاء:")
        print("-"*60)
        
        for area in report['priority_areas']:
            percentage = report['error_percentages'].get(area['category'], 0)
            print(f"{area['emoji']} {area['category']:20s}: {area['count']:4d} ({percentage:5.1f}%) - {area['priority']}")
        
        print("\n🔝 أكثر 10 استعلامات فاشلة:")
        print("-"*60)
        
        for i, item in enumerate(report['top_failed_queries'][:10], 1):
            print(f"{i:2d}. [{item['count']:3d}x] {item['query']}")
        
        print("\n" + "="*60)
        print("💡 التوصيات:")
        print("="*60)
        
        # توصيات بناءً على الأولويات
        for area in report['priority_areas'][:3]:
            if area['priority'] in ['عالية جداً', 'عالية']:
                print(f"\n{area['emoji']} {area['category']}:")
                
                if area['category'] == 'search_failures':
                    print("   - راجع دالة البحث في faiss_menu_search.py")
                    print("   - أضف aliases جديدة للأصناف الشائعة")
                    print("   - حسّن بيانات التدريب")
                
                elif area['category'] == 'intent_errors':
                    print("   - راجع تعريفات النوايا في definitions.py")
                    print("   - أضف كلمات مفتاحية جديدة")
                    print("   - حسّن قواعد intent_guard.py")
                
                elif area['category'] == 'item_not_found':
                    print("   - أضف variations للأصناف في المنيو")
                    print("   - حسّن قاموس التصحيح الإملائي")
                    print("   - أضف tags جديدة")
                
                elif area['category'] == 'dialect_issues':
                    print("   - أضف كلمات لهجية جديدة في text_helpers.py")
                    print("   - راجع دالة normalize_order_text()")
                
                elif area['category'] == 'spelling_errors':
                    print("   - أضف أخطاء إملائية شائعة في word_correction_dict.py")
                    print("   - حسّن دالة normalize_arabic_spelling()")


def main():
    """الدالة الرئيسية"""
    
    print("🚀 محلل الأخطاء - نظام Chatbot المطعم")
    print("="*60)
    
    # إنشاء المحلل
    analyzer = ErrorAnalyzer('logs/app.log')
    
    # تشغيل التحليل
    report = analyzer.analyze()
    
    print("\n✅ اكتمل التحليل!")
    print(f"📄 راجع التقرير الكامل في: analysis/error_report.json")


if __name__ == '__main__':
    main()

