"""
Add Repeat Order Intent to prompt_sections table
"""
import os
from supabase import create_client
from dotenv import load_dotenv

# Load environment variables
load_dotenv()

# Initialize Supabase
supabase = create_client(
    os.getenv('SUPABASE_URL'),
    os.getenv('SUPABASE_KEY')
)

# Repeat Order Intent content
repeat_order_content = """
## 🔄 Intent: Repeat Order

🚨🚨🚨 **MANDATORY INSTRUCTION - FOLLOW EXACTLY!** 🚨🚨🚨

**When customer says "كرر آخر طلب", "repeat last order", "same order", "نفس الطلب":**

**YOU MUST READ THE USER MESSAGE AND FIND "Last Completed Order:" SECTION!**

---

## 📋 DECISION TREE (FOLLOW EXACTLY):

### ✅ CASE A: If you see in the context:
```
Last Completed Order:
- order_id: 10012
- created_at: 2025-10-07...
- status: confirmed
- items_count: 1
```

**THIS MEANS: Customer HAS a previous order!**

**YOU MUST:**
1. Use `"action": "repeat_order"` with `"data": null`
2. Reply in customer's language confirming you'll repeat the order
3. Ask for delivery address

**Example Reply (Arabic):**
"تمام! سأكرر طلبك السابق (رقم 10012). الآن، أرسل موقعك 📍 أو اكتب عنوانك"

**Example Reply (English):**
"Perfect! I'll repeat your previous order (#10012). Now, share your location 📍 or type your address"

---

### ❌ CASE B: If you see in the context:
```
Last Completed Order: None (no previous orders)
```

**THIS MEANS: Customer has NO previous orders!**

**YOU MUST:**
1. Use `"action": null` (NO action)
2. Reply telling them they have no previous orders
3. Ask what they want to order

**Example Reply (Arabic):**
"ليس لديك طلبات سابقة لتكرارها. ماذا تريد أن تطلب؟"

**Example Reply (English):**
"You don't have any previous orders to repeat. What would you like to order?"

---

## 🔍 HOW TO CHECK (MANDATORY):

**STEP 1: Read the USER MESSAGE and find "Last Completed Order:" section**

**STEP 2: Check what it says:**
- If it shows `order_id: 10012` (or any number) → Customer HAS previous orders → Use `"action": "repeat_order"`
- If it shows `None (no previous orders)` → Customer has NO previous orders → Use `"action": null`

**DO NOT GUESS! READ THE CONTEXT!**
"""

def add_repeat_order_intent():
    """Add Repeat Order Intent to prompt_sections"""
    try:
        print("🔄 Adding Repeat Order Intent...")
        
        # Check if already exists
        existing = supabase.table('prompt_sections') \
            .select('*') \
            .eq('section_key', 'intent_repeat_order') \
            .execute()
        
        if existing.data:
            print("⚠️  Repeat Order Intent already exists. Updating...")
            response = supabase.table('prompt_sections') \
                .update({
                    'content': repeat_order_content,
                    'updated_at': 'now()'
                }) \
                .eq('section_key', 'intent_repeat_order') \
                .execute()
            print("✅ Updated Repeat Order Intent")
        else:
            print("➕ Creating new Repeat Order Intent...")
            response = supabase.table('prompt_sections') \
                .insert({
                    'section_key': 'intent_repeat_order',
                    'section_name': 'Repeat Order Intent',
                    'content': repeat_order_content,
                    'priority': 8,
                    'is_required': False,
                    'active': True
                }) \
                .execute()
            print("✅ Created Repeat Order Intent")
        
        print(f"✅ Repeat Order Intent added successfully!")
        
    except Exception as e:
        print(f"❌ Error: {e}")
        raise

if __name__ == "__main__":
    add_repeat_order_intent()

