#!/usr/bin/env python3
"""
Script to add Contact Request intent to the database
"""

import os
import sys
from supabase import create_client
import logging

# Setup logging
logging.basicConfig(
    level=logging.INFO,
    format='%(asctime)s | %(levelname)-8s | %(name)s:%(funcName)s:%(lineno)d - %(message)s',
    datefmt='%Y-%m-%d %H:%M:%S'
)
logger = logging.getLogger(__name__)

def add_contact_request_intent():
    """Add Contact Request intent to prompt_sections table"""
    
    print("="*80)
    print("🔧 Adding Contact Request Intent")
    print("="*80)
    print()
    print("📝 Problem:")
    print("   When customer says 'أريد أتكلم مع أحد', bot doesn't recognize it")
    print("   because intent_contact_request doesn't exist in database!")
    print()
    print("✅ Solution:")
    print("   Create new intent: intent_contact_request")
    print()
    print("="*80)
    print("🔄 Adding intent to database...")
    print("="*80)
    print()
    
    # Load environment variables
    env_vars = {}
    env_path = os.path.join(os.path.dirname(__file__), '..', '.env')
    with open(env_path, 'r') as f:
        for line in f:
            line = line.strip()
            if line and not line.startswith('#') and '=' in line:
                key, value = line.split('=', 1)
                env_vars[key] = value.strip('"').strip("'")
    
    # Initialize Supabase client
    supabase = create_client(
        env_vars['SUPABASE_URL'],
        env_vars['SUPABASE_KEY']
    )
    
    # Contact Request intent content
    content = """## 📞 Intent: Contact Request

**When to use**: Customer wants to talk to a human / contact support

**Trigger phrases**:
- Arabic: "أريد أتكلم مع أحد", "أريد التحدث مع شخص", "أريد التواصل معكم", "اتصلوا بي", "تواصلوا معي", "أريد أتكلم مع الإدارة", "أريد أتكلم مع المدير", "أريد رقم التواصل", "كيف أتواصل معكم"
- English: "I want to talk to someone", "I want to speak to a person", "contact me", "call me", "I want to talk to management", "I want to talk to the manager", "how can I contact you", "contact number"

**How to handle**:

1. **Acknowledge the request**:
   - Arabic: "بالتأكيد! سأوصلك بفريق الدعم. 📞"
   - English: "Of course! I'll connect you with our support team. 📞"

2. **Ask for contact information**:
   - Arabic: "من فضلك، أعطني رقم هاتفك وسنتواصل معك في أقرب وقت."
   - English: "Please provide your phone number and we'll contact you as soon as possible."

3. **If customer provides phone number**:
   - Extract the phone number
   - Confirm: "شكراً! سنتواصل معك على الرقم [NUMBER] قريباً. 📱"
   - Use `action: "escalate_support"` to notify support team

4. **Response Strategy**:
   - Be empathetic and professional
   - Assure them someone will contact them
   - Thank them for reaching out
   - Always respond in the same language as customer's message

**Action Required**:
```json
{
  "intent": "contact_request",
  "reply": "[Your response in customer's language]",
  "action": "escalate_support",
  "data": {
    "phone_number": "[extracted phone number OR null]",
    "reason": "customer_requested_contact"
  }
}
```

**Example Scenarios**:

Scenario 1: Customer requests contact (Arabic)
```
Customer: "أريد أتكلم مع أحد"

✅ CORRECT Response:
{
  "intent": "contact_request",
  "reply": "بالتأكيد! سأوصلك بفريق الدعم. 📞\\n\\nمن فضلك، أعطني رقم هاتفك وسنتواصل معك في أقرب وقت.",
  "action": "escalate_support",
  "data": {
    "phone_number": null,
    "reason": "customer_requested_contact"
  }
}

❌ WRONG Response:
{
  "intent": "small_talk",
  "reply": "يمكنني مساعدتك في أي شيء تحتاجه. كيف يمكنني مساعدتك؟",
  "action": null,
  "data": null
}
```

Scenario 2: Customer provides phone number (Arabic)
```
Customer: "رقمي 99887766"

✅ CORRECT Response:
{
  "intent": "contact_request",
  "reply": "شكراً! سنتواصل معك على الرقم 99887766 في أقرب وقت. 📱\\n\\nهل هناك أي شيء آخر يمكنني مساعدتك به الآن؟",
  "action": "escalate_support",
  "data": {
    "phone_number": "99887766",
    "reason": "customer_requested_contact"
  }
}
```

Scenario 3: Customer requests contact (English)
```
Customer: "I want to talk to someone"

✅ CORRECT Response:
{
  "intent": "contact_request",
  "reply": "Of course! I'll connect you with our support team. 📞\\n\\nPlease provide your phone number and we'll contact you as soon as possible.",
  "action": "escalate_support",
  "data": {
    "phone_number": null,
    "reason": "customer_requested_contact"
  }
}
```

Scenario 4: Customer provides phone number (English)
```
Customer: "My number is 99887766"

✅ CORRECT Response:
{
  "intent": "contact_request",
  "reply": "Thank you! We'll contact you at 99887766 as soon as possible. 📱\\n\\nIs there anything else I can help you with now?",
  "action": "escalate_support",
  "data": {
    "phone_number": "99887766",
    "reason": "customer_requested_contact"
  }
}
```

**Important Notes**:
- Always be professional and empathetic
- Extract phone numbers from messages (look for 8-digit numbers in Oman format)
- Use `action: "escalate_support"` to notify support team
- Confirm the phone number back to the customer
- Respond in the same language as the customer
"""
    
    # Check if intent already exists
    result = supabase.table('prompt_sections').select('*').eq('section_key', 'intent_contact_request').execute()

    if result.data:
        print("⚠️  Intent already exists. Updating...")
        result = supabase.table('prompt_sections').update({
            'content': content,
            'priority': 50
        }).eq('section_key', 'intent_contact_request').execute()
        logger.info("✅ Successfully updated intent_contact_request")
    else:
        print("➕ Creating new intent...")
        result = supabase.table('prompt_sections').insert({
            'section_key': 'intent_contact_request',
            'section_name': 'Contact Request Intent',
            'content': content,
            'priority': 50,
            'is_required': False,
            'active': True
        }).execute()
        logger.info("✅ Successfully created intent_contact_request")
    
    print(f"✅ Contact Request intent added successfully!")
    print(f"Updated/Created {len(result.data)} row(s)")
    print()
    print("📋 Changes made:")
    print("   1. ✅ Created intent_contact_request")
    print("   2. ✅ Added trigger phrases: 'أريد أتكلم مع أحد', 'I want to talk to someone'")
    print("   3. ✅ Added action: 'escalate_support'")
    print("   4. ✅ Added phone number extraction logic")
    print("   5. ✅ Added example scenarios")
    print()
    print("🎯 This fixes the bug where bot doesn't recognize contact requests")
    print()
    print("="*80)
    print("✅ Done!")
    print("="*80)
    print()
    print("📋 Next steps:")
    print("   1. Server will auto-reload (--reload flag)")
    print("   2. Test from WhatsApp: 'أريد أتكلم مع أحد'")
    print("   3. Expected: Bot asks for phone number")
    print()
    print("="*80)

if __name__ == '__main__':
    try:
        add_contact_request_intent()
    except Exception as e:
        logger.error(f"❌ Error: {e}")
        sys.exit(1)

