"""
🧩 قاموس المرادفات والاشتقاقات الصرفية
Word Variants and Morphological Derivations

يوسع القاموس الأساسي بإضافة:
- المرادفات التلقائية
- الاشتقاقات الصرفية
- التصريفات المختلفة
- الأخطاء الإملائية الشائعة
"""

from typing import Dict, List, Set
import re


class WordVariantsGenerator:
    """مولد المرادفات والاشتقاقات"""
    
    def __init__(self):
        # المرادفات الأساسية
        self.arabic_variants: Dict[str, List[str]] = {
            # أفعال الطلب
            "أريد": ["أبغى", "بغيت", "ارغب", "ودي", "عايز", "عاوز", "بدي"],
            "أطلب": ["اطلب", "احجز", "جيب", "هات"],
            
            # التأكيد
            "نعم": ["اي", "ايوه", "ايه", "تمام", "صح", "اوكي", "ماشي", "زين"],
            "لا": ["لأ", "ما ابي", "ما ابغى", "مش عايز", "ما بدي"],
            
            # الكمية
            "خليها": ["خليه", "خلي", "خلوها", "اجعلها", "اجعله"],
            "زود": ["زيد", "ضاعف", "كرر", "أكثر"],
            "قلل": ["نقص", "خفف", "أقل"],
            
            # التعديل
            "الغ": ["احذف", "شيل", "امسح", "بدل", "غير"],
            "عدل": ["غير", "بدل", "صحح"],
            
            # الاستفسار
            "عندكم": ["فيه", "موجود", "متوفر", "هل يوجد"],
            "كم سعر": ["بكم", "كم قيمة", "كم ثمن", "سعره"],
            
            # السلة
            "سلتي": ["السلة", "طلبي", "الطلب", "ايش طلبت"],
            
            # الإكمال
            "اكمل": ["جهز", "ارسل", "خلاص", "تم"],
        }
        
        # المرادفات الإنجليزية
        self.english_variants: Dict[str, List[str]] = {
            "want": ["need", "would like", "wish"],
            "order": ["get", "have", "buy"],
            "yes": ["yeah", "yep", "sure", "ok", "okay"],
            "no": ["nope", "nah", "never"],
            "cancel": ["delete", "remove", "stop"],
            "change": ["modify", "edit", "update"],
            "cart": ["basket", "order"],
        }
        
        # الأخطاء الإملائية الشائعة
        self.common_typos: Dict[str, str] = {
            # أخطاء عربية
            "ابغا": "ابغى",
            "ابي": "ابغى",
            "عاوز": "عايز",
            "ايوا": "ايوه",
            "اوكى": "اوكي",
            "تمان": "تمام",
            "خلاص": "خلاص",
            "الغي": "الغ",
            "احدف": "احذف",
            
            # أخطاء إنجليزية
            "wnat": "want",
            "oder": "order",
            "cancle": "cancel",
            "chagne": "change",
        }
        
        # التصريفات الصرفية
        self.morphological_forms: Dict[str, List[str]] = {
            # المذكر والمؤنث
            "واحد": ["واحدة"],
            "اثنين": ["اثنتين", "ثنتين"],
            "خليه": ["خليها"],
            
            # المفرد والجمع
            "صنف": ["أصناف"],
            "طلب": ["طلبات"],
            "سلة": ["سلات"],
        }
        
        # اللهجات المختلفة لنفس المعنى
        self.dialect_equivalents: Dict[str, List[str]] = {
            "أريد": {
                "خليجي": ["ابغى", "ابي", "بغيت", "ودي"],
                "مصري": ["عايز", "عاوز"],
                "شامي": ["بدي", "بدنا"],
                "عراقي": ["اريد", "ارغب"],
                "مغربي": ["بغيت", "نحب"],
            },
            "نعم": {
                "خليجي": ["ايوه", "اي", "زين"],
                "مصري": ["ايوة", "اه"],
                "شامي": ["اي", "ايه"],
            },
            "عندكم": {
                "خليجي": ["عندكم", "فيه"],
                "مصري": ["عندكو", "فيه"],
                "شامي": ["عندكن", "في"],
            },
        }
    
    def get_all_variants(self, word: str) -> Set[str]:
        """
        الحصول على جميع المرادفات والاشتقاقات لكلمة معينة
        
        Args:
            word: الكلمة الأصلية
            
        Returns:
            مجموعة من جميع المرادفات والاشتقاقات
        """
        variants = {word}
        
        # إضافة المرادفات العربية
        if word in self.arabic_variants:
            variants.update(self.arabic_variants[word])
        
        # إضافة المرادفات الإنجليزية
        if word.lower() in self.english_variants:
            variants.update(self.english_variants[word.lower()])
        
        # إضافة التصريفات الصرفية
        if word in self.morphological_forms:
            variants.update(self.morphological_forms[word])
        
        # إضافة اللهجات المختلفة
        if word in self.dialect_equivalents:
            for dialect_words in self.dialect_equivalents[word].values():
                variants.update(dialect_words)
        
        return variants
    
    def correct_typo(self, word: str) -> str:
        """
        تصحيح الأخطاء الإملائية الشائعة
        
        Args:
            word: الكلمة التي قد تحتوي على خطأ
            
        Returns:
            الكلمة المصححة
        """
        return self.common_typos.get(word, word)
    
    def expand_expression(self, expression: str) -> List[str]:
        """
        توسيع تعبير كامل بإضافة جميع المرادفات الممكنة
        
        Args:
            expression: التعبير الأصلي
            
        Returns:
            قائمة بجميع الصيغ الممكنة للتعبير
        """
        words = expression.split()
        expanded = [expression]
        
        for i, word in enumerate(words):
            variants = self.get_all_variants(word)
            new_expanded = []
            
            for exp in expanded:
                exp_words = exp.split()
                for variant in variants:
                    new_exp = exp_words.copy()
                    new_exp[i] = variant
                    new_expanded.append(" ".join(new_exp))
            
            expanded = new_expanded
        
        return list(set(expanded))
    
    def normalize_numbers(self, text: str) -> str:
        """
        توحيد الأرقام (عربي → إنجليزي)
        
        Args:
            text: النص المحتوي على أرقام
            
        Returns:
            النص بأرقام إنجليزية
        """
        arabic_to_english = {
            '٠': '0', '١': '1', '٢': '2', '٣': '3', '٤': '4',
            '٥': '5', '٦': '6', '٧': '7', '٨': '8', '٩': '9'
        }
        
        for ar, en in arabic_to_english.items():
            text = text.replace(ar, en)
        
        return text
    
    def extract_number_from_text(self, text: str) -> int:
        """
        استخراج الرقم من النص (سواء كان رقماً أو كلمة)
        
        Args:
            text: النص المحتوي على رقم
            
        Returns:
            الرقم المستخرج أو None
        """
        # الأرقام العربية (كلمات)
        arabic_number_words = {
            "واحد": 1, "واحدة": 1,
            "اثنين": 2, "اتنين": 2, "ثنتين": 2, "اثنتين": 2,
            "ثلاثة": 3, "تلاتة": 3, "ثلاث": 3,
            "أربعة": 4, "اربعة": 4, "أربع": 4,
            "خمسة": 5, "خمس": 5,
            "ستة": 6, "ست": 6,
            "سبعة": 7, "سبع": 7,
            "ثمانية": 8, "تمانية": 8, "ثماني": 8,
            "تسعة": 9, "تسع": 9,
            "عشرة": 10, "عشر": 10,
        }
        
        # الأرقام الإنجليزية (كلمات)
        english_number_words = {
            "one": 1, "two": 2, "three": 3, "four": 4, "five": 5,
            "six": 6, "seven": 7, "eight": 8, "nine": 9, "ten": 10,
        }
        
        text = text.lower().strip()
        
        # البحث عن أرقام عربية (كلمات)
        for word, num in arabic_number_words.items():
            if word in text:
                return num
        
        # البحث عن أرقام إنجليزية (كلمات)
        for word, num in english_number_words.items():
            if word in text:
                return num
        
        # البحث عن أرقام (digits)
        text = self.normalize_numbers(text)
        numbers = re.findall(r'\d+', text)
        if numbers:
            return int(numbers[0])
        
        return None


# مثيل عام للاستخدام
word_variants_generator = WordVariantsGenerator()


# دوال مساعدة للاستخدام السريع
def get_variants(word: str) -> Set[str]:
    """الحصول على جميع المرادفات لكلمة"""
    return word_variants_generator.get_all_variants(word)


def correct_typo(word: str) -> str:
    """تصحيح خطأ إملائي"""
    return word_variants_generator.correct_typo(word)


def expand_expression(expression: str) -> List[str]:
    """توسيع تعبير بجميع المرادفات"""
    return word_variants_generator.expand_expression(expression)


def extract_number(text: str) -> int:
    """استخراج رقم من نص"""
    return word_variants_generator.extract_number_from_text(text)


def normalize_numbers(text: str) -> str:
    """توحيد الأرقام"""
    return word_variants_generator.normalize_numbers(text)

