"""
Word Correction Dictionary - قاموس تصحيح الكلمات
يصحح الأخطاء الإملائية واللهجات والكلمات الجزئية قبل التصنيف
"""

from typing import Dict, List, Set, Optional, Tuple
from app.utils.logger import app_logger as logger
import json
import os
import re


class WordCorrectionDict:
    """قاموس تصحيح الكلمات - يصحح الأخطاء الإملائية واللهجات"""
    
    # قاموس التصحيحات الشائعة
    COMMON_CORRECTIONS = {
        # أخطاء إملائية شائعة في أسماء الأصناف
        "ar": {
            # سمك
            "سمج": "سمك",
            "صمك": "سمك",
            "سمگ": "سمك",
            
            # دجاج
            "دجاچ": "دجاج",
            "دياي": "دجاج",
            "ديك": "دجاج",
            
            # هريس
            "هريص": "هريس",
            "حريس": "هريس",
            "هرس": "هريس",
            
            # كبسة
            "كبصة": "كبسة",
            "گبسة": "كبسة",
            "كبسه": "كبسة",
            
            # برجر
            "برقر": "برجر",
            "برغر": "برجر",
            "بيرجر": "برجر",
            "بورجر": "برجر",
            
            # شاورما
            "شورما": "شاورما",
            "شاورمه": "شاورما",
            "شورمة": "شاورما",
            "شاورمة": "شاورما",
            
            # روبيان
            "ربيان": "روبيان",
            "روبيان": "روبيان",
            "جمبري": "روبيان",
            "قمبري": "روبيان",
            
            # حمص
            "حمض": "حمص",
            "حمس": "حمص",
            
            # فتوش
            "فتوچ": "فتوش",
            "فتوص": "فتوش",
            
            # تبولة
            "تبوله": "تبولة",
            "تبولا": "تبولة",
            "طبولة": "تبولة",
            
            # عصير
            "عصر": "عصير",
            "عصیر": "عصير",
            
            # قهوة
            "قهوه": "قهوة",
            "گهوة": "قهوة",
            "كهوة": "قهوة",
            
            # رز
            "رس": "رز",
            "ارز": "رز",
            "ريز": "رز",
            
            # لحم
            "لحوم": "لحم",
            "لحمة": "لحم",
            
            # مقلي
            "مقلى": "مقلي",
            
            # مشوي
            "مشوى": "مشوي",
            "مشویة": "مشوي",
            
            # كلمات عامة
            "عندكم": "عندكم",
            "عندكو": "عندكم",
            "عندك": "عندكم",
            
            "أريد": "أريد",
            "اريد": "أريد",
            "ابي": "أريد",
            "ابغى": "أريد",
            "بدي": "أريد",
            
            "كم": "كم",
            "چم": "كم",
            "گم": "كم",
        },
        
        "en": {
            # fish
            "fich": "fish",
            "fesh": "fish",
            "fsh": "fish",
            
            # chicken
            "chiken": "chicken",
            "chikn": "chicken",
            "chickin": "chicken",
            
            # burger
            "burgr": "burger",
            "burgar": "burger",
            "borgir": "burger",
            
            # shawarma
            "shawerma": "shawarma",
            "shawrma": "shawarma",
            "shaurma": "shawarma",
            
            # shrimp
            "shremp": "shrimp",
            "shreemp": "shrimp",
            
            # rice
            "rize": "rice",
            "ryce": "rice",
            
            # grilled
            "griled": "grilled",
            "griled": "grilled",
            
            # fried
            "fryed": "fried",
            "fride": "fried",
        }
    }
    
    # لهجات مختلفة لنفس الكلمة
    DIALECT_VARIATIONS = {
        "ar": {
            # "أريد"
            "ابي": "أريد",
            "ابغى": "أريد",
            "ابا": "أريد",
            "بدي": "أريد",
            "عايز": "أريد",
            "عاوز": "أريد",
            
            # "عندكم"
            "عندك": "عندكم",
            "عندكو": "عندكم",
            "عندكن": "عندكم",
            
            # "ما"/"ماذا"
            "ايش": "ما",
            "شو": "ما",
            "وش": "ما",
            "شنو": "ما",
            "ايه": "ما",
            
            # "كم"
            "بكم": "كم",
            "بكام": "كم",
            "قديش": "كم",
            "بجم": "كم",
            
            # "نعم"
            "ايوه": "نعم",
            "اه": "نعم",
            "آه": "نعم",
            "ايه": "نعم",
            "زين": "نعم",
            "تمام": "نعم",
            
            # "لا"
            "لأ": "لا",
            "لاء": "لا",
            "مو": "لا",
            "ما": "لا",
        }
    }
    
    def __init__(self):
        """تهيئة قاموس التصحيح"""
        self.menu_keywords: Set[str] = set()
        self.correction_cache: Dict[str, str] = {}
        self._load_menu_keywords()
        self._build_correction_dict()
    
    def _load_menu_keywords(self):
        """تحميل كلمات المنيو من الكاش"""
        try:
            cache_path = os.path.join(
                os.path.dirname(os.path.dirname(os.path.dirname(__file__))),
                'data', 'cache', 'menu_cache.json'
            )
            
            if os.path.exists(cache_path):
                with open(cache_path, 'r', encoding='utf-8') as f:
                    menu_data = json.load(f)
                
                for item in menu_data.get('items', []):
                    # Arabic name
                    name_ar = item.get('name_ar', '')
                    if name_ar:
                        self.menu_keywords.add(name_ar.lower())
                        for word in name_ar.split():
                            if len(word) > 2:
                                self.menu_keywords.add(word.lower())
                    
                    # English name
                    name_en = item.get('name_en', '')
                    if name_en:
                        self.menu_keywords.add(name_en.lower())
                        for word in name_en.lower().split():
                            if len(word) > 2:
                                self.menu_keywords.add(word)
                    
                    # Tags
                    for tag in item.get('tags', []):
                        if len(tag) > 2:
                            self.menu_keywords.add(tag.lower())
                
                logger.info(f"✅ Loaded {len(self.menu_keywords)} menu keywords for correction")
            
        except Exception as e:
            logger.error(f"❌ Error loading menu keywords: {e}")
    
    def _build_correction_dict(self):
        """بناء قاموس التصحيح الكامل"""
        # دمج التصحيحات الشائعة
        for lang in ['ar', 'en']:
            for wrong, correct in self.COMMON_CORRECTIONS[lang].items():
                self.correction_cache[wrong.lower()] = correct.lower()
        
        # دمج اللهجات
        for wrong, correct in self.DIALECT_VARIATIONS['ar'].items():
            self.correction_cache[wrong.lower()] = correct.lower()
        
        logger.info(f"✅ Built correction dictionary with {len(self.correction_cache)} entries")

    def correct_word(self, word: str) -> str:
        """
        تصحيح كلمة واحدة

        Args:
            word: الكلمة المراد تصحيحها

        Returns:
            الكلمة المصححة أو الكلمة الأصلية إذا لم يتم العثور على تصحيح
        """
        word_lower = word.lower().strip()

        # 1. البحث في قاموس التصحيح المباشر
        if word_lower in self.correction_cache:
            corrected = self.correction_cache[word_lower]
            logger.debug(f"🔧 Corrected: '{word}' → '{corrected}'")
            return corrected

        # 2. البحث عن تطابق جزئي مع كلمات المنيو
        # (للأخطاء الإملائية البسيطة)
        if len(word_lower) >= 3:
            for menu_word in self.menu_keywords:
                if self._is_similar(word_lower, menu_word):
                    logger.debug(f"🔧 Fuzzy corrected: '{word}' → '{menu_word}'")
                    return menu_word

        # 3. لم يتم العثور على تصحيح
        return word

    def correct_text(self, text: str) -> Tuple[str, List[Dict[str, str]]]:
        """
        تصحيح نص كامل

        Args:
            text: النص المراد تصحيحه

        Returns:
            Tuple[corrected_text, corrections_made]
            - corrected_text: النص المصحح
            - corrections_made: قائمة بالتصحيحات التي تمت
        """
        words = text.split()
        corrected_words = []
        corrections_made = []

        for word in words:
            # حفظ علامات الترقيم
            punctuation = ""
            clean_word = word

            if word and word[-1] in ".,!?؟،":
                punctuation = word[-1]
                clean_word = word[:-1]

            # تصحيح الكلمة
            corrected = self.correct_word(clean_word)

            # تسجيل التصحيح إذا تم
            if corrected != clean_word.lower():
                corrections_made.append({
                    'original': clean_word,
                    'corrected': corrected,
                    'position': len(corrected_words)
                })

            corrected_words.append(corrected + punctuation)

        corrected_text = " ".join(corrected_words)

        if corrections_made:
            logger.info(f"🔧 Text corrected: {len(corrections_made)} word(s) fixed")
            for correction in corrections_made:
                logger.debug(f"   '{correction['original']}' → '{correction['corrected']}'")

        return corrected_text, corrections_made

    def _is_similar(self, word1: str, word2: str, threshold: float = 0.8) -> bool:
        """
        التحقق من التشابه بين كلمتين باستخدام Levenshtein distance

        Args:
            word1: الكلمة الأولى
            word2: الكلمة الثانية
            threshold: نسبة التشابه المطلوبة (0-1)

        Returns:
            True إذا كانت الكلمتان متشابهتان
        """
        # تجاهل إذا كان الفرق في الطول كبير جداً
        if abs(len(word1) - len(word2)) > 2:
            return False

        # حساب Levenshtein distance
        distance = self._levenshtein_distance(word1, word2)
        max_len = max(len(word1), len(word2))

        if max_len == 0:
            return True

        similarity = 1 - (distance / max_len)
        return similarity >= threshold

    def _levenshtein_distance(self, s1: str, s2: str) -> int:
        """
        حساب Levenshtein distance بين نصين

        Args:
            s1: النص الأول
            s2: النص الثاني

        Returns:
            المسافة بين النصين
        """
        if len(s1) < len(s2):
            return self._levenshtein_distance(s2, s1)

        if len(s2) == 0:
            return len(s1)

        previous_row = range(len(s2) + 1)

        for i, c1 in enumerate(s1):
            current_row = [i + 1]
            for j, c2 in enumerate(s2):
                # Cost of insertions, deletions, or substitutions
                insertions = previous_row[j + 1] + 1
                deletions = current_row[j] + 1
                substitutions = previous_row[j] + (c1 != c2)
                current_row.append(min(insertions, deletions, substitutions))
            previous_row = current_row

        return previous_row[-1]

    def add_correction(self, wrong: str, correct: str):
        """
        إضافة تصحيح جديد إلى القاموس

        Args:
            wrong: الكلمة الخاطئة
            correct: الكلمة الصحيحة
        """
        self.correction_cache[wrong.lower()] = correct.lower()
        logger.info(f"✅ Added correction: '{wrong}' → '{correct}'")

    def get_corrections_count(self) -> int:
        """الحصول على عدد التصحيحات في القاموس"""
        return len(self.correction_cache)

    def get_all_corrections(self) -> Dict[str, str]:
        """الحصول على جميع التصحيحات"""
        return self.correction_cache.copy()


# Singleton instance
word_correction_dict = WordCorrectionDict()

