"""
Text Helper Functions
Utilities for text processing, number conversion, etc.
"""
import re
from typing import Optional
from app.utils.logger import app_logger as logger


def arabic_to_english_numbers(text: str) -> str:
    """
    Convert Arabic-Indic numerals (٠-٩) to Western Arabic numerals (0-9)
    
    Args:
        text: Text containing Arabic numerals
        
    Returns:
        Text with English numerals
        
    Examples:
        >>> arabic_to_english_numbers("١٠٠٠١")
        "10001"
        >>> arabic_to_english_numbers("٢٠٢٥-٠١٩")
        "2025-019"
    """
    arabic_numerals = '٠١٢٣٤٥٦٧٨٩'
    english_numerals = '0123456789'
    
    translation_table = str.maketrans(arabic_numerals, english_numerals)
    return text.translate(translation_table)


def extract_order_number(text: str) -> Optional[str]:
    """
    Extract order number from text message
    
    Supports formats:
    - "2025-019" or "019-2025" (with dash)
    - "10001" or "١٠٠٠١" (5+ digits)
    - "019" or "٠١٩" (3 digits - will add current year)
    
    Args:
        text: Message text
        
    Returns:
        Extracted order number in standard format, or None
        
    Examples:
        >>> extract_order_number("ما حالة طلبي رقم ١٠٠٠١")
        "10001"
        >>> extract_order_number("order status 2025-019")
        "2025-019"
        >>> extract_order_number("وين طلبي 019")
        "2025-019"
    """
    # Convert Arabic numerals to English first
    text = arabic_to_english_numbers(text)
    
    # Pattern 1: Format with dash (2025-019 or 019-2025)
    dash_pattern = r'\b(\d{3,4})-(\d{3,4})\b'
    match = re.search(dash_pattern, text)
    if match:
        part1, part2 = match.groups()
        # Normalize to YYYY-NNN format
        if len(part1) == 4:  # 2025-019
            return f"{part1}-{part2}"
        else:  # 019-2025
            return f"{part2}-{part1}"
    
    # Pattern 2: 4+ digits (1001, 10001, 10002, etc.)
    # This covers both short (1001) and long (10001) order numbers
    number_pattern = r'\b(\d{4,})\b'
    match = re.search(number_pattern, text)
    if match:
        return match.group(1)

    # Pattern 3: 3 digits only (019, etc.) - add current year
    short_number_pattern = r'\b(\d{3})\b'
    match = re.search(short_number_pattern, text)
    if match:
        number = match.group(1)
        from datetime import datetime
        current_year = datetime.now().year
        return f"{current_year}-{number}"

    return None


def detect_language(text: str) -> str:
    """
    Detect if text is Arabic or English
    
    Args:
        text: Text to analyze
        
    Returns:
        'ar' for Arabic, 'en' for English
    """
    # Check for Arabic characters
    arabic_pattern = r'[\u0600-\u06FF]'
    if re.search(arabic_pattern, text):
        return 'ar'
    return 'en'


def normalize_text(text: str) -> str:
    """
    Normalize text for processing
    - Convert Arabic numerals to English
    - Strip extra whitespace
    - Convert to lowercase for English text

    Args:
        text: Text to normalize

    Returns:
        Normalized text
    """
    # Convert Arabic numerals
    text = arabic_to_english_numbers(text)

    # Strip extra whitespace
    text = ' '.join(text.split())

    return text


def clean_user_query(text: str) -> str:
    """
    تنظيف استعلام المستخدم واستخراج المصطلح الأساسي (core term)

    يزيل الكلمات الاستفهامية والحشو ويعيد المصطلح الأساسي فقط

    Args:
        text: النص الأصلي

    Returns:
        المصطلح الأساسي (core term)

    Examples:
        >>> clean_user_query("عندكم هريس؟")
        "هريس"
        >>> clean_user_query("هل لديك سمك مقلي؟")
        "سمك مقلي"
        >>> clean_user_query("do you have fish?")
        "fish"
    """
    return clean_query_for_search(text)


def clean_query_for_search(text: str) -> str:
    """
    تنظيف النص قبل البحث في المنيو
    يزيل الكلمات الاستفهامية والحشو التي لا تساعد في البحث

    الكلمات المحذوفة:
    - عربي: عندكم، عندك، فيه، في، هل، لو، ممكن، أريد، ابي، بدي، عايز
    - إنجليزي: do you have, have you, is there, can i get, i want

    Args:
        text: النص الأصلي

    Returns:
        النص المنظف للبحث

    Examples:
        >>> clean_query_for_search("عندكم سمك؟")
        "سمك"
        >>> clean_query_for_search("هل فيه هريس دجاج؟")
        "هريس دجاج"
        >>> clean_query_for_search("ابي كبسة عوان")
        "كبسة عوان"
        >>> clean_query_for_search("do you have fish?")
        "fish"
    """
    # قائمة الكلمات الاستفهامية والحشو بالعربي
    arabic_stopwords = [
        # تحيات (الأطول أولاً)
        r'\bالسلام عليكم ورحمة الله وبركاته\b',
        r'\bالسلام عليكم ورحمة الله\b',
        r'\bالسلام عليكم\b',
        r'\bوعليكم السلام\b',
        r'\bعليكم السلام\b',
        r'\bالسلام\b',
        r'\bعليكم\b',

        # أسئلة (الأطول أولاً)
        r'\bلو سمحت\b',
        r'\bلو سمحتوا\b',
        r'\bمن فضلك\b',
        r'\bمن فضلكم\b',
        r'\bعندكم\b',
        r'\bعندك\b',
        r'\bعنده\b',
        r'\bعندها\b',
        r'\bفيه\b',
        r'\bفي\b',
        r'\bهل\b',
        r'\bلو\b',

        # طلبات
        r'\bممكن\b',
        r'\bأريد\b',
        r'\bاريد\b',
        r'\bابي\b',
        r'\bابغي\b',  # إضافة "ابغي" بالياء العادية
        r'\bابغى\b',
        r'\bبدي\b',
        r'\bعايز\b',
        r'\bعاوز\b',
        r'\bبغيت\b',

        # استفهام
        r'\bايش\b',
        r'\bإيش\b',
        r'\bشو\b',
        r'\bوش\b',
        r'\bشنو\b',
        r'\bوين\b',
        r'\bفين\b',
        r'\bكيف\b',
        r'\bمتى\b',

        # حروف
        r'\bمن\b',
        r'\bإلى\b',
        r'\bالى\b',
        r'\bمع\b',

        # علامات استفهام وتعجب
        r'\?',
        r'!',
        r'؟',
        r'،',
    ]

    # قائمة الكلمات الاستفهامية بالإنجليزي (الأطول أولاً)
    english_stopwords = [
        r'\bhave you got\b',
        r'\bdo you have\b',
        r'\bhave you\b',
        r'\bcould i have\b',
        r'\bcould i get\b',
        r'\bcan i have\b',
        r'\bcan i get\b',
        r'\bi would like\b',
        r'\bi\'d like\b',
        r'\bis there\b',
        r'\bare there\b',
        r'\bi want\b',
        r'\bwhat is\b',
        r'\bwhat\b',
        r'\bwhere\b',
        r'\bwhen\b',
        r'\bhow\b',
        r'\bwhy\b',
        r'\bwhich\b',
        r'\bplease\b',
        r'\bany\b',
        r'\bsome\b',
        r'\bthe\b',
        r'\bis\b',
        r'\bgot\b',
        r'\bwith\b',
        r'\ba\b',
        r'\ban\b',

        # علامات
        r'\?',
        r'!',
        r',',
    ]

    # تنظيف النص
    cleaned = text.strip()

    # إزالة الكلمات العربية
    for pattern in arabic_stopwords:
        cleaned = re.sub(pattern, ' ', cleaned, flags=re.IGNORECASE)

    # إزالة الكلمات الإنجليزية
    for pattern in english_stopwords:
        cleaned = re.sub(pattern, ' ', cleaned, flags=re.IGNORECASE)

    # إزالة المسافات الزائدة
    cleaned = ' '.join(cleaned.split())

    return cleaned.strip()


def extract_quantity_and_item(text: str) -> tuple:
    """
    استخراج الكمية واسم الصنف من النص

    Args:
        text: النص المدخل من المستخدم

    Returns:
        tuple: (quantity: int, item_name: str)

    Examples:
        "عايز ٣ هريس" → (3, "هريس")
        "أبغى 2 شوربة عدس" → (2, "شوربة عدس")
        "بغيت هريس" → (1, "هريس")
        "واحد كبسة لحم" → (1, "كبسة لحم")
    """
    import re

    # تطبيع النص أولاً
    normalized = normalize_order_text(text)

    # أنماط الكمية
    quantity_patterns = [
        # أرقام إنجليزية
        (r'\b(\d+)\s*', 1),
        # أرقام عربية
        (r'([٠-٩]+)\s*', 1),
        # كلمات عربية
        (r'\b(واحد|وحدة)\b', 1),
        (r'\b(اثنين|اتنين|ثنتين)\b', 2),
        (r'\b(ثلاثة|تلاتة)\b', 3),
        (r'\b(أربعة|اربعة|أربع)\b', 4),
        (r'\b(خمسة|خمس)\b', 5),
        (r'\b(ستة|ست)\b', 6),
        (r'\b(سبعة|سبع)\b', 7),
        (r'\b(ثمانية|تمانية|ثماني)\b', 8),
        (r'\b(تسعة|تسع)\b', 9),
        (r'\b(عشرة|عشر)\b', 10),
        # صيغ أخرى
        (r'\b(طبق|طبقين|طبقان)\b', lambda m: 2 if 'ين' in m.group(1) or 'ان' in m.group(1) else 1),
    ]

    quantity = 1  # القيمة الافتراضية
    item_text = normalized

    # البحث عن الكمية
    for pattern, qty_value in quantity_patterns:
        match = re.search(pattern, normalized, re.IGNORECASE)
        if match:
            # تحديد الكمية
            if callable(qty_value):
                quantity = qty_value(match)
            else:
                # إذا كان رقم، استخرجه
                if qty_value == 1 and match.group(1).isdigit():
                    quantity = int(match.group(1))
                # إذا كان رقم عربي، حوله
                elif qty_value == 1 and re.match(r'[٠-٩]+', match.group(1)):
                    arabic_to_english = str.maketrans('٠١٢٣٤٥٦٧٨٩', '0123456789')
                    quantity = int(match.group(1).translate(arabic_to_english))
                else:
                    quantity = qty_value

            # إزالة الكمية من النص
            item_text = re.sub(pattern, '', normalized, count=1).strip()
            break

    # تنظيف اسم الصنف من كلمات الطلب والعبارات الزائدة
    order_words = [
        # أفعال الطلب
        'أريد', 'اريد', 'ابي', 'ابغى', 'بغيت', 'عايز', 'نفسي في', 'هات', 'جيب',
        'خذ', 'حط', 'زيد', 'ضيف', 'أضف', 'اضف', 'سجل', 'جهز', 'اضبط',
        'أبغى', 'أبي', 'بدي', 'بدّي', 'ودي', 'ودّي', 'أوصي', 'اوصي',

        # عبارات التجربة
        'أجرب', 'اجرب', 'نجرب', 'جرب', 'أذوق', 'اذوق',

        # ظروف الزمان
        'اليوم', 'يوم', 'الحين', 'حين', 'الآن', 'آن', 'بكرة', 'غداً', 'غدا',
        'من أمس', 'أمس', 'من زمان', 'زمان',

        # عبارات الطلب
        'لو سمحت', 'سمحت', 'تكرمت', 'لو تكرمت', 'من فضلك', 'فضلك',
        'الله يعطيك العافية', 'يعطيك العافية', 'العافية',
        'تكفى', 'تكفا', 'لا تنسى', 'تنسى', 'لا تنسَ',

        # عبارات السرعة
        'بسرعة', 'سريع', 'ضروري', 'بدون تأخير', 'تأخير', 'يلا',

        # عبارات أخرى
        'ممم', 'شكله', 'لذيذ', 'يشهي', 'خلني', 'أطلبه', 'اطلبه',
        'صراحة', 'نفسي بـ', 'نفسي ب', 'سمعت كثير عن', 'كثير عن',
        'خل', 'خلاص', 'قررت', 'أعتقد', 'اعتقد', 'بيناسبني', 'يناسبني',
        'ما لي خلق', 'لي خلق', 'أطبخ', 'اطبخ',
        'جاهز', 'متوفر', 'فيه', 'في', 'إذا', 'اذا',
        'هاه', 'ترى', 'جوعان', 'أطلب', 'اطلب',
        'ممكن', 'السلام عليكم', 'عليكم السلام', 'وعليكم السلام',
        'السلام', 'عليكم', 'وعليكم', 'سلام',
        'الرحمة', 'والرحمة', 'البركات', 'والبركات', 'ورحمة الله وبركاته',
        'رحمة', 'بركات',
        'كم سعر', 'سعر', 'طيب', 'خلاص',
        'بدون تفكير', 'تفكير', 'أحسن شي', 'احسن شي',
        'اشتقت لـ', 'اشتقت ل', 'للمرة الأولى', 'المرة الأولى',
        'للغداء', 'الغداء', 'للعشاء', 'العشاء',
        'خفيف', 'زيادة على الطلب', 'على الطلب', 'الطلب', 'طلب',
        'في القائمة', 'القائمة', 'قائمة', 'مع الطلب',
        'قبل ما أغير رأيي', 'أغير رأيي', 'رأيي',
        'عطِني', 'عطني', 'أرسل الرقم', 'الرقم',
        'مع مشروب بارد', 'مشروب بارد', 'بارد',
        'مع السلطة الجانبية', 'السلطة الجانبية', 'الجانبية',
        'منور', 'المطعم', 'وش', 'تجهز', 'لك', 'اليوم',
        'بعد', 'مع', 'و', 'ال', 'لي', 'لنا'
    ]

    for word in order_words:
        # استخدام word boundaries للتأكد من مطابقة الكلمة كاملة
        item_text = re.sub(rf'\b{re.escape(word)}\b', '', item_text, flags=re.IGNORECASE).strip()

    # إزالة علامات الترقيم الزائدة
    item_text = re.sub(r'[،؟!.,:;]', '', item_text)

    # إزالة "بـ" و "ب" في البداية
    item_text = re.sub(r'^(بـ|ب)\s+', '', item_text)

    # إزالة "يا ليت" في البداية
    item_text = re.sub(r'^(يا ليت|ياليت)\s+', '', item_text, flags=re.IGNORECASE)

    # إزالة "الناس تمدحه/تمدحة" في النهاية
    item_text = re.sub(r'\s+(الناس تمدحه|الناس تمدحة|ناس تمدحه|ناس تمدحة)$', '', item_text, flags=re.IGNORECASE)

    # تنظيف نهائي
    item_text = re.sub(r'\s+', ' ', item_text).strip()

    # التأكد من أن الكمية معقولة
    if quantity < 1:
        quantity = 1
    elif quantity > 99:
        quantity = 99

    return quantity, item_text


def normalize_arabic_spelling(text: str) -> str:
    """
    توحيد الاختلافات الإملائية في النص العربي

    يقوم بـ:
    1. توحيد الهمزات (أ، إ، آ، ا)
    2. توحيد الياءات (ي، ى)
    3. توحيد التاء المربوطة والهاء (ة، ه)
    4. إزالة التشكيل
    5. توحيد الألف المقصورة
    6. إزالة التاء المربوطة في نهاية الكلمات (هريسة → هريس)

    Args:
        text: النص الأصلي

    Returns:
        النص الموحد

    Examples:
        >>> normalize_arabic_spelling("إبغى")
        "ابغي"
        >>> normalize_arabic_spelling("كبسه")
        "كبسة"
        >>> normalize_arabic_spelling("هريسة")
        "هريس"
    """
    import re

    # إزالة التشكيل
    text = re.sub(r'[\u064B-\u065F]', '', text)

    # توحيد الهمزات → ا
    text = re.sub(r'[إأآ]', 'ا', text)

    # توحيد الياءات → ي
    text = re.sub(r'ى', 'ي', text)

    # توحيد التاء المربوطة والهاء في نهاية الكلمات
    # ه في نهاية الكلمة بعد حرف → ة
    text = re.sub(r'ه\b', 'ة', text)

    # إزالة التاء المربوطة في نهاية الكلمات (اختياري - لتوحيد أفضل)
    # هريسة → هريس، كبسة → كبس (لكن نبقي على بعض الكلمات)
    # نطبق فقط على كلمات معينة
    food_words_with_ta = [
        (r'\bهريسة\b', 'هريس'),
        (r'\bكبسة\b', 'كبسة'),  # نبقيها
        (r'\bشوربة\b', 'شوربة'),  # نبقيها
        (r'\bسلطة\b', 'سلطة'),  # نبقيها
        (r'\bفرخة\b', 'فرخ'),
        (r'\bسمكة\b', 'سمك'),
        (r'\bلحمة\b', 'لحم'),
    ]

    for pattern, replacement in food_words_with_ta:
        text = re.sub(pattern, replacement, text, flags=re.IGNORECASE)

    return text


def normalize_order_text(text: str) -> str:
    """
    تطبيع نص الطلب - نسخة محسّنة مع 150+ كلمة لهجية

    التحسينات:
    - 150+ كلمة لهجية (خليجي، مصري، شامي، عراقي، مغربي، يمني، سوداني)
    - معالجة أفضل للهمزات والياءات
    - إزالة "ال" التعريف بشكل أفضل
    - توحيد الأرقام العربية والإنجليزية
    - توحيد مرادفات الطعام (فراخ→دجاج، عيش→رز، لحمة→لحم)

    Args:
        text: النص الأصلي

    Returns:
        النص المطبّع

    Examples:
        >>> normalize_order_text("إبي هريس")
        "أريد هريس"
        >>> normalize_order_text("عايز فراخ")
        "أريد دجاج"
        >>> normalize_order_text("بجم الكبسة")
        "كم كبسة"
    """
    # 1. توحيد الاختلافات الإملائية أولاً
    text = normalize_arabic_spelling(text)

    # 2. قاموس توحيد اللهجات الموسّع (150+ كلمة)
    dialect_normalization = {
        # === خليجي ===
        'ابي': 'أريد', 'ابغي': 'أريد', 'ابغا': 'أريد', 'بغيت': 'أريد',
        'ودي': 'أريد', 'ودني': 'أريد', 'عطني': 'أريد', 'عطنا': 'أريد',
        'عطيني': 'أريد', 'جيب لي': 'أريد', 'جيبلي': 'أريد',
        'خذ لي': 'أريد', 'خذلي': 'أريد', 'خذ': 'أريد',
        'حط لي': 'أريد', 'حطلي': 'أريد', 'حط': 'أريد',
        'خلي': 'أريد', 'خليها': 'أريد', 'خليه': 'أريد',
        'زيد لي': 'أريد', 'زيدلي': 'أريد', 'زيد': 'أريد', 'زيدني': 'أريد',
        'ضيف لي': 'أريد', 'ضيفلي': 'أريد', 'ضيف': 'أريد',
        'شحالة': 'كم', 'شحالته': 'كم', 'بجم': 'كم', 'بكام': 'كم', 'بكم': 'كم',
        'وش': 'ما', 'ويش': 'ما', 'شنو': 'ما', 'شو': 'ما', 'ايش': 'ما',
        'وين': 'أين', 'وينه': 'أين', 'وينها': 'أين',
        'يالله': 'هيا', 'تكفي': 'من فضلك', 'تكفون': 'من فضلكم',
        'الحين': 'الان', 'حاليا': 'الان',

        # === مصري ===
        'عايز': 'أريد', 'عاوز': 'أريد', 'عاوزة': 'أريد', 'عايزة': 'أريد',
        'عايزين': 'نريد', 'عاوزين': 'نريد', 'نفسي في': 'أريد', 'نفسي': 'أريد',
        'هاتلي': 'أريد', 'هات': 'أريد', 'هاتي': 'أريد',
        'جيبلي': 'أريد', 'جيب': 'أريد', 'جيبي': 'أريد',
        'حطلي': 'أريد', 'ضيفلي': 'أريد',
        'خليهلي': 'أريد', 'خليه لي': 'أريد', 'سجلي': 'أريد', 'سجل': 'أريد',
        'بقد اية': 'كم',
        'اية': 'ما', 'اية دة': 'ما هذا', 'فين': 'أين', 'فينه': 'أين', 'فينها': 'أين',
        'ازاي': 'كيف', 'امتي': 'متى', 'لية': 'لماذا',
        'فراخ': 'دجاج', 'فرخة': 'دجاج', 'فرخ': 'دجاج', 'فراخ': 'دجاج',
        'عيش': 'رز', 'ارز': 'رز', 'لحمة': 'لحم', 'سمكة': 'سمك',
        'بكام': 'كم',  # إضافة "بكام" للمصري
        'دلوقتي': 'الان', 'دلوقت': 'الان', 'حالا': 'الان',
        'يلا': 'هيا', 'ماشي': 'حسنا', 'تمام': 'حسنا', 'اوكي': 'حسنا',

        # === شامي ===
        'بدي': 'أريد', 'بدنا': 'نريد', 'بدة': 'يريد', 'بدها': 'تريد',
        'بدهم': 'يريدون', 'بدك': 'تريد', 'بدكم': 'تريدون',
        'جبلي': 'أريد', 'زيدلي': 'أريد',
        'قديش': 'كم', 'اديش': 'كم', 'كديش': 'كم', 'شو سعر': 'كم سعر',
        'شي': 'شيء', 'ليش': 'لماذا', 'لية': 'لماذا',
        'هلا': 'الان', 'هلق': 'الان',

        # === عراقي ===
        'اريد': 'أريد', 'نريد': 'نريد', 'تريد': 'تريد', 'يريد': 'يريد',
        'هاتلي': 'أريد',
        'جم': 'كم', 'شكد': 'كم', 'شلون': 'كيف',
        'اكو': 'يوجد', 'ماكو': 'لا يوجد', 'موجود': 'يوجد',
        'هسة': 'الان', 'هسع': 'الان',

        # === مغربي ===
        'نحب': 'أريد', 'نبغي': 'أريد', 'نزيد': 'أريد', 'حطني': 'أريد',
        'شحال': 'كم', 'بشحال': 'كم',
        'واش': 'هل', 'اش': 'ما', 'كيفاش': 'كيف', 'علاش': 'لماذا',
        'دابا': 'الان', 'دبا': 'الان', 'واخا': 'حسنا', 'مزيان': 'جيد', 'بصح': 'لكن',

        # === أفعال أمر مباشرة ===
        'أضف لي': 'أريد', 'أضف': 'أريد', 'ضع لي': 'أريد', 'ضع': 'أريد',
        'جهز لي': 'أريد', 'جهز': 'أريد', 'اضبط لي': 'أريد', 'اضبط': 'أريد',
        'أرغب في': 'أريد', 'أرغب': 'أريد',

        # === صيغ رقمية ===
        'واحد': '1', 'وحدة': '1', 'اثنين': '2', 'اتنين': '2', 'ثنتين': '2',
        'ثلاثة': '3', 'تلاتة': '3', 'أربعة': '4', 'اربعة': '4', 'أربع': '4',
        'خمسة': '5', 'خمس': '5', 'ستة': '6', 'ست': '6',
        'سبعة': '7', 'سبع': '7', 'ثمانية': '8', 'تمانية': '8', 'ثماني': '8',
        'تسعة': '9', 'تسع': '9', 'عشرة': '10', 'عشر': '10',
        'طبقين': '2', 'طبقان': '2', 'طبق': '1',

        # === عبارات مهذبة (تحذف) ===
        'لو سمحت': '', 'لو سمحتوا': '', 'من فضلك': '', 'من فضلكم': '',
        'الله يخليك': '', 'الله يسعدك': '', 'تكرم': '', 'تكرمون': '',
        'رجاءا': '', 'رجاء': '', 'please': '', 'thank you': '', 'thanks': '',
    }

    # 3. تطبيق التوحيد (الأطول أولاً لتجنب التعارضات)
    normalized = text.strip()
    sorted_keys = sorted(dialect_normalization.keys(), key=len, reverse=True)

    for dialect_word in sorted_keys:
        standard_word = dialect_normalization[dialect_word]
        # استخدام word boundaries للتطابق الدقيق
        pattern = r'\b' + re.escape(dialect_word) + r'\b'
        normalized = re.sub(pattern, standard_word, normalized, flags=re.IGNORECASE)

    # 4. إزالة "ال" التعريف و "و" العطف
    # إزالة "ال" في بداية الكلمات
    normalized = re.sub(r'\bال([^\s]+)', r'\1', normalized)
    # إزالة "الـ" (مع الشرطة)
    normalized = re.sub(r'الـ', '', normalized)
    # إزالة "و" العطف في بداية الكلمات (والكبسة → كبسة)
    normalized = re.sub(r'\bو([^\s]+)', r'\1', normalized)

    # 5. توحيد الأرقام
    normalized = arabic_to_english_numbers(normalized)

    # 6. إزالة علامات الترقيم
    normalized = re.sub(r'[؟?!،,.]', ' ', normalized)

    # 7. إزالة المسافات الزائدة
    normalized = ' '.join(normalized.split())

    return normalized.strip()


def combine_if_related(last_message: str, new_message: str, time_diff_seconds: float = 10.0) -> Optional[str]:
    """
    دمج رسالتين إذا كانتا مرتبطتين ومتقاربتين زمنياً

    يستخدم لدمج الرسائل المتقطعة مثل:
    - "أريد شوربة" + "عدس" = "أريد شوربة عدس"
    - "هريس" + "دجاج" = "هريس دجاج"

    Args:
        last_message: الرسالة السابقة
        new_message: الرسالة الجديدة
        time_diff_seconds: الفرق الزمني المسموح (بالثواني)

    Returns:
        الرسالة المدمجة أو None إذا لم يكن الدمج مناسباً
    """
    # تنظيف الرسائل
    last_clean = last_message.strip().lower()
    new_clean = new_message.strip().lower()

    # إذا كانت الرسالة الجديدة قصيرة جداً (كلمة واحدة أو كلمتين)
    new_words = new_clean.split()
    if len(new_words) <= 2:
        # قد تكون استكمال للرسالة السابقة

        # تحقق من أن الرسالة السابقة تحتوي على كلمة طلب
        order_keywords = ['أريد', 'ابغى', 'ابي', 'بدي', 'عايز', 'طلب']
        has_order_keyword = any(kw in last_clean for kw in order_keywords)

        if has_order_keyword:
            # دمج الرسالتين
            combined = f"{last_message} {new_message}"
            logger.info(f"🔗 Combined messages: '{last_message}' + '{new_message}' = '{combined}'")
            return combined

    # لا يمكن الدمج
    return None

