"""
Question Context Manager - مدير سياق الأسئلة
يضمن طرح سؤال واحد فقط في كل مرة ويتتبع السياق
"""

from enum import Enum
from typing import Dict, Optional, Any
from datetime import datetime
from app.utils.logger import app_logger as logger


class QuestionType(str, Enum):
    """أنواع الأسئلة"""
    
    # أسئلة الطلب
    ADD_MORE_ITEMS = "add_more_items"           # هل تريد إضافة المزيد من الأصناف؟
    CONFIRM_ORDER = "confirm_order"             # هل تريد تأكيد الطلب؟
    CONFIRM_QUANTITY = "confirm_quantity"       # هل الكمية صحيحة؟
    ADD_EXTRAS = "add_extras"                   # هل تريد إضافات؟
    
    # أسئلة العنوان والدفع
    PROVIDE_ADDRESS = "provide_address"         # هل تريد إرسال الموقع؟
    CONFIRM_ADDRESS = "confirm_address"         # هل العنوان صحيح؟
    CHOOSE_PAYMENT = "choose_payment"           # اختر طريقة الدفع
    CONFIRM_PAYMENT = "confirm_payment"         # هل طريقة الدفع صحيحة؟
    
    # أسئلة عامة
    CANCEL_ORDER = "cancel_order"               # هل تريد إلغاء الطلب؟
    CLEAR_CART = "clear_cart"                   # هل تريد حذف السلة؟
    REORDER = "reorder"                         # هل تريد إعادة الطلب السابق؟
    
    # لا يوجد سؤال نشط
    NONE = "none"


class QuestionContextManager:
    """مدير سياق الأسئلة"""
    
    # رسائل الأسئلة (عربي)
    QUESTION_MESSAGES_AR: Dict[QuestionType, str] = {
        QuestionType.ADD_MORE_ITEMS: "هل تريد إضافة المزيد من الأصناف؟ 🛒",
        QuestionType.CONFIRM_ORDER: "هل تريد تأكيد الطلب والمتابعة للدفع؟ ✅",
        QuestionType.CONFIRM_QUANTITY: "هل الكمية صحيحة؟",
        QuestionType.ADD_EXTRAS: "هل تريد إضافة أي إضافات للطلب؟",
        QuestionType.PROVIDE_ADDRESS: "📍 أرسل موقعك الآن\n(اضغط 📎 ← Location)",
        QuestionType.CONFIRM_ADDRESS: "هل العنوان صحيح؟",
        QuestionType.CHOOSE_PAYMENT: "💵 اختر طريقة الدفع:\n\n1️⃣ كاش عند الاستلام 💵\n2️⃣ بطاقة ائتمانية 💳\n\nأرسل رقم الخيار أو اسم طريقة الدفع",
        QuestionType.CONFIRM_PAYMENT: "هل طريقة الدفع صحيحة؟",
        QuestionType.CANCEL_ORDER: "هل أنت متأكد من إلغاء الطلب؟ ❌",
        QuestionType.CLEAR_CART: "هل تريد حذف جميع الأصناف من السلة؟ 🛍️",
        QuestionType.REORDER: "هل تريد إعادة طلبك السابق؟ 🔄"
    }

    # رسائل الأسئلة (إنجليزي)
    QUESTION_MESSAGES_EN: Dict[QuestionType, str] = {
        QuestionType.ADD_MORE_ITEMS: "Would you like to add more items? 🛒",
        QuestionType.CONFIRM_ORDER: "Would you like to confirm the order and proceed to payment? ✅",
        QuestionType.CONFIRM_QUANTITY: "Is the quantity correct?",
        QuestionType.ADD_EXTRAS: "Would you like to add any extras to your order?",
        QuestionType.PROVIDE_ADDRESS: "📍 Send your location now\n(Tap 📎 ← Location)",
        QuestionType.CONFIRM_ADDRESS: "Is the address correct?",
        QuestionType.CHOOSE_PAYMENT: "💵 Choose payment method:\n\n1️⃣ Cash on delivery 💵\n2️⃣ Credit card 💳\n\nSend the number or payment method name",
        QuestionType.CONFIRM_PAYMENT: "Is the payment method correct?",
        QuestionType.CANCEL_ORDER: "Are you sure you want to cancel the order? ❌",
        QuestionType.CLEAR_CART: "Do you want to clear all items from the cart? 🛍️",
        QuestionType.REORDER: "Would you like to reorder your last order? 🔄"
    }
    
    # رسائل "نعم" لكل نوع سؤال (عربي)
    YES_RESPONSES_AR: Dict[QuestionType, str] = {
        QuestionType.ADD_MORE_ITEMS: "تمام! اختر الصنف الذي تريد إضافته 🛒",
        QuestionType.CONFIRM_ORDER: "ممتاز! ✅",  # 🆕 سيتم إضافة سؤال الموقع بعدها
        QuestionType.CONFIRM_QUANTITY: "تمام! الكمية مؤكدة ✓",
        QuestionType.ADD_EXTRAS: "حسناً، ما الإضافات التي تريدها؟",
        QuestionType.PROVIDE_ADDRESS: "رائع! 📍",  # 🆕 سيتم إضافة سؤال الدفع بعدها
        QuestionType.CONFIRM_ADDRESS: "تمام! العنوان مؤكد ✓",
        QuestionType.CHOOSE_PAYMENT: "اختر طريقة الدفع:",
        QuestionType.CONFIRM_PAYMENT: "تمام! طريقة الدفع مؤكدة ✓",
        QuestionType.CANCEL_ORDER: "تم إلغاء الطلب ❌",
        QuestionType.CLEAR_CART: "تم حذف السلة 🛍️",
        QuestionType.REORDER: "جاري إعادة طلبك السابق... 🔄"
    }

    # رسائل "نعم" لكل نوع سؤال (إنجليزي)
    YES_RESPONSES_EN: Dict[QuestionType, str] = {
        QuestionType.ADD_MORE_ITEMS: "Great! Choose the item you want to add 🛒",
        QuestionType.CONFIRM_ORDER: "Excellent! ✅",  # 🆕 Location question will be added after
        QuestionType.CONFIRM_QUANTITY: "Great! Quantity confirmed ✓",
        QuestionType.ADD_EXTRAS: "Okay, what extras would you like?",
        QuestionType.PROVIDE_ADDRESS: "Great! 📍",  # 🆕 Payment question will be added after
        QuestionType.CONFIRM_ADDRESS: "Great! Address confirmed ✓",
        QuestionType.CHOOSE_PAYMENT: "Choose payment method:",
        QuestionType.CONFIRM_PAYMENT: "Great! Payment method confirmed ✓",
        QuestionType.CANCEL_ORDER: "Order cancelled ❌",
        QuestionType.CLEAR_CART: "Cart cleared 🛍️",
        QuestionType.REORDER: "Reordering your last order... 🔄"
    }
    
    # رسائل "لا" لكل نوع سؤال (عربي)
    NO_RESPONSES_AR: Dict[QuestionType, str] = {
        QuestionType.ADD_MORE_ITEMS: "حسناً! هل تريد تأكيد الطلب؟",
        QuestionType.CONFIRM_ORDER: "لا مشكلة! يمكنك تعديل الطلب أو إضافة المزيد",
        QuestionType.CONFIRM_QUANTITY: "حسناً، كم الكمية التي تريدها؟",
        QuestionType.ADD_EXTRAS: "تمام! لا إضافات",
        QuestionType.PROVIDE_ADDRESS: "حسناً، اكتب عنوانك بالتفصيل",
        QuestionType.CONFIRM_ADDRESS: "حسناً، أرسل العنوان الصحيح",
        QuestionType.CHOOSE_PAYMENT: "اختر طريقة دفع أخرى:",
        QuestionType.CONFIRM_PAYMENT: "حسناً، اختر طريقة دفع أخرى",
        QuestionType.CANCEL_ORDER: "تمام! الطلب لا يزال نشطاً",
        QuestionType.CLEAR_CART: "تمام! السلة لا تزال كما هي",
        QuestionType.REORDER: "حسناً! يمكنك اختيار أصناف جديدة"
    }
    
    # رسائل "لا" لكل نوع سؤال (إنجليزي)
    NO_RESPONSES_EN: Dict[QuestionType, str] = {
        QuestionType.ADD_MORE_ITEMS: "Okay! Would you like to confirm the order?",
        QuestionType.CONFIRM_ORDER: "No problem! You can modify the order or add more",
        QuestionType.CONFIRM_QUANTITY: "Okay, what quantity would you like?",
        QuestionType.ADD_EXTRAS: "Okay! No extras",
        QuestionType.PROVIDE_ADDRESS: "Okay, write your address in detail",
        QuestionType.CONFIRM_ADDRESS: "Okay, send the correct address",
        QuestionType.CHOOSE_PAYMENT: "Choose another payment method:",
        QuestionType.CONFIRM_PAYMENT: "Okay, choose another payment method",
        QuestionType.CANCEL_ORDER: "Okay! Order is still active",
        QuestionType.CLEAR_CART: "Okay! Cart remains as is",
        QuestionType.REORDER: "Okay! You can choose new items"
    }
    
    def set_question(
        self,
        session: Dict,
        question_type: QuestionType,
        context_data: Optional[Dict[str, Any]] = None
    ) -> Dict:
        """
        تعيين سؤال نشط في السياق
        
        Args:
            session: بيانات الجلسة
            question_type: نوع السؤال
            context_data: بيانات إضافية للسياق
            
        Returns:
            بيانات السياق المحدثة
        """
        question_context = {
            'type': question_type.value,
            'asked_at': datetime.utcnow().isoformat(),
            'data': context_data or {}
        }
        
        session['last_question'] = question_context
        
        logger.info(f"❓ Question set: {question_type.value}")
        
        return question_context
    
    def get_question(self, session: Dict) -> Optional[QuestionType]:
        """
        الحصول على السؤال النشط
        
        Args:
            session: بيانات الجلسة
            
        Returns:
            نوع السؤال أو None
        """
        question_context = session.get('last_question')
        
        if not question_context:
            return None
        
        question_type_str = question_context.get('type')
        
        if not question_type_str or question_type_str == QuestionType.NONE.value:
            return None
        
        try:
            return QuestionType(question_type_str)
        except ValueError:
            logger.warning(f"⚠️ Invalid question type: {question_type_str}")
            return None
    
    def clear_question(self, session: Dict) -> None:
        """
        مسح السؤال النشط
        
        Args:
            session: بيانات الجلسة
        """
        if 'last_question' in session:
            old_type = session['last_question'].get('type')
            session['last_question'] = {'type': QuestionType.NONE.value}
            logger.info(f"✓ Question cleared: {old_type}")
    
    def get_question_message(
        self,
        question_type: QuestionType,
        lang: str = 'ar'
    ) -> str:
        """
        الحصول على رسالة السؤال
        
        Args:
            question_type: نوع السؤال
            lang: اللغة
            
        Returns:
            رسالة السؤال
        """
        messages = self.QUESTION_MESSAGES_AR if lang == 'ar' else self.QUESTION_MESSAGES_EN
        return messages.get(question_type, "")
    
    def get_yes_response(
        self,
        question_type: QuestionType,
        lang: str = 'ar'
    ) -> str:
        """
        الحصول على رد "نعم"
        
        Args:
            question_type: نوع السؤال
            lang: اللغة
            
        Returns:
            رد "نعم"
        """
        responses = self.YES_RESPONSES_AR if lang == 'ar' else self.YES_RESPONSES_EN
        return responses.get(question_type, "")
    
    def get_no_response(
        self,
        question_type: QuestionType,
        lang: str = 'ar'
    ) -> str:
        """
        الحصول على رد "لا"
        
        Args:
            question_type: نوع السؤال
            lang: اللغة
            
        Returns:
            رد "لا"
        """
        responses = self.NO_RESPONSES_AR if lang == 'ar' else self.NO_RESPONSES_EN
        return responses.get(question_type, "")
    
    def get_ambiguous_response(self, lang: str = 'ar') -> str:
        """
        رد عند عدم وجود سياق واضح
        
        Args:
            lang: اللغة
            
        Returns:
            رسالة توضيح
        """
        if lang == 'ar':
            return """❓ **هل يمكنك التوضيح؟**

لم أفهم على ماذا تجيب بـ "نعم".

💡 يمكنك:
• اكتب "المنيو" لعرض القائمة
• اكتب "السلة" لعرض طلبك
• اكتب "تأكيد" لتأكيد الطلب"""
        else:
            return """❓ **Can you clarify?**

I didn't understand what you're saying "yes" to.

💡 You can:
• Type "menu" to view the menu
• Type "cart" to view your order
• Type "confirm" to confirm the order"""


# Singleton instance
question_context_manager = QuestionContextManager()

