"""
Order Intent Classifier - مصنف نية الطلب
يميز بين السلة الحالية والطلبات السابقة
مدمج مع نظام القاموس الذكي
"""

from typing import Dict, Optional, List, Set, Tuple
from app.utils.logger import app_logger as logger
# 🚀 استخدام النظام السريع الجديد (99x أسرع!)
from app.utils.fast_expression_matcher import fast_expression_matcher as expression_detector
from app.utils.word_variants import extract_number, normalize_numbers
import json
import os


class OrderIntentClassifier:
    """مصنف نية الطلب - يميز بين السلة الحالية والطلبات السابقة"""
    
    # كلمات مفتاحية للسلة الحالية
    CART_KEYWORDS_AR = [
        'السلة', 'سلة', 'سلتي',
        'الآن', 'حالياً', 'الحالية',
        'أضف', 'احذف', 'عدل',
        'اضافة', 'حذف', 'تعديل',
        'موجود', 'عندي', 'لدي'
    ]
    
    CART_KEYWORDS_EN = [
        'cart', 'basket',
        'now', 'current', 'currently',
        'add', 'remove', 'delete', 'edit', 'modify',
        'have', 'got'
    ]
    
    # كلمات مفتاحية للطلبات السابقة
    LAST_ORDER_KEYWORDS_AR = [
        'آخر طلب', 'اخر طلب', 'طلبي السابق', 'الطلب السابق',
        'طلب سابق', 'طلبات سابقة', 'طلباتي السابقة',
        'قبل', 'سابق', 'سابقاً', 'قديم',
        'تاريخ', 'history', 'previous'
    ]

    LAST_ORDER_KEYWORDS_EN = [
        'last order', 'previous order', 'old order',
        'past order', 'order history',
        'before', 'previous', 'old',
        'history'
    ]

    # كلمات مفتاحية لإعادة الطلب
    REORDER_KEYWORDS_AR = [
        'أعد الطلب', 'اعد الطلب', 'أعد طلب', 'اعد طلب',
        'كرر', 'تكرار', 'كرر الطلب', 'تكرار الطلب',
        'نفس الطلب', 'مثل السابق', 'مثل قبل',
        'reorder', 'أريد تكرار'
    ]

    REORDER_KEYWORDS_EN = [
        'reorder', 'repeat order', 'repeat my order',
        'same order', 'same as before', 'same as last time',
        'order again', 'again'
    ]
    
    # كلمات غامضة تحتاج توضيح
    AMBIGUOUS_KEYWORDS_AR = [
        'الطلب', 'طلب', 'طلبي',
        'وريني', 'اعرض', 'شوف',
        'ايش', 'وش', 'شنو',
        'عندي', 'لدي'
    ]
    
    AMBIGUOUS_KEYWORDS_EN = [
        'order', 'my order',
        'show', 'view', 'see',
        'what', 'whats',
        'have', 'got'
    ]
    
    def __init__(self):
        """تهيئة المصنف"""
        self.menu_keywords: Set[str] = set()
        self.menu_synonyms: Dict[str, List[str]] = {}
        self._load_menu_data()

    def _load_menu_data(self):
        """تحميل بيانات المنيو من الكاش والكلمات المفتاحية"""
        try:
            # 1. Load from menu cache
            cache_path = os.path.join(
                os.path.dirname(os.path.dirname(os.path.dirname(__file__))),
                'data', 'cache', 'menu_cache.json'
            )

            if os.path.exists(cache_path):
                with open(cache_path, 'r', encoding='utf-8') as f:
                    menu_data = json.load(f)

                # Extract keywords from menu items
                for item in menu_data.get('items', []):
                    # Arabic name
                    name_ar = item.get('name_ar', '')
                    if name_ar:
                        self.menu_keywords.add(name_ar.lower())
                        # Add individual words
                        words = name_ar.split()
                        for word in words:
                            if len(word) > 2 and word not in ['مع', 'و', 'ب', 'من', 'في', 'على', 'رز', 'أبيض']:
                                self.menu_keywords.add(word.lower())

                    # English name
                    name_en = item.get('name_en', '')
                    if name_en:
                        self.menu_keywords.add(name_en.lower())
                        # Add individual words
                        words = name_en.lower().split()
                        for word in words:
                            if len(word) > 2 and word not in ['with', 'and', 'or', 'the', 'rice', 'white']:
                                self.menu_keywords.add(word)

                    # Tags
                    tags = item.get('tags', [])
                    for tag in tags:
                        if len(tag) > 2:
                            self.menu_keywords.add(tag.lower())

                logger.info(f"✅ Loaded {len(self.menu_keywords)} menu keywords from cache")

            # 2. Load from intents_keywords.json
            keywords_path = os.path.join(
                os.path.dirname(os.path.dirname(os.path.dirname(__file__))),
                'data', 'intents_keywords.json'
            )

            if os.path.exists(keywords_path):
                with open(keywords_path, 'r', encoding='utf-8') as f:
                    keywords_data = json.load(f)

                # Extract food items from categories
                food_categories = keywords_data.get('food_categories', {})
                for category_name, category_data in food_categories.items():
                    items = category_data.get('items', {})

                    # Arabic items
                    ar_items = items.get('ar', [])
                    for item in ar_items:
                        self.menu_keywords.add(item.lower())

                    # English items
                    en_items = items.get('en', [])
                    for item in en_items:
                        self.menu_keywords.add(item.lower())

                # Extract meat types
                meat_types = food_categories.get('meat_types', {})
                for meat in meat_types.get('ar', []):
                    self.menu_keywords.add(meat.lower())
                for meat in meat_types.get('en', []):
                    self.menu_keywords.add(meat.lower())

                logger.info(f"✅ Total menu keywords: {len(self.menu_keywords)}")
            else:
                logger.warning(f"⚠️ Keywords file not found at {keywords_path}")

        except Exception as e:
            logger.error(f"❌ Error loading menu data: {e}")
            self.menu_keywords = set()

    def detect_product_inquiry(self, text: str) -> Dict[str, any]:
        """
        اكتشاف وجود صنف في النص

        Args:
            text: النص المراد فحصه

        Returns:
            Dict مع:
                - has_product: bool - هل يحتوي على صنف
                - products_found: List[str] - الأصناف المكتشفة
                - confidence: str - مستوى الثقة (high/medium/low)
        """
        text_lower = text.lower().strip()
        products_found = []

        # البحث عن الأصناف في النص
        for keyword in self.menu_keywords:
            if keyword in text_lower:
                products_found.append(keyword)

        # تحديد مستوى الثقة
        if len(products_found) == 0:
            confidence = 'none'
        elif len(products_found) == 1:
            confidence = 'high'
        elif len(products_found) <= 3:
            confidence = 'medium'
        else:
            confidence = 'low'  # قد يكون نص طويل يحتوي على كلمات عامة

        has_product = len(products_found) > 0

        if has_product:
            logger.info(f"🍽️ Product inquiry detected: {products_found[:3]}")

        return {
            'has_product': has_product,
            'products_found': products_found,
            'confidence': confidence,
            'count': len(products_found)
        }

    def classify_order_intent(self, message: str) -> Dict[str, any]:
        """
        تصنيف نية الطلب
        
        Args:
            message: رسالة المستخدم
            
        Returns:
            Dict مع intent, confidence, keywords_found
        """
        
        message_lower = message.lower().strip()

        # 1. التحقق من إعادة الطلب (أعلى أولوية)
        reorder_keywords_found = []
        for keyword in self.REORDER_KEYWORDS_AR + self.REORDER_KEYWORDS_EN:
            if keyword in message_lower:
                reorder_keywords_found.append(keyword)

        # 2. التحقق من السلة الحالية
        cart_keywords_found = []
        for keyword in self.CART_KEYWORDS_AR + self.CART_KEYWORDS_EN:
            if keyword in message_lower:
                cart_keywords_found.append(keyword)

        # 3. التحقق من الطلبات السابقة
        last_order_keywords_found = []
        for keyword in self.LAST_ORDER_KEYWORDS_AR + self.LAST_ORDER_KEYWORDS_EN:
            if keyword in message_lower:
                last_order_keywords_found.append(keyword)

        # 4. التحقق من الكلمات الغامضة
        ambiguous_keywords_found = []
        for keyword in self.AMBIGUOUS_KEYWORDS_AR + self.AMBIGUOUS_KEYWORDS_EN:
            if keyword in message_lower:
                ambiguous_keywords_found.append(keyword)
        
        # 5. اتخاذ القرار
        reorder_score = len(reorder_keywords_found)
        cart_score = len(cart_keywords_found)
        last_order_score = len(last_order_keywords_found)
        ambiguous_score = len(ambiguous_keywords_found)

        # حالة 0: واضح أنه إعادة طلب (أعلى أولوية)
        if reorder_score > 0:
            logger.info(f"🔄 Reorder intent detected: {reorder_keywords_found}")
            return {
                'intent': 'reorder_intent',
                'confidence': 'high',
                'keywords_found': reorder_keywords_found,
                'reason': 'Reorder keywords found'
            }

        # حالة 1: واضح أنه السلة الحالية
        if cart_score > 0 and last_order_score == 0:
            logger.info(f"🛒 Cart intent detected: {cart_keywords_found}")
            return {
                'intent': 'cart_intent',
                'confidence': 'high',
                'keywords_found': cart_keywords_found,
                'reason': 'Cart keywords found, no last order keywords'
            }
        
        # حالة 2: واضح أنه الطلب السابق
        if last_order_score > 0 and cart_score == 0:
            logger.info(f"📜 Last order intent detected: {last_order_keywords_found}")
            return {
                'intent': 'last_order_intent',
                'confidence': 'high',
                'keywords_found': last_order_keywords_found,
                'reason': 'Last order keywords found, no cart keywords'
            }
        
        # حالة 3: كلاهما موجود (تعارض)
        if cart_score > 0 and last_order_score > 0:
            # الأولوية للأكثر وضوحاً
            if cart_score > last_order_score:
                logger.warning(f"⚠️ Conflicting keywords, cart wins: cart={cart_score}, last_order={last_order_score}")
                return {
                    'intent': 'cart_intent',
                    'confidence': 'medium',
                    'keywords_found': cart_keywords_found,
                    'reason': f'Both found, cart score higher ({cart_score} vs {last_order_score})'
                }
            elif last_order_score > cart_score:
                logger.warning(f"⚠️ Conflicting keywords, last_order wins: cart={cart_score}, last_order={last_order_score}")
                return {
                    'intent': 'last_order_intent',
                    'confidence': 'medium',
                    'keywords_found': last_order_keywords_found,
                    'reason': f'Both found, last_order score higher ({last_order_score} vs {cart_score})'
                }
            else:
                # نفس النقاط - غامض
                logger.warning(f"⚠️ Ambiguous: equal scores ({cart_score} vs {last_order_score})")
                return {
                    'intent': 'ambiguous',
                    'confidence': 'low',
                    'keywords_found': cart_keywords_found + last_order_keywords_found,
                    'reason': f'Equal scores ({cart_score} vs {last_order_score})'
                }
        
        # حالة 4: فقط كلمات غامضة
        if ambiguous_score > 0 and cart_score == 0 and last_order_score == 0:
            logger.warning(f"⚠️ Ambiguous keywords only: {ambiguous_keywords_found}")
            return {
                'intent': 'ambiguous',
                'confidence': 'low',
                'keywords_found': ambiguous_keywords_found,
                'reason': 'Only ambiguous keywords found'
            }
        
        # حالة 5: لا توجد كلمات مفتاحية
        logger.info(f"❓ No order-related keywords found")
        return {
            'intent': 'unknown',
            'confidence': 'none',
            'keywords_found': [],
            'reason': 'No order-related keywords found'
        }

    def detect_with_smart_dictionary(self, message: str) -> Dict:
        """
        كشف النية باستخدام القاموس الذكي المتكامل

        Args:
            message: رسالة المستخدم

        Returns:
            قاموس يحتوي على النية والثقة والتفاصيل
        """
        # الحصول على معلومات التعبير من القاموس الذكي
        expr_info = expression_detector.get_expression_info(message)

        logger.info(f"🎯 Smart dictionary analysis: {expr_info}")

        # إذا كان طلب عرض السلة
        if expr_info.get('is_cart'):
            return {
                'intent': 'cart_intent',
                'confidence': 'high',
                'category': expr_info.get('category'),
                'subcategory': expr_info.get('subcategory'),
                'reason': 'Smart dictionary: view_cart detected'
            }

        # إذا كان طلب جديد
        if expr_info.get('is_order'):
            return {
                'intent': 'new_order',
                'confidence': 'high',
                'category': expr_info.get('category'),
                'subcategory': expr_info.get('subcategory'),
                'quantity': expr_info.get('quantity'),
                'reason': 'Smart dictionary: new_order detected'
            }

        # إذا كان تأكيد
        if expr_info.get('is_confirmation'):
            return {
                'intent': 'confirmation',
                'confidence': 'high',
                'is_accept': expr_info.get('is_accept'),
                'category': expr_info.get('category'),
                'subcategory': expr_info.get('subcategory'),
                'reason': 'Smart dictionary: confirmation detected'
            }

        # إذا كان طلب إلغاء
        if expr_info.get('is_cancel'):
            return {
                'intent': 'cancel_order',
                'confidence': 'high',
                'category': expr_info.get('category'),
                'subcategory': expr_info.get('subcategory'),
                'reason': 'Smart dictionary: cancel detected'
            }

        # إذا كان طلب إكمال
        if expr_info.get('is_finish'):
            return {
                'intent': 'finish_order',
                'confidence': 'high',
                'category': expr_info.get('category'),
                'subcategory': expr_info.get('subcategory'),
                'reason': 'Smart dictionary: finish detected'
            }

        # إذا لم يتم التعرف على نية واضحة، استخدم الطريقة التقليدية
        return self.classify_order_intent(message)

    def extract_quantity_smart(self, message: str) -> Optional[int]:
        """
        استخراج الكمية باستخدام القاموس الذكي

        Args:
            message: رسالة المستخدم

        Returns:
            الكمية أو None
        """
        quantity = extract_number(message)
        if quantity:
            logger.info(f"🔢 Extracted quantity: {quantity}")
        return quantity

    def is_quantity_modification_smart(self, message: str) -> Tuple[bool, Optional[str], Optional[int]]:
        """
        التحقق من تعديل الكمية باستخدام القاموس الذكي

        Returns:
            Tuple[is_modification, modification_type, quantity]
            مثال: (True, "increase", 2)
        """
        is_mod, mod_type = expression_detector.is_quantity_modification(message)
        quantity = None

        if is_mod:
            quantity = self.extract_quantity_smart(message)
            logger.info(f"📊 Quantity modification: type={mod_type}, quantity={quantity}")

        return is_mod, mod_type, quantity

    def get_clarification_message(self, lang: str = 'ar') -> str:
        """
        الحصول على رسالة توضيح
        
        Args:
            lang: اللغة (ar أو en)
            
        Returns:
            رسالة توضيح
        """
        
        if lang == 'ar':
            return """🤔 **هل تقصد:**

1️⃣ **السلة الحالية** (الأصناف التي تضيفها الآن)
2️⃣ **آخر طلب سابق** (طلب قديم تم تأكيده)

من فضلك اختر رقم 1 أو 2."""
        else:
            return """🤔 **Do you mean:**

1️⃣ **Current cart** (items you're adding now)
2️⃣ **Last previous order** (old confirmed order)

Please choose 1 or 2."""
    
    def is_cart_related(self, message: str) -> bool:
        """
        التحقق إذا كانت الرسالة متعلقة بالسلة
        
        Args:
            message: رسالة المستخدم
            
        Returns:
            True إذا كانت متعلقة بالسلة
        """
        
        result = self.classify_order_intent(message)
        return result['intent'] in ['cart_intent', 'ambiguous']
    
    def is_last_order_related(self, message: str) -> bool:
        """
        التحقق إذا كانت الرسالة متعلقة بالطلب السابق

        Args:
            message: رسالة المستخدم

        Returns:
            True إذا كانت متعلقة بالطلب السابق
        """

        result = self.classify_order_intent(message)
        return result['intent'] in ['last_order_intent', 'ambiguous']

    def extract_menu_item_from_text(self, text: str) -> Optional[Dict[str, any]]:
        """
        استخراج اسم الصنف من النص حتى لو كان متقطعاً أو غير مكتمل

        يستخدم:
        1. FAISS للبحث الدلالي
        2. Regex للتطابق الجزئي
        3. تصحيح الأخطاء الإملائية

        Args:
            text: النص المدخل من المستخدم

        Returns:
            Dict مع item_name, confidence, method أو None
        """
        import re
        from app.utils.word_correction_dict import word_correction_dict

        text_lower = text.lower().strip()

        # 1. إزالة الكلمات الزائدة (الحشو)
        filler_words = [
            'أبغى', 'ابغى', 'ابي', 'أريد', 'بدي', 'عايز',
            'لو سمحت', 'من فضلك', 'ممكن', 'الله يخليك',
            'بس', 'فقط', 'please', 'want', 'need'
        ]

        cleaned_text = text_lower
        for filler in filler_words:
            cleaned_text = cleaned_text.replace(filler, ' ')

        cleaned_text = ' '.join(cleaned_text.split())  # إزالة المسافات الزائدة

        if not cleaned_text:
            return None

        # 2. تصحيح الأخطاء الإملائية
        corrected_text, corrections = word_correction_dict.correct_text(cleaned_text)

        if corrections:
            logger.info(f"🔧 Corrected text: '{cleaned_text}' → '{corrected_text}'")
            cleaned_text = corrected_text

        # 3. البحث في قائمة الأصناف المحملة
        best_match = None
        best_score = 0.0

        # 3a. التطابق التام
        for keyword in self.menu_keywords:
            if cleaned_text == keyword:
                logger.info(f"🎯 Exact match found: '{keyword}'")
                return {
                    'item_name': keyword,
                    'confidence': 1.0,
                    'method': 'exact_match',
                    'original_text': text
                }

        # 3b. التطابق الجزئي (substring)
        for keyword in self.menu_keywords:
            if keyword in cleaned_text or cleaned_text in keyword:
                # حساب نسبة التطابق
                score = len(cleaned_text) / max(len(keyword), len(cleaned_text))
                if score > best_score:
                    best_score = score
                    best_match = keyword

        if best_match and best_score >= 0.6:
            logger.info(f"🔍 Partial match found: '{best_match}' (score: {best_score:.2f})")
            return {
                'item_name': best_match,
                'confidence': best_score,
                'method': 'partial_match',
                'original_text': text
            }

        # 3c. البحث بالـ Regex (للكلمات المتقطعة)
        words = cleaned_text.split()
        for word in words:
            if len(word) < 3:  # تجاهل الكلمات القصيرة جداً
                continue

            for keyword in self.menu_keywords:
                # استخدام regex للبحث عن تطابق جزئي
                pattern = re.compile(re.escape(word), re.IGNORECASE)
                if pattern.search(keyword):
                    score = len(word) / len(keyword)
                    if score > best_score:
                        best_score = score
                        best_match = keyword

        if best_match and best_score >= 0.5:
            logger.info(f"🔎 Regex match found: '{best_match}' (score: {best_score:.2f})")
            return {
                'item_name': best_match,
                'confidence': best_score,
                'method': 'regex_match',
                'original_text': text
            }

        # 4. لم يتم العثور على تطابق
        logger.warning(f"⚠️ No menu item found in text: '{text}'")
        return None


# Singleton instance
order_intent_classifier = OrderIntentClassifier()

