"""
Order Context Guard - حارس سياق الطلب
يمنع الخلط بين السلة الحالية والطلبات السابقة
"""

from typing import Dict, Optional
from app.state_machine.order_states import OrderPhase
from app.utils.logger import app_logger as logger


class OrderContextGuard:
    """حارس سياق الطلب - يمنع الخلط بين السياقات"""
    
    # المراحل التي تمنع عرض الطلبات السابقة
    BLOCKING_PHASES = [
        OrderPhase.CHOOSE_ITEM.value,
        OrderPhase.CONFIRM_QUANTITY.value,
        OrderPhase.ADD_EXTRAS.value,
        OrderPhase.ADDRESS_INFO.value,
        OrderPhase.PAYMENT.value,
        OrderPhase.CONFIRMATION.value
    ]
    
    def __init__(self):
        """تهيئة الحارس"""
        pass
    
    def can_view_last_order(
        self,
        session: Dict,
        intent: str
    ) -> Dict[str, any]:
        """
        التحقق إذا كان يمكن عرض الطلب السابق
        
        Args:
            session: بيانات الجلسة
            intent: النية المكتشفة
            
        Returns:
            Dict مع allowed, reason, message
        """
        
        # 1. الحصول على المرحلة الحالية
        current_phase = session.get('order_phase', OrderPhase.START.value)
        stage = session.get('stage', 'items')
        
        # 2. التحقق من السلة
        cart_items = session.get('items', [])
        has_items = len(cart_items) > 0
        
        # 3. التحقق من المرحلة
        is_in_blocking_phase = current_phase in self.BLOCKING_PHASES
        
        logger.info(f"🛡️ Guard check: phase={current_phase}, stage={stage}, has_items={has_items}, intent={intent}")
        
        # حالة 1: في مرحلة الدفع - ممنوع تماماً
        if current_phase == OrderPhase.PAYMENT.value:
            logger.warning(f"🚫 Blocked: User in PAYMENT phase")
            return {
                'allowed': False,
                'reason': 'in_payment_phase',
                'message_ar': """⚠️ **لا يمكن عرض الطلبات السابقة الآن**

أنت في مرحلة الدفع. من فضلك أكمل الطلب الحالي أولاً.

💡 يمكنك:
• إكمال الدفع
• أو إلغاء الطلب الحالي""",
                'message_en': """⚠️ **Cannot view previous orders now**

You're in the payment phase. Please complete the current order first.

💡 You can:
• Complete payment
• Or cancel the current order"""
            }
        
        # حالة 2: في مرحلة التأكيد - ممنوع
        if current_phase == OrderPhase.CONFIRMATION.value:
            logger.warning(f"🚫 Blocked: User in CONFIRMATION phase")
            return {
                'allowed': False,
                'reason': 'in_confirmation_phase',
                'message_ar': """⚠️ **لا يمكن عرض الطلبات السابقة الآن**

أنت في مرحلة تأكيد الطلب. من فضلك أكمل أو ألغِ الطلب الحالي أولاً.""",
                'message_en': """⚠️ **Cannot view previous orders now**

You're in the confirmation phase. Please complete or cancel the current order first."""
            }
        
        # حالة 3: في مرحلة العنوان - ممنوع
        if current_phase == OrderPhase.ADDRESS_INFO.value:
            logger.warning(f"🚫 Blocked: User in ADDRESS_INFO phase")
            return {
                'allowed': False,
                'reason': 'in_address_phase',
                'message_ar': """⚠️ **لا يمكن عرض الطلبات السابقة الآن**

أنت في مرحلة إدخال العنوان. من فضلك أكمل الطلب الحالي أولاً.

💡 يمكنك:
• إرسال موقعك
• أو إلغاء الطلب الحالي""",
                'message_en': """⚠️ **Cannot view previous orders now**

You're in the address phase. Please complete the current order first.

💡 You can:
• Send your location
• Or cancel the current order"""
            }
        
        # حالة 4: لديه أصناف في السلة - تحذير
        if has_items and is_in_blocking_phase:
            logger.warning(f"⚠️ Warning: User has items in cart and in blocking phase")
            return {
                'allowed': False,
                'reason': 'has_items_in_cart',
                'message_ar': """⚠️ **لديك أصناف في السلة الحالية**

لعرض الطلبات السابقة، من فضلك:
1️⃣ أكمل الطلب الحالي
2️⃣ أو احذف الأصناف من السلة

💡 لعرض السلة الحالية، اكتب: "السلة" أو "cart" """,
                'message_en': """⚠️ **You have items in your current cart**

To view previous orders, please:
1️⃣ Complete the current order
2️⃣ Or remove items from cart

💡 To view current cart, type: "cart" or "السلة" """
            }
        
        # حالة 5: مسموح
        logger.info(f"✅ Allowed: User can view last order")
        return {
            'allowed': True,
            'reason': 'allowed',
            'message_ar': None,
            'message_en': None
        }
    
    def can_view_cart(
        self,
        session: Dict,
        intent: str
    ) -> Dict[str, any]:
        """
        التحقق إذا كان يمكن عرض السلة الحالية
        
        Args:
            session: بيانات الجلسة
            intent: النية المكتشفة
            
        Returns:
            Dict مع allowed, reason, message
        """
        
        # السلة الحالية دائماً مسموحة
        return {
            'allowed': True,
            'reason': 'always_allowed',
            'message_ar': None,
            'message_en': None
        }
    
    def should_clarify(
        self,
        session: Dict,
        intent: str,
        confidence: str
    ) -> bool:
        """
        التحقق إذا كان يجب طلب توضيح
        
        Args:
            session: بيانات الجلسة
            intent: النية المكتشفة
            confidence: مستوى الثقة
            
        Returns:
            True إذا كان يجب طلب توضيح
        """
        
        # 1. إذا كانت النية غامضة
        if intent == 'ambiguous':
            return True
        
        # 2. إذا كانت الثقة منخفضة
        if confidence == 'low':
            return True
        
        # 3. إذا كانت الثقة متوسطة ولديه أصناف في السلة
        cart_items = session.get('items', [])
        has_items = len(cart_items) > 0
        
        if confidence == 'medium' and has_items:
            return True
        
        return False
    
    def get_blocking_message(
        self,
        reason: str,
        lang: str = 'ar'
    ) -> str:
        """
        الحصول على رسالة الحظر
        
        Args:
            reason: سبب الحظر
            lang: اللغة
            
        Returns:
            رسالة الحظر
        """
        
        messages = {
            'in_payment_phase': {
                'ar': """⚠️ **لا يمكن عرض الطلبات السابقة الآن**

أنت في مرحلة الدفع. من فضلك أكمل الطلب الحالي أولاً.""",
                'en': """⚠️ **Cannot view previous orders now**

You're in the payment phase. Please complete the current order first."""
            },
            'in_confirmation_phase': {
                'ar': """⚠️ **لا يمكن عرض الطلبات السابقة الآن**

أنت في مرحلة تأكيد الطلب. من فضلك أكمل أو ألغِ الطلب الحالي أولاً.""",
                'en': """⚠️ **Cannot view previous orders now**

You're in the confirmation phase. Please complete or cancel the current order first."""
            },
            'in_address_phase': {
                'ar': """⚠️ **لا يمكن عرض الطلبات السابقة الآن**

أنت في مرحلة إدخال العنوان. من فضلك أكمل الطلب الحالي أولاً.""",
                'en': """⚠️ **Cannot view previous orders now**

You're in the address phase. Please complete the current order first."""
            },
            'has_items_in_cart': {
                'ar': """⚠️ **لديك أصناف في السلة الحالية**

لعرض الطلبات السابقة، من فضلك أكمل أو احذف الطلب الحالي أولاً.""",
                'en': """⚠️ **You have items in your current cart**

To view previous orders, please complete or remove the current order first."""
            }
        }
        
        if reason in messages:
            return messages[reason].get(lang, messages[reason]['ar'])
        
        # رسالة افتراضية
        if lang == 'ar':
            return "⚠️ لا يمكن تنفيذ هذا الإجراء الآن."
        else:
            return "⚠️ Cannot perform this action now."


# Singleton instance
order_context_guard = OrderContextGuard()

