"""
تنسيق الرسائل العربية لـ WhatsApp
يوفر تنسيقاً موحداً واحترافياً لجميع الرسائل
"""

from typing import List, Dict, Optional
from decimal import Decimal

# علامة RTL لضمان محاذاة النص من اليمين لليسار
RTL = "\u200F"


def format_menu_items(items: List[Dict]) -> str:
    """
    تنسيق قائمة الأصناف بشكل احترافي

    Args:
        items: قائمة الأصناف مع التفاصيل

    Returns:
        رسالة منسقة بشكل جميل
    """
    if not items:
        return RTL + "❌ عذراً، لم أجد أي أصناف متاحة."

    lines = [RTL + "📋 الأصناف المتاحة:\n"]

    for i, item in enumerate(items, 1):
        name = item.get('name_ar', 'صنف')
        # دعم كلا الحقلين: price و price_omr
        price = float(item.get('price_omr', item.get('price', 0)))
        emoji = _get_category_emoji(item.get('category', ''))

        lines.append(
            RTL + f"{i}️⃣ {name} {emoji} — 💰 {price:.3f} ر.ع"
        )

    lines.append("\n" + RTL + "💬 أرسل الرقم (مثال: 1)")

    return "\n".join(lines)


def format_single_item(item: Dict) -> str:
    """
    تنسيق صنف واحد مع طلب الكمية

    Args:
        item: تفاصيل الصنف

    Returns:
        رسالة منسقة
    """
    name = item.get('name_ar', 'صنف')
    # دعم كلا الحقلين: price و price_omr
    price = float(item.get('price_omr', item.get('price', 0)))
    emoji = _get_category_emoji(item.get('category', ''))

    lines = [
        RTL + "✨ وجدت صنف واحد:\n",
        RTL + f"{emoji} {name}",
        RTL + f"💰 السعر: {price:.3f} ر.ع\n",
        RTL + "💬 كم الكمية المطلوبة؟"
    ]

    return "\n".join(lines)


def format_cart_summary(cart_items: List[Dict], delivery_fee: float = 0.500) -> str:
    """
    تنسيق ملخص السلة بشكل احترافي
    
    Args:
        cart_items: قائمة الأصناف في السلة
        delivery_fee: رسوم التوصيل
        
    Returns:
        ملخص السلة منسق
    """
    if not cart_items:
        return RTL + "🛒 سلتك فارغة حالياً.\n\n" + RTL + "💬 أرسل اسم الصنف الذي تريده!"
    
    lines = [RTL + "🛒 سلتك الحالية:\n"]
    
    subtotal = 0.0
    
    for i, item in enumerate(cart_items, 1):
        name = item.get('name_ar', 'صنف')
        quantity = item.get('quantity', 1)
        # دعم كلا الحقلين: price و price_omr
        price = float(item.get('price_omr', item.get('price', 0)))
        total = quantity * price
        subtotal += total
        emoji = _get_category_emoji(item.get('category', ''))

        lines.append(RTL + f"{i}️⃣ {name} {emoji}")
        lines.append(RTL + f"   الكمية: {quantity} × 💰 {price:.3f} = 💵 {total:.3f} ر.ع")
    
    total = subtotal + delivery_fee
    
    lines.extend([
        "\n" + RTL + "----------------------------",
        RTL + f"🧾 إجمالي الأصناف: {subtotal:.3f} ر.ع",
        RTL + f"🚚 رسوم التوصيل: {delivery_fee:.3f} ر.ع",
        RTL + f"💰 الإجمالي النهائي: {total:.3f} ر.ع\n",
        RTL + "📍 للمتابعة أرسل \"العنوان\"، أو \"تأكيد\" لإرسال الطلب."
    ])
    
    return "\n".join(lines)


def format_order_confirmation(order_number: int, cart_items: List[Dict], total: float) -> str:
    """
    تنسيق رسالة تأكيد الطلب
    
    Args:
        order_number: رقم الطلب
        cart_items: الأصناف المطلوبة
        total: الإجمالي النهائي
        
    Returns:
        رسالة تأكيد منسقة
    """
    lines = [
        RTL + "✅ تم تأكيد طلبك بنجاح!\n",
        RTL + f"🔢 رقم الطلب: {order_number}",
        RTL + "📦 الأصناف:"
    ]
    
    for item in cart_items:
        name = item.get('name_ar', 'صنف')
        quantity = item.get('quantity', 1)
        lines.append(RTL + f"• {name} × {quantity}")
    
    lines.extend([
        "\n" + RTL + f"💰 الإجمالي النهائي: {total:.3f} ر.ع",
        RTL + "🕓 الحالة: جاري التحضير 🍳\n",
        RTL + "🙏 شكرًا لاختيارك مطعمنا!"
    ])
    
    return "\n".join(lines)


def format_delivery_status(status: str) -> str:
    """
    تنسيق حالة التوصيل
    
    Args:
        status: حالة الطلب
        
    Returns:
        رسالة الحالة منسقة
    """
    mapping = {
        "preparing": "🕓 طلبك قيد التجهيز 🍳",
        "ready": "📦 طلبك جاهز للاستلام ✅",
        "out_for_delivery": "🚗 طلبك في الطريق إليك",
        "delivered": "✅ تم تسليم الطلب بنجاح 🎉",
        "cancelled": "❌ تم إلغاء الطلب بناءً على طلبك",
    }
    
    return RTL + mapping.get(status, "📦 الطلب قيد المعالجة.")


def format_clarification_prompt() -> str:
    """
    رسالة طلب توضيح عند الإجابات الغامضة

    Returns:
        رسالة طلب توضيح
    """
    lines = [
        RTL + "هل تريد:",
        RTL + "1️⃣ إكمال الطلب",
        RTL + "2️⃣ إضافة صنف آخر"
    ]

    return "\n".join(lines)


def format_item_added(item_name: str, quantity: int) -> str:
    """
    تنسيق رسالة إضافة صنف للسلة

    Args:
        item_name: اسم الصنف
        quantity: الكمية

    Returns:
        رسالة تأكيد الإضافة
    """
    # رسالة بسيطة - السلة الكاملة ستظهر بعدها تلقائياً
    return RTL + f"✅ تمت إضافة {item_name} × {quantity} إلى سلتك!"


def format_no_results(query: str) -> str:
    """
    رسالة عدم وجود نتائج

    Args:
        query: الكلمة المفتاحية

    Returns:
        رسالة عدم وجود نتائج
    """
    lines = [
        RTL + f"❌ لم أجد \"{query}\"\n",
        RTL + "💡 جرب كلمة أخرى أو اكتب \"المنيو\""
    ]

    return "\n".join(lines)


def format_greeting(customer_name: Optional[str] = None) -> str:
    """
    رسالة الترحيب

    Args:
        customer_name: اسم العميل (اختياري)

    Returns:
        رسالة ترحيب
    """
    if customer_name:
        greeting = RTL + f"👋 مرحباً {customer_name}!\n"
    else:
        greeting = RTL + "👋 مرحباً بك!\n"

    lines = [
        greeting,
        RTL + "🍽️ أهلاً بك في مطعمنا\n",
        RTL + "📋 اكتب اسم الصنف أو \"المنيو\""
    ]

    return "\n".join(lines)


def format_address_request() -> str:
    """
    طلب العنوان من العميل

    Returns:
        رسالة طلب العنوان
    """
    lines = [
        RTL + "📍 أرسل عنوان التوصيل:\n",
        RTL + "مثال: الخوير، شارع السلطان قابوس، مبنى 123\n",
        RTL + "💬 أو أرسل الموقع 📌"
    ]

    return "\n".join(lines)


def format_payment_method_request() -> str:
    """
    طلب طريقة الدفع
    
    Returns:
        رسالة طلب طريقة الدفع
    """
    lines = [
        RTL + "💳 اختر طريقة الدفع:\n",
        RTL + "1️⃣ نقداً عند الاستلام 💵",
        RTL + "2️⃣ بطاقة ائتمان 💳",
        RTL + "3️⃣ تحويل بنكي 🏦\n",
        RTL + "💬 أرسل رقم الطريقة المفضلة."
    ]
    
    return "\n".join(lines)


def format_invalid_selection() -> str:
    """
    رسالة اختيار غير صحيح
    
    Returns:
        رسالة خطأ في الاختيار
    """
    lines = [
        RTL + "❌ عذراً، الاختيار غير صحيح.\n",
        RTL + "💬 من فضلك أرسل رقم الصنف من القائمة أعلاه."
    ]
    
    return "\n".join(lines)


def format_quantity_request(item_name: str) -> str:
    """
    طلب الكمية بعد اختيار الصنف

    Args:
        item_name: اسم الصنف المختار

    Returns:
        رسالة طلب الكمية
    """
    lines = [
        RTL + f"✅ اخترت: {item_name}\n",
        RTL + "💬 كم الكمية المطلوبة؟"
    ]

    return "\n".join(lines)


def _get_category_emoji(category: str) -> str:
    """
    الحصول على الإيموجي المناسب للفئة
    
    Args:
        category: اسم الفئة
        
    Returns:
        الإيموجي المناسب
    """
    category_lower = category.lower() if category else ""
    
    emoji_map = {
        'soup': '🥣',
        'شوربة': '🥣',
        'appetizer': '🥗',
        'مقبلات': '🥗',
        'main': '🍽️',
        'رئيسي': '🍽️',
        'meat': '🥩',
        'لحم': '🥩',
        'chicken': '🍗',
        'دجاج': '🍗',
        'fish': '🐟',
        'سمك': '🐟',
        'seafood': '🦐',
        'مأكولات بحرية': '🦐',
        'rice': '🍚',
        'رز': '🍚',
        'dessert': '🍰',
        'حلويات': '🍰',
        'drink': '🥤',
        'مشروبات': '🥤',
        'beverage': '🥤',
    }
    
    for key, emoji in emoji_map.items():
        if key in category_lower:
            return emoji
    
    return '🏷️'

