"""
🍽️ محلل أصناف المنيو الذكي
Smart Menu Item Detector

يكشف أصناف المنيو من النص بجميع اللهجات والمرادفات
"""

import yaml
import os
from typing import Optional, List, Dict, Tuple, Set
from pathlib import Path
from difflib import SequenceMatcher

from app.utils.text_helpers import normalize_order_text
from app.utils.word_variants import word_variants_generator
from app.utils.logger import app_logger as logger


class MenuItemDetector:
    """كاشف أصناف المنيو الذكي"""
    
    def __init__(self):
        self.menu_dict: Dict = {}
        self.item_index: Dict[str, str] = {}  # keyword -> item_id
        self.synonyms: Dict = {}
        self._load_dictionary()
        self._build_index()
    
    def _load_dictionary(self):
        """تحميل قاموس أصناف المنيو من ملف YAML"""
        try:
            dict_path = Path(__file__).parent / "menu_items_dictionary.yaml"
            
            if not dict_path.exists():
                logger.error(f"❌ Menu items dictionary not found: {dict_path}")
                return
            
            with open(dict_path, 'r', encoding='utf-8') as f:
                data = yaml.safe_load(f)
                self.menu_dict = data.get('menu_items', {})
                self.synonyms = data.get('general_synonyms', {})
            
            logger.info(f"✅ Loaded menu items dictionary with {len(self.menu_dict)} categories")
        
        except Exception as e:
            logger.error(f"❌ Error loading menu items dictionary: {e}")
    
    def _build_index(self):
        """بناء فهرس سريع للبحث"""
        for category, items in self.menu_dict.items():
            for item_id, item_data in items.items():
                # الأسماء العربية
                for name in item_data.get('arabic_names', []):
                    self.item_index[name.lower()] = item_id
                
                # اللهجات
                dialect_variants = item_data.get('dialect_variants', {})
                for dialect, names in dialect_variants.items():
                    for name in names:
                        self.item_index[name.lower()] = item_id
                
                # الأسماء الإنجليزية
                for name in item_data.get('english_names', []):
                    self.item_index[name.lower()] = item_id
                
                # الكلمات المفتاحية
                for keyword in item_data.get('keywords', []):
                    self.item_index[keyword.lower()] = item_id
        
        logger.info(f"✅ Built menu index with {len(self.item_index)} entries")
    
    def detect_menu_item(self, text: str) -> List[Dict]:
        """
        كشف أصناف المنيو من النص
        
        Args:
            text: النص المراد البحث فيه
            
        Returns:
            قائمة بالأصناف المكتشفة مع درجة الثقة
        """
        if not text:
            return []
        
        # تطبيع النص
        normalized_text = normalize_order_text(text.lower())
        
        # تصحيح الأخطاء الإملائية
        words = normalized_text.split()
        corrected_words = [word_variants_generator.correct_typo(word) for word in words]
        corrected_text = " ".join(corrected_words)
        
        results = []
        
        # 1. البحث المباشر (Exact Match)
        exact_matches = self._exact_match_search(corrected_text)
        results.extend(exact_matches)
        
        # 2. البحث الجزئي (Partial Match)
        if not results:
            partial_matches = self._partial_match_search(corrected_text)
            results.extend(partial_matches)
        
        # 3. البحث بالكلمات المفتاحية (Keyword Match)
        if not results:
            keyword_matches = self._keyword_match_search(corrected_text)
            results.extend(keyword_matches)
        
        # 4. البحث الضبابي (Fuzzy Match)
        if not results:
            fuzzy_matches = self._fuzzy_match_search(corrected_text)
            results.extend(fuzzy_matches)
        
        # إزالة التكرارات وترتيب حسب الأولوية ثم الثقة
        unique_results = self._deduplicate_results(results)
        sorted_results = sorted(
            unique_results,
            key=lambda x: (x.get('priority', 999), -x['confidence'])
        )

        return sorted_results[:5]  # أفضل 5 نتائج
    
    def _exact_match_search(self, text: str) -> List[Dict]:
        """البحث المباشر"""
        results = []
        
        for keyword, item_id in self.item_index.items():
            if keyword == text:
                results.append({
                    'item_id': item_id,
                    'matched_keyword': keyword,
                    'confidence': 1.0,
                    'match_type': 'exact'
                })
        
        return results
    
    def _partial_match_search(self, text: str) -> List[Dict]:
        """البحث الجزئي"""
        results = []
        text_words = set(text.split())
        
        for keyword, item_id in self.item_index.items():
            keyword_words = set(keyword.split())
            
            # إذا كانت جميع كلمات الكلمة المفتاحية موجودة في النص
            if keyword_words.issubset(text_words):
                confidence = len(keyword_words) / len(text_words)
                results.append({
                    'item_id': item_id,
                    'matched_keyword': keyword,
                    'confidence': min(confidence, 0.95),
                    'match_type': 'partial'
                })
            
            # أو إذا كانت الكلمة المفتاحية موجودة في النص
            elif keyword in text:
                confidence = len(keyword) / len(text)
                results.append({
                    'item_id': item_id,
                    'matched_keyword': keyword,
                    'confidence': min(confidence, 0.90),
                    'match_type': 'partial'
                })
        
        return results
    
    def _keyword_match_search(self, text: str) -> List[Dict]:
        """البحث بالكلمات المفتاحية"""
        results = []
        text_words = set(text.split())
        
        for keyword, item_id in self.item_index.items():
            keyword_words = set(keyword.split())
            common_words = keyword_words & text_words
            
            if common_words:
                confidence = len(common_words) / len(keyword_words)
                if confidence > 0.5:  # على الأقل 50% من الكلمات مشتركة
                    results.append({
                        'item_id': item_id,
                        'matched_keyword': keyword,
                        'confidence': confidence * 0.85,
                        'match_type': 'keyword'
                    })
        
        return results
    
    def _fuzzy_match_search(self, text: str) -> List[Dict]:
        """البحث الضبابي"""
        results = []
        
        for keyword, item_id in self.item_index.items():
            # حساب التشابه
            similarity = SequenceMatcher(None, text, keyword).ratio()
            
            if similarity > 0.7:  # على الأقل 70% تشابه
                results.append({
                    'item_id': item_id,
                    'matched_keyword': keyword,
                    'confidence': similarity * 0.80,
                    'match_type': 'fuzzy'
                })
        
        return results
    
    def _deduplicate_results(self, results: List[Dict]) -> List[Dict]:
        """إزالة التكرارات"""
        seen = {}

        for result in results:
            item_id = result['item_id']

            # الحصول على أولوية الصنف
            item_info = self.get_item_info(item_id)
            priority = item_info.get('priority', 999) if item_info else 999
            result['priority'] = priority

            # الاحتفاظ بالنتيجة ذات الثقة الأعلى أو الأولوية الأعلى
            if item_id not in seen:
                seen[item_id] = result
            else:
                # إذا كانت الثقة متساوية، استخدم الأولوية
                if result['confidence'] == seen[item_id]['confidence']:
                    if priority < seen[item_id]['priority']:
                        seen[item_id] = result
                # وإلا استخدم الثقة الأعلى
                elif result['confidence'] > seen[item_id]['confidence']:
                    seen[item_id] = result

        return list(seen.values())
    
    def get_item_info(self, item_id: str) -> Optional[Dict]:
        """
        الحصول على معلومات صنف معين
        
        Args:
            item_id: معرف الصنف
            
        Returns:
            معلومات الصنف أو None
        """
        for category, items in self.menu_dict.items():
            if item_id in items:
                return {
                    'item_id': item_id,
                    'category': category,
                    **items[item_id]
                }
        
        return None
    
    def get_all_names(self, item_id: str) -> List[str]:
        """الحصول على جميع أسماء الصنف"""
        item_info = self.get_item_info(item_id)
        
        if not item_info:
            return []
        
        names = []
        names.extend(item_info.get('arabic_names', []))
        names.extend(item_info.get('english_names', []))
        
        # إضافة اللهجات
        dialect_variants = item_info.get('dialect_variants', {})
        for dialect_names in dialect_variants.values():
            names.extend(dialect_names)
        
        return list(set(names))  # إزالة التكرارات
    
    def search_by_category(self, category: str) -> List[str]:
        """البحث حسب الفئة"""
        if category in self.menu_dict:
            return list(self.menu_dict[category].keys())
        return []
    
    def get_similar_items(self, item_id: str, limit: int = 3) -> List[str]:
        """الحصول على أصناف مشابهة"""
        item_info = self.get_item_info(item_id)
        
        if not item_info:
            return []
        
        # البحث في نفس الفئة
        category = item_info['category']
        similar = self.search_by_category(category)
        
        # إزالة الصنف نفسه
        similar = [item for item in similar if item != item_id]
        
        return similar[:limit]


# مثيل عام للاستخدام
menu_item_detector = MenuItemDetector()


# دوال مساعدة للاستخدام السريع
def detect_menu_item(text: str) -> List[Dict]:
    """كشف أصناف المنيو من النص"""
    return menu_item_detector.detect_menu_item(text)


def get_item_info(item_id: str) -> Optional[Dict]:
    """الحصول على معلومات صنف"""
    return menu_item_detector.get_item_info(item_id)


def get_all_item_names(item_id: str) -> List[str]:
    """الحصول على جميع أسماء الصنف"""
    return menu_item_detector.get_all_names(item_id)


def search_by_category(category: str) -> List[str]:
    """البحث حسب الفئة"""
    return menu_item_detector.search_by_category(category)

