"""
Location Extractor - استخراج الإحداثيات من الرسائل
يمنع Hallucination - لا يولد إحداثيات، فقط يستخرجها
"""

import re
from typing import Optional, Dict, Tuple
from app.utils.logger import app_logger as logger


class LocationExtractor:
    """مستخرج الموقع - يستخرج الإحداثيات من الرسائل"""
    
    # أنماط Google Maps
    GOOGLE_MAPS_PATTERNS = [
        # Pattern 1: https://maps.google.com/?q=23.5880,58.3829
        r'maps\.google\.com/\?q=(-?\d+\.?\d*),(-?\d+\.?\d*)',
        
        # Pattern 2: https://www.google.com/maps/place/@23.5880,58.3829
        r'google\.com/maps/place/@(-?\d+\.?\d*),(-?\d+\.?\d*)',
        
        # Pattern 3: https://goo.gl/maps/xyz (short link - can't extract coords)
        r'goo\.gl/maps/([a-zA-Z0-9]+)',
        
        # Pattern 4: https://maps.app.goo.gl/xyz (new short link)
        r'maps\.app\.goo\.gl/([a-zA-Z0-9]+)',
        
        # Pattern 5: @23.5880,58.3829 (direct coordinates)
        r'@(-?\d+\.?\d*),(-?\d+\.?\d*)',
        
        # Pattern 6: lat,lng format
        r'(?:lat|latitude)[:\s]*(-?\d+\.?\d*)[,\s]+(?:lng|long|longitude)[:\s]*(-?\d+\.?\d*)',
        
        # Pattern 7: Simple comma-separated numbers (23.5880, 58.3829)
        r'(?:^|\s)(-?\d{1,3}\.\d{4,})[,\s]+(-?\d{1,3}\.\d{4,})(?:\s|$)',
    ]
    
    # كلمات مفتاحية للموقع
    LOCATION_KEYWORDS = {
        'ar': ['موقع', 'لوكيشن', 'مكان', 'عنوان', 'خريطة', 'احداثيات'],
        'en': ['location', 'place', 'address', 'map', 'coordinates', 'coords']
    }
    
    def __init__(self):
        """تهيئة المستخرج"""
        self.compiled_patterns = [re.compile(pattern, re.IGNORECASE) for pattern in self.GOOGLE_MAPS_PATTERNS]
    
    def contains_location_keyword(self, message: str) -> bool:
        """
        التحقق من وجود كلمات مفتاحية للموقع
        
        Args:
            message: الرسالة
            
        Returns:
            True إذا كانت الرسالة تحتوي على كلمات موقع
        """
        message_lower = message.lower()
        
        # فحص الكلمات العربية
        for keyword in self.LOCATION_KEYWORDS['ar']:
            if keyword in message_lower:
                logger.info(f"🔍 Location keyword detected (AR): '{keyword}'")
                return True
        
        # فحص الكلمات الإنجليزية
        for keyword in self.LOCATION_KEYWORDS['en']:
            if keyword in message_lower:
                logger.info(f"🔍 Location keyword detected (EN): '{keyword}'")
                return True
        
        return False
    
    def extract_coordinates(self, message: str) -> Optional[Dict[str, any]]:
        """
        استخراج الإحداثيات من الرسالة
        
        Args:
            message: الرسالة (قد تحتوي على رابط أو إحداثيات)
            
        Returns:
            Dict مع lat, lng, link أو None
        """
        
        logger.info(f"🔍 Attempting to extract location from: {message[:100]}...")
        
        # محاولة استخراج من كل نمط
        for i, pattern in enumerate(self.compiled_patterns):
            match = pattern.search(message)
            
            if match:
                logger.info(f"✅ Pattern {i+1} matched!")
                
                # Pattern 3 & 4: Short links (can't extract coords directly)
                if i in [2, 3]:
                    short_code = match.group(1)
                    logger.warning(f"⚠️ Short link detected: {short_code} - Cannot extract coordinates")
                    return {
                        'lat': None,
                        'lng': None,
                        'link': match.group(0),
                        'type': 'short_link',
                        'note': 'Short link detected - coordinates not extractable'
                    }
                
                # All other patterns: Extract lat, lng
                try:
                    lat = float(match.group(1))
                    lng = float(match.group(2))
                    
                    # Validate coordinates
                    if not self._validate_coordinates(lat, lng):
                        logger.warning(f"⚠️ Invalid coordinates: lat={lat}, lng={lng}")
                        continue
                    
                    # Extract full link if present
                    link = None
                    if 'http' in message:
                        link_match = re.search(r'https?://[^\s]+', message)
                        if link_match:
                            link = link_match.group(0)
                    
                    result = {
                        'lat': lat,
                        'lng': lng,
                        'link': link,
                        'type': f'pattern_{i+1}',
                        'note': 'Coordinates extracted successfully'
                    }
                    
                    logger.info(f"✅ Location extracted: lat={lat}, lng={lng}")
                    return result
                
                except (ValueError, IndexError) as e:
                    logger.error(f"❌ Error parsing coordinates: {e}")
                    continue
        
        logger.warning("⚠️ No location found in message")
        return None
    
    def _validate_coordinates(self, lat: float, lng: float) -> bool:
        """
        التحقق من صحة الإحداثيات
        
        Args:
            lat: خط العرض
            lng: خط الطول
            
        Returns:
            True إذا كانت الإحداثيات صحيحة
        """
        
        # Latitude: -90 to 90
        if lat < -90 or lat > 90:
            logger.warning(f"⚠️ Invalid latitude: {lat} (must be -90 to 90)")
            return False
        
        # Longitude: -180 to 180
        if lng < -180 or lng > 180:
            logger.warning(f"⚠️ Invalid longitude: {lng} (must be -180 to 180)")
            return False
        
        # Check if coordinates are not (0, 0) - likely invalid
        if lat == 0 and lng == 0:
            logger.warning("⚠️ Coordinates are (0, 0) - likely invalid")
            return False
        
        return True
    
    def extract_from_whatsapp_location(self, location_data: Dict) -> Optional[Dict[str, any]]:
        """
        استخراج الموقع من رسالة WhatsApp Location
        
        Args:
            location_data: بيانات الموقع من WhatsApp API
            
        Returns:
            Dict مع lat, lng, link أو None
        """
        
        try:
            # WhatsApp location format:
            # {
            #   "latitude": 23.5880,
            #   "longitude": 58.3829,
            #   "name": "Optional name",
            #   "address": "Optional address"
            # }
            
            lat = location_data.get('latitude')
            lng = location_data.get('longitude')
            
            if lat is None or lng is None:
                logger.warning("⚠️ WhatsApp location missing lat/lng")
                return None
            
            # Validate
            if not self._validate_coordinates(lat, lng):
                return None
            
            # Generate Google Maps link
            link = f"https://maps.google.com/?q={lat},{lng}"
            
            result = {
                'lat': lat,
                'lng': lng,
                'link': link,
                'type': 'whatsapp_location',
                'name': location_data.get('name'),
                'address': location_data.get('address'),
                'note': 'Location from WhatsApp'
            }
            
            logger.info(f"✅ WhatsApp location extracted: lat={lat}, lng={lng}")
            return result
        
        except Exception as e:
            logger.error(f"❌ Error extracting WhatsApp location: {e}")
            return None
    
    def is_location_message(self, message: str, message_type: str = "text") -> bool:
        """
        التحقق من أن الرسالة تحتوي على موقع
        
        Args:
            message: نص الرسالة
            message_type: نوع الرسالة (text, location, etc.)
            
        Returns:
            True إذا كانت رسالة موقع
        """
        
        # 1. WhatsApp location message
        if message_type == "location":
            return True
        
        # 2. Contains location keyword
        if self.contains_location_keyword(message):
            return True
        
        # 3. Contains Google Maps link
        if 'maps.google.com' in message.lower() or 'goo.gl/maps' in message.lower():
            return True
        
        # 4. Contains coordinates pattern
        for pattern in self.compiled_patterns:
            if pattern.search(message):
                return True
        
        return False
    
    def format_location_response(
        self,
        location_data: Dict,
        lang: str = "ar"
    ) -> str:
        """
        تنسيق رد استلام الموقع
        
        Args:
            location_data: بيانات الموقع المستخرجة
            lang: اللغة
            
        Returns:
            رسالة منسقة
        """
        
        if lang == "ar":
            if location_data.get('lat') and location_data.get('lng'):
                response = "✅ **تم استلام موقعك بنجاح!**\n\n"
                response += f"📍 الإحداثيات: {location_data['lat']:.4f}, {location_data['lng']:.4f}\n\n"
                
                if location_data.get('link'):
                    response += f"🔗 [عرض على الخريطة]({location_data['link']})\n\n"
                
                response += "هل ترغب بإضافة ملاحظات للتوصيل؟\n"
                response += "(مثلاً: الطابق الثاني، بجانب المسجد، إلخ)"
            else:
                response = "⚠️ **تم استلام رابط الموقع**\n\n"
                response += "لكن لم نتمكن من استخراج الإحداثيات.\n"
                response += "من فضلك أرسل موقعك مباشرة من WhatsApp أو اكتب العنوان بالتفصيل."
        
        else:  # English
            if location_data.get('lat') and location_data.get('lng'):
                response = "✅ **Location received successfully!**\n\n"
                response += f"📍 Coordinates: {location_data['lat']:.4f}, {location_data['lng']:.4f}\n\n"
                
                if location_data.get('link'):
                    response += f"🔗 [View on map]({location_data['link']})\n\n"
                
                response += "Would you like to add delivery notes?\n"
                response += "(e.g., 2nd floor, next to mosque, etc.)"
            else:
                response = "⚠️ **Location link received**\n\n"
                response += "But we couldn't extract coordinates.\n"
                response += "Please send your location directly from WhatsApp or write the address in detail."
        
        return response


# Singleton instance
location_extractor = LocationExtractor()

