"""
🚀 Fast Expression Matcher - محرك البحث السريع
استخدام Aho-Corasick Automaton للبحث الفوري في آلاف العبارات

الفوائد:
- سرعة O(m) حيث m = طول النص فقط (بدلاً من O(n*m))
- دعم آلاف العبارات بدون تباطؤ
- استهلاك ذاكرة أقل
- استجابة فورية (< 1ms)
"""

import ahocorasick
import yaml
from typing import Optional, Tuple, Dict, List, Set
from pathlib import Path
from app.utils.word_variants import word_variants_generator
from app.utils.logger import app_logger as logger


class FastExpressionMatcher:
    """محرك البحث السريع باستخدام Aho-Corasick"""
    
    def __init__(self):
        self.automaton = None
        self.expressions_dict: Dict = {}
        self.numbers_dict: Dict = {}
        self.phrase_to_category: Dict[str, Tuple[str, Optional[str]]] = {}
        self._load_and_build()
    
    def _load_and_build(self):
        """تحميل القاموس وبناء الـ Automaton"""
        try:
            # 1. تحميل القاموس من YAML
            dict_path = Path(__file__).parent / "order_expression_dict.yaml"
            
            if not dict_path.exists():
                logger.error(f"❌ Expression dictionary not found: {dict_path}")
                return
            
            with open(dict_path, 'r', encoding='utf-8') as f:
                data = yaml.safe_load(f)
                self.expressions_dict = data.get('order_expressions', {})
                self.numbers_dict = data.get('numbers', {})
            
            logger.info(f"✅ Loaded expression dictionary with {len(self.expressions_dict)} categories")
            
            # 2. بناء الـ Automaton
            self._build_automaton()
            
        except Exception as e:
            logger.error(f"❌ Error loading expression dictionary: {e}")
    
    def _build_automaton(self):
        """بناء Aho-Corasick Automaton من القاموس"""
        try:
            self.automaton = ahocorasick.Automaton()
            phrase_count = 0
            
            # استخراج جميع العبارات من القاموس
            for category, subcategories in self.expressions_dict.items():
                
                # إذا كانت الفئة تحتوي على فئات فرعية
                if isinstance(subcategories, dict):
                    for subcategory, phrases in subcategories.items():
                        
                        # إذا كانت الفئة الفرعية تحتوي على لهجات
                        if isinstance(phrases, dict):
                            for dialect, dialect_phrases in phrases.items():
                                phrase_count += self._add_phrases_to_automaton(
                                    dialect_phrases, 
                                    category, 
                                    subcategory
                                )
                        
                        # إذا كانت الفئة الفرعية قائمة مباشرة
                        elif isinstance(phrases, list):
                            phrase_count += self._add_phrases_to_automaton(
                                phrases, 
                                category, 
                                subcategory
                            )
                
                # إذا كانت الفئة قائمة مباشرة (بدون فئات فرعية)
                elif isinstance(subcategories, list):
                    phrase_count += self._add_phrases_to_automaton(
                        subcategories, 
                        category, 
                        None
                    )
            
            # إنهاء بناء الـ Automaton
            self.automaton.make_automaton()
            
            logger.info(f"🚀 Built Aho-Corasick automaton with {phrase_count} phrases")
            logger.info(f"📊 Memory usage: ~{len(self.phrase_to_category) * 100 / 1024:.2f} KB")
            
        except Exception as e:
            logger.error(f"❌ Error building automaton: {e}")
            self.automaton = None
    
    def _add_phrases_to_automaton(
        self,
        phrases: List[str],
        category: str,
        subcategory: Optional[str]
    ) -> int:
        """
        إضافة قائمة عبارات إلى الـ Automaton

        Returns:
            عدد العبارات المضافة
        """
        if not phrases:
            return 0

        count = 0
        for phrase in phrases:
            if not phrase:
                continue

            # تطبيع خفيف فقط (lowercase + إزالة المسافات الزائدة)
            # لا نستخدم normalize_order_text لأنه يحول كل شيء إلى "أريد"
            normalized_phrase = phrase.lower().strip()
            normalized_phrase = " ".join(normalized_phrase.split())

            # تصحيح الأخطاء الإملائية
            words = normalized_phrase.split()
            corrected_words = [word_variants_generator.correct_typo(word) for word in words]
            corrected_phrase = " ".join(corrected_words)

            # إضافة العبارة إلى الـ Automaton
            try:
                self.automaton.add_word(corrected_phrase, (category, subcategory, corrected_phrase))

                # حفظ التطابق في القاموس للرجوع إليه
                self.phrase_to_category[corrected_phrase] = (category, subcategory)

                count += 1
            except Exception as e:
                logger.warning(f"⚠️ Could not add phrase '{phrase}': {e}")

        return count
    
    def search(self, text: str) -> Tuple[Optional[str], Optional[str], float, Optional[str]]:
        """
        البحث السريع عن تطابق في النص

        Args:
            text: النص المراد البحث فيه

        Returns:
            Tuple[category, subcategory, confidence, matched_phrase]
        """
        if not text or not self.automaton:
            return None, None, 0.0, None

        # تطبيع خفيف فقط (lowercase + إزالة المسافات الزائدة)
        normalized_text = text.lower().strip()
        normalized_text = " ".join(normalized_text.split())

        # تصحيح الأخطاء الإملائية
        words = normalized_text.split()
        corrected_words = [word_variants_generator.correct_typo(word) for word in words]
        corrected_text = " ".join(corrected_words)

        text_words = set(corrected_text.split())

        best_match = None
        best_confidence = 0.0
        best_phrase = None

        # البحث باستخدام Aho-Corasick (سريع جداً!)
        all_matches = []
        for end_index, (category, subcategory, phrase) in self.automaton.iter(corrected_text):
            all_matches.append((category, subcategory, phrase, end_index))

        # معالجة جميع التطابقات - إعطاء أولوية للعبارات الأطول
        for category, subcategory, phrase, end_index in all_matches:
            phrase_words = set(phrase.split())
            phrase_len = len(phrase_words)

            # تطابق تام
            if phrase == corrected_text:
                return category, subcategory, 1.0, phrase

            # تطابق كامل داخل النص
            if phrase in corrected_text:
                # حساب نسبة التطابق
                confidence = len(phrase) / len(corrected_text)

                # إذا كانت العبارة كلمة واحدة
                if phrase_len == 1:
                    confidence = min(confidence, 0.95)
                else:
                    confidence = min(confidence, 0.95)

                # إعطاء أولوية للعبارات الأطول عند نفس الثقة
                if confidence > best_confidence or (confidence == best_confidence and phrase_len > len(best_phrase.split()) if best_phrase else 0):
                    best_confidence = confidence
                    best_match = (category, subcategory)
                    best_phrase = phrase

            # تطابق جزئي (كلمات مشتركة)
            elif phrase_words & text_words:
                common_words = phrase_words & text_words

                # إذا كانت جميع كلمات العبارة موجودة في النص
                if len(common_words) == len(phrase_words):
                    confidence = 0.90
                else:
                    # حساب نسبة التطابق
                    confidence = len(common_words) / len(phrase_words) * 0.85

                # إعطاء أولوية للعبارات الأطول عند نفس الثقة
                if confidence > best_confidence or (confidence == best_confidence and phrase_len > len(best_phrase.split()) if best_phrase else 0):
                    best_confidence = confidence
                    best_match = (category, subcategory)
                    best_phrase = phrase

        if best_match:
            return best_match[0], best_match[1], best_confidence, best_phrase

        return None, None, 0.0, None
    
    def detect_expression_type(self, text: str) -> Tuple[Optional[str], Optional[str], float]:
        """
        كشف نوع التعبير من النص (واجهة متوافقة مع النظام القديم)
        
        Args:
            text: النص المراد تحليله
            
        Returns:
            Tuple[category, subcategory, confidence]
        """
        category, subcategory, confidence, _ = self.search(text)
        return category, subcategory, confidence
    
    def is_confirmation(self, text: str) -> Tuple[bool, bool]:
        """التحقق من أن النص تأكيد أو رفض"""
        category, subcategory, confidence = self.detect_expression_type(text)
        
        if category == "confirmation" and confidence > 0.7:
            is_accept = (subcategory == "accept")
            return True, is_accept
        
        return False, False
    
    def is_order_request(self, text: str) -> bool:
        """التحقق من أن النص طلب جديد"""
        category, _, confidence = self.detect_expression_type(text)
        return category == "new_order" and confidence > 0.5
    
    def is_cart_request(self, text: str) -> bool:
        """التحقق من أن النص طلب عرض السلة"""
        category, _, confidence = self.detect_expression_type(text)
        return category == "view_cart" and confidence > 0.7
    
    def is_cancel_request(self, text: str) -> bool:
        """التحقق من أن النص طلب إلغاء"""
        category, subcategory, confidence = self.detect_expression_type(text)
        return (category == "completion" and 
                subcategory == "cancel_order" and 
                confidence > 0.7)
    
    def is_finish_request(self, text: str) -> bool:
        """التحقق من أن النص طلب إكمال الطلب"""
        category, subcategory, confidence = self.detect_expression_type(text)
        return (category == "completion" and 
                subcategory == "finish_order" and 
                confidence > 0.7)
    
    def is_price_inquiry(self, text: str) -> bool:
        """التحقق من أن النص استفسار عن السعر"""
        category, subcategory, confidence = self.detect_expression_type(text)
        return (category == "inquiry" and
                subcategory == "price" and
                confidence > 0.5)

    def is_availability_inquiry(self, text: str) -> bool:
        """التحقق من أن النص استفسار عن التوفر"""
        category, subcategory, confidence = self.detect_expression_type(text)
        return (category == "inquiry" and
                subcategory == "availability" and
                confidence > 0.5)
    
    def is_menu_request(self, text: str) -> bool:
        """التحقق من أن النص طلب عرض المنيو"""
        category, subcategory, confidence = self.detect_expression_type(text)
        return (category == "inquiry" and 
                subcategory == "menu" and 
                confidence > 0.7)
    
    def extract_quantity(self, text: str) -> Optional[int]:
        """استخراج الكمية من النص"""
        return word_variants_generator.extract_number_from_text(text)
    
    def is_quantity_modification(self, text: str) -> Tuple[bool, Optional[str]]:
        """التحقق من أن النص تعديل للكمية"""
        category, subcategory, confidence = self.detect_expression_type(text)
        
        if category == "quantity" and confidence > 0.6:
            return True, subcategory
        
        return False, None
    
    def is_modification_request(self, text: str) -> Tuple[bool, Optional[str]]:
        """التحقق من أن النص طلب تعديل"""
        category, subcategory, confidence = self.detect_expression_type(text)
        
        if category == "modification" and confidence > 0.7:
            return True, subcategory
        
        return False, None
    
    def get_expression_info(self, text: str) -> Dict:
        """الحصول على معلومات كاملة عن التعبير"""
        category, subcategory, confidence = self.detect_expression_type(text)
        quantity = self.extract_quantity(text)
        is_conf, is_accept = self.is_confirmation(text)
        
        return {
            "category": category,
            "subcategory": subcategory,
            "confidence": confidence,
            "quantity": quantity,
            "is_confirmation": is_conf,
            "is_accept": is_accept if is_conf else None,
            "is_order": self.is_order_request(text),
            "is_cart": self.is_cart_request(text),
            "is_cancel": self.is_cancel_request(text),
            "is_finish": self.is_finish_request(text),
        }
    
    def get_stats(self) -> Dict:
        """الحصول على إحصائيات الأداء"""
        return {
            "total_phrases": len(self.phrase_to_category),
            "categories": len(self.expressions_dict),
            "memory_kb": len(self.phrase_to_category) * 100 / 1024,
            "automaton_built": self.automaton is not None,
        }


# مثيل عام للاستخدام
fast_expression_matcher = FastExpressionMatcher()


# دوال مساعدة للاستخدام السريع
def detect_expression_fast(text: str) -> Tuple[Optional[str], Optional[str], float]:
    """كشف نوع التعبير بسرعة فائقة"""
    return fast_expression_matcher.detect_expression_type(text)


def is_confirmation_fast(text: str) -> Tuple[bool, bool]:
    """التحقق من التأكيد بسرعة"""
    return fast_expression_matcher.is_confirmation(text)


def is_order_request_fast(text: str) -> bool:
    """التحقق من طلب جديد بسرعة"""
    return fast_expression_matcher.is_order_request(text)


def get_expression_info_fast(text: str) -> Dict:
    """الحصول على معلومات كاملة بسرعة"""
    return fast_expression_matcher.get_expression_info(text)

