"""
🎯 محلل التعبيرات الذكي
Smart Expression Detector

يكشف نوع التعبير من النص باستخدام القاموس الموحد
"""

import yaml
import os
from typing import Optional, Tuple, Dict, List, Set
from pathlib import Path

from app.utils.text_helpers import normalize_order_text
from app.utils.word_variants import word_variants_generator
from app.utils.logger import app_logger as logger


class ExpressionDetector:
    """كاشف التعبيرات الذكي"""
    
    def __init__(self):
        self.expressions_dict: Dict = {}
        self.numbers_dict: Dict = {}
        self._load_dictionary()
    
    def _load_dictionary(self):
        """تحميل قاموس التعبيرات من ملف YAML"""
        try:
            dict_path = Path(__file__).parent / "order_expression_dict.yaml"
            
            if not dict_path.exists():
                logger.error(f"❌ Expression dictionary not found: {dict_path}")
                return
            
            with open(dict_path, 'r', encoding='utf-8') as f:
                data = yaml.safe_load(f)
                self.expressions_dict = data.get('order_expressions', {})
                self.numbers_dict = data.get('numbers', {})
            
            logger.info(f"✅ Loaded expression dictionary with {len(self.expressions_dict)} categories")
        
        except Exception as e:
            logger.error(f"❌ Error loading expression dictionary: {e}")
    
    def detect_expression_type(self, text: str) -> Tuple[Optional[str], Optional[str], float]:
        """
        كشف نوع التعبير من النص
        
        Args:
            text: النص المراد تحليله
            
        Returns:
            Tuple[category, subcategory, confidence]
            مثال: ("confirmation", "accept", 0.95)
        """
        if not text:
            return None, None, 0.0
        
        # تطبيع النص
        normalized_text = normalize_order_text(text.lower())
        
        # تصحيح الأخطاء الإملائية
        words = normalized_text.split()
        corrected_words = [word_variants_generator.correct_typo(word) for word in words]
        corrected_text = " ".join(corrected_words)
        
        best_match = None
        best_confidence = 0.0
        
        # البحث في كل فئة
        for category, subcategories in self.expressions_dict.items():
            
            # إذا كانت الفئة تحتوي على فئات فرعية
            if isinstance(subcategories, dict):
                for subcategory, phrases in subcategories.items():
                    
                    # إذا كانت الفئة الفرعية تحتوي على لهجات
                    if isinstance(phrases, dict):
                        for dialect, dialect_phrases in phrases.items():
                            confidence = self._check_phrases(
                                corrected_text, 
                                dialect_phrases
                            )
                            if confidence > best_confidence:
                                best_confidence = confidence
                                best_match = (category, subcategory)
                    
                    # إذا كانت الفئة الفرعية قائمة مباشرة
                    elif isinstance(phrases, list):
                        confidence = self._check_phrases(
                            corrected_text, 
                            phrases
                        )
                        if confidence > best_confidence:
                            best_confidence = confidence
                            best_match = (category, subcategory)
            
            # إذا كانت الفئة قائمة مباشرة (بدون فئات فرعية)
            elif isinstance(subcategories, list):
                confidence = self._check_phrases(
                    corrected_text, 
                    subcategories
                )
                if confidence > best_confidence:
                    best_confidence = confidence
                    best_match = (category, None)
        
        if best_match:
            return best_match[0], best_match[1], best_confidence
        
        return None, None, 0.0
    
    def _check_phrases(self, text: str, phrases: List[str]) -> float:
        """
        التحقق من وجود أي عبارة من القائمة في النص

        Args:
            text: النص المراد البحث فيه
            phrases: قائمة العبارات للبحث عنها

        Returns:
            درجة الثقة (0.0 - 1.0)
        """
        if not phrases:
            return 0.0

        max_confidence = 0.0
        text_words = set(text.split())

        for phrase in phrases:
            phrase_lower = phrase.lower()
            phrase_words = set(phrase_lower.split())

            # تطابق تام
            if phrase_lower == text:
                return 1.0

            # تطابق كامل داخل النص
            if phrase_lower in text:
                # إذا كانت العبارة كلمة واحدة وموجودة في النص
                if len(phrase_words) == 1:
                    max_confidence = max(max_confidence, 0.95)
                else:
                    # حساب نسبة التطابق
                    confidence = len(phrase_lower) / len(text)
                    max_confidence = max(max_confidence, min(confidence, 0.95))

            # تطابق جزئي (كلمات مشتركة)
            common_words = phrase_words & text_words

            if common_words:
                # إذا كانت جميع كلمات العبارة موجودة في النص
                if len(common_words) == len(phrase_words):
                    max_confidence = max(max_confidence, 0.90)
                else:
                    # حساب نسبة التطابق
                    confidence = len(common_words) / len(phrase_words)
                    max_confidence = max(max_confidence, confidence * 0.85)

        return max_confidence
    
    def is_confirmation(self, text: str) -> Tuple[bool, bool]:
        """
        التحقق من أن النص تأكيد أو رفض
        
        Returns:
            Tuple[is_confirmation, is_accept]
            مثال: (True, True) = تأكيد موافقة
                  (True, False) = تأكيد رفض
                  (False, False) = ليس تأكيد
        """
        category, subcategory, confidence = self.detect_expression_type(text)
        
        if category == "confirmation" and confidence > 0.7:
            is_accept = (subcategory == "accept")
            return True, is_accept
        
        return False, False
    
    def is_order_request(self, text: str) -> bool:
        """التحقق من أن النص طلب جديد"""
        category, _, confidence = self.detect_expression_type(text)
        return category == "new_order" and confidence > 0.5
    
    def is_cart_request(self, text: str) -> bool:
        """التحقق من أن النص طلب عرض السلة"""
        category, _, confidence = self.detect_expression_type(text)
        return category == "view_cart" and confidence > 0.7
    
    def is_cancel_request(self, text: str) -> bool:
        """التحقق من أن النص طلب إلغاء"""
        category, subcategory, confidence = self.detect_expression_type(text)
        return (category == "completion" and 
                subcategory == "cancel_order" and 
                confidence > 0.7)
    
    def is_finish_request(self, text: str) -> bool:
        """التحقق من أن النص طلب إكمال الطلب"""
        category, subcategory, confidence = self.detect_expression_type(text)
        return (category == "completion" and 
                subcategory == "finish_order" and 
                confidence > 0.7)
    
    def is_price_inquiry(self, text: str) -> bool:
        """التحقق من أن النص استفسار عن السعر"""
        category, subcategory, confidence = self.detect_expression_type(text)
        return (category == "inquiry" and
                subcategory == "price" and
                confidence > 0.5)

    def is_availability_inquiry(self, text: str) -> bool:
        """التحقق من أن النص استفسار عن التوفر"""
        category, subcategory, confidence = self.detect_expression_type(text)
        return (category == "inquiry" and
                subcategory == "availability" and
                confidence > 0.5)
    
    def is_menu_request(self, text: str) -> bool:
        """التحقق من أن النص طلب عرض المنيو"""
        category, subcategory, confidence = self.detect_expression_type(text)
        return (category == "inquiry" and 
                subcategory == "menu" and 
                confidence > 0.7)
    
    def extract_quantity(self, text: str) -> Optional[int]:
        """
        استخراج الكمية من النص
        
        Returns:
            الكمية أو None
        """
        return word_variants_generator.extract_number_from_text(text)
    
    def is_quantity_modification(self, text: str) -> Tuple[bool, Optional[str]]:
        """
        التحقق من أن النص تعديل للكمية
        
        Returns:
            Tuple[is_modification, modification_type]
            modification_type: "increase", "decrease", "exact"
        """
        category, subcategory, confidence = self.detect_expression_type(text)
        
        if category == "quantity" and confidence > 0.6:
            return True, subcategory
        
        return False, None
    
    def is_modification_request(self, text: str) -> Tuple[bool, Optional[str]]:
        """
        التحقق من أن النص طلب تعديل
        
        Returns:
            Tuple[is_modification, modification_type]
            modification_type: "change_item", "edit_last"
        """
        category, subcategory, confidence = self.detect_expression_type(text)
        
        if category == "modification" and confidence > 0.7:
            return True, subcategory
        
        return False, None
    
    def get_expression_info(self, text: str) -> Dict:
        """
        الحصول على معلومات كاملة عن التعبير
        
        Returns:
            قاموس يحتوي على جميع المعلومات
        """
        category, subcategory, confidence = self.detect_expression_type(text)
        quantity = self.extract_quantity(text)
        is_conf, is_accept = self.is_confirmation(text)
        
        return {
            "category": category,
            "subcategory": subcategory,
            "confidence": confidence,
            "quantity": quantity,
            "is_confirmation": is_conf,
            "is_accept": is_accept if is_conf else None,
            "is_order": self.is_order_request(text),
            "is_cart": self.is_cart_request(text),
            "is_cancel": self.is_cancel_request(text),
            "is_finish": self.is_finish_request(text),
        }


# مثيل عام للاستخدام
expression_detector = ExpressionDetector()


# دوال مساعدة للاستخدام السريع
def detect_expression(text: str) -> Tuple[Optional[str], Optional[str], float]:
    """كشف نوع التعبير"""
    return expression_detector.detect_expression_type(text)


def is_confirmation(text: str) -> Tuple[bool, bool]:
    """التحقق من التأكيد"""
    return expression_detector.is_confirmation(text)


def is_order_request(text: str) -> bool:
    """التحقق من طلب جديد"""
    return expression_detector.is_order_request(text)


def extract_quantity(text: str) -> Optional[int]:
    """استخراج الكمية"""
    return expression_detector.extract_quantity(text)


def get_expression_info(text: str) -> Dict:
    """الحصول على معلومات كاملة"""
    return expression_detector.get_expression_info(text)

