"""
Enhanced Language Detector
كاشف لغة متقدم مع دعم النصوص المختلطة وتذكر التفضيلات
"""

from typing import Tuple, Optional, List, Dict
import re
from collections import Counter
import logging

logger = logging.getLogger(__name__)


class EnhancedLanguageDetector:
    """
    كاشف لغة متقدم
    
    الميزات:
    - كشف العربية والإنجليزية
    - دعم النصوص المختلطة
    - Confidence scoring
    - تذكر تفضيلات المستخدم
    - كشف الأرقام والرموز
    """
    
    def __init__(self):
        # أنماط اللغات
        self.patterns = {
            'ar': r'[\u0600-\u06FF\u0750-\u077F\u08A0-\u08FF]',  # عربي موسع
            'en': r'[a-zA-Z]',
            'number': r'[0-9٠-٩]',
            'emoji': r'[\U0001F600-\U0001F64F\U0001F300-\U0001F5FF\U0001F680-\U0001F6FF\U0001F1E0-\U0001F1FF]'
        }
        
        # كلمات مفتاحية لكل لغة
        self.keywords = {
            'ar': [
                # أفعال شائعة
                'أريد', 'بغيت', 'عايز', 'أبي', 'بدي', 'أبغى',
                # ردود
                'نعم', 'لا', 'تمام', 'خلاص', 'أكمل', 'كفاية',
                # تحيات
                'شكرا', 'مرحبا', 'السلام', 'صباح', 'مساء', 'أهلا',
                # أسئلة
                'كيف', 'متى', 'أين', 'ماذا', 'لماذا', 'من', 'هل',
                # أصناف شائعة
                'حمص', 'متبل', 'تبولة', 'فتوش', 'شوربة', 'سلطة',
                'كبسة', 'مشوي', 'سمك', 'دجاج', 'لحم',
                # أخرى
                'المنيو', 'القائمة', 'طلب', 'توصيل', 'عنوان'
            ],
            'en': [
                # verbs
                'want', 'need', 'like', 'order', 'get', 'have',
                # responses
                'yes', 'no', 'ok', 'okay', 'done', 'finish',
                # greetings
                'hello', 'hi', 'hey', 'thanks', 'thank', 'bye',
                # questions
                'how', 'when', 'where', 'what', 'why', 'who',
                # common items
                'hummus', 'salad', 'soup', 'chicken', 'fish', 'meat',
                # other
                'menu', 'order', 'delivery', 'address', 'cart'
            ]
        }
        
        # تاريخ اللغات للمستخدمين
        self.user_language_history: dict[str, List[str]] = {}
    
    def detect(
        self,
        text: str,
        user_id: Optional[str] = None,
        session_lang: Optional[str] = None
    ) -> Tuple[str, float]:
        """
        كشف اللغة
        
        Args:
            text: النص
            user_id: معرف المستخدم (اختياري)
            session_lang: لغة الجلسة الحالية (اختياري)
        
        Returns:
            (language, confidence) - ('ar' أو 'en', 0.0-1.0)
        """
        if not text or not text.strip():
            return ('ar', 0.5)  # افتراضي
        
        text = text.strip()
        
        # 1. فحص خاص: رسائل قصيرة جداً
        if len(text) <= 3:
            return self._detect_short_text(text, session_lang)
        
        # 2. فحص خاص: أرقام فقط
        if text.replace(' ', '').isdigit():
            # استخدام لغة الجلسة أو تاريخ المستخدم
            return (session_lang or self._get_user_preferred_lang(user_id), 0.7)
        
        # 3. حساب نسبة كل لغة
        ar_score = self._calculate_language_score(text, 'ar')
        en_score = self._calculate_language_score(text, 'en')
        
        # 4. تحديد اللغة
        if ar_score > en_score:
            language = 'ar'
            confidence = ar_score / (ar_score + en_score) if (ar_score + en_score) > 0 else 0.5
        elif en_score > ar_score:
            language = 'en'
            confidence = en_score / (ar_score + en_score) if (ar_score + en_score) > 0 else 0.5
        else:
            # متساوية - استخدام السياق
            language = session_lang or self._get_user_preferred_lang(user_id)
            confidence = 0.5
        
        # 5. تعديل الثقة بناءً على السياق
        if session_lang and session_lang == language:
            confidence = min(confidence + 0.1, 1.0)
        
        # 6. حفظ في التاريخ
        if user_id:
            self._update_user_history(user_id, language)
        
        logger.debug(f"Language detected: {language} (confidence: {confidence:.2f}) for text: '{text[:50]}'")
        
        return (language, confidence)
    
    def _calculate_language_score(self, text: str, lang: str) -> float:
        """حساب درجة اللغة"""
        score = 0.0
        
        # 1. عدد الأحرف
        pattern = self.patterns[lang]
        matches = re.findall(pattern, text)
        char_count = len(matches)
        char_ratio = char_count / len(text) if len(text) > 0 else 0
        score += char_ratio * 10
        
        # 2. الكلمات المفتاحية
        text_lower = text.lower()
        keywords = self.keywords[lang]
        keyword_matches = sum(1 for keyword in keywords if keyword in text_lower)
        score += keyword_matches * 2
        
        # 3. أنماط خاصة
        if lang == 'ar':
            # فحص التشكيل
            if re.search(r'[\u064B-\u065F]', text):
                score += 1
            
            # فحص حروف خاصة بالعربية
            if re.search(r'[ةىأإآؤئ]', text):
                score += 1
        
        elif lang == 'en':
            # فحص مسافات منتظمة (الإنجليزية عادة أكثر تنظيماً)
            words = text.split()
            if len(words) > 1:
                avg_word_length = sum(len(w) for w in words) / len(words)
                if 3 <= avg_word_length <= 7:
                    score += 0.5
        
        return score
    
    def _detect_short_text(
        self,
        text: str,
        session_lang: Optional[str]
    ) -> Tuple[str, float]:
        """كشف النصوص القصيرة (1-3 أحرف)"""
        text_lower = text.lower()
        
        # ردود عربية شائعة
        ar_short = ['نعم', 'لا', 'اه', 'آه', 'لأ', 'مو', 'ما', 'كم', 'اي']
        if text_lower in ar_short:
            return ('ar', 0.95)
        
        # ردود إنجليزية شائعة
        en_short = ['yes', 'no', 'ok', 'hi', 'bye', 'yep', 'nah']
        if text_lower in en_short:
            return ('en', 0.95)
        
        # فحص الأحرف
        if re.search(self.patterns['ar'], text):
            return ('ar', 0.9)
        elif re.search(self.patterns['en'], text):
            return ('en', 0.9)
        
        # استخدام لغة الجلسة
        return (session_lang or 'ar', 0.6)
    
    def _get_user_preferred_lang(self, user_id: Optional[str]) -> str:
        """الحصول على اللغة المفضلة للمستخدم"""
        if not user_id or user_id not in self.user_language_history:
            return 'ar'  # افتراضي
        
        history = self.user_language_history[user_id]
        if not history:
            return 'ar'
        
        # أكثر لغة استخداماً
        counter = Counter(history)
        most_common = counter.most_common(1)[0][0]
        
        return most_common
    
    def _update_user_history(self, user_id: str, language: str):
        """تحديث تاريخ اللغة للمستخدم"""
        if user_id not in self.user_language_history:
            self.user_language_history[user_id] = []
        
        self.user_language_history[user_id].append(language)
        
        # الاحتفاظ بآخر 20 رسالة فقط
        if len(self.user_language_history[user_id]) > 20:
            self.user_language_history[user_id] = self.user_language_history[user_id][-20:]
    
    def detect_mixed_language(self, text: str) -> Dict[str, float]:
        """
        كشف النصوص المختلطة
        
        Returns:
            {'ar': 0.6, 'en': 0.4} - نسبة كل لغة
        """
        if not text:
            return {'ar': 0.0, 'en': 0.0}
        
        # حساب عدد أحرف كل لغة
        ar_chars = len(re.findall(self.patterns['ar'], text))
        en_chars = len(re.findall(self.patterns['en'], text))
        total_chars = ar_chars + en_chars
        
        if total_chars == 0:
            return {'ar': 0.0, 'en': 0.0}
        
        return {
            'ar': ar_chars / total_chars,
            'en': en_chars / total_chars
        }
    
    def is_arabic_dominant(self, text: str, threshold: float = 0.5) -> bool:
        """هل العربية هي اللغة السائدة؟"""
        mixed = self.detect_mixed_language(text)
        return mixed['ar'] >= threshold
    
    def is_english_dominant(self, text: str, threshold: float = 0.5) -> bool:
        """هل الإنجليزية هي اللغة السائدة؟"""
        mixed = self.detect_mixed_language(text)
        return mixed['en'] >= threshold
    
    def normalize_text(self, text: str, target_lang: str = 'ar') -> str:
        """
        تطبيع النص
        - تحويل الأرقام الهندية للعربية
        - إزالة التشكيل
        - توحيد المسافات
        """
        # إزالة المسافات الزائدة
        text = ' '.join(text.split())
        
        if target_lang == 'ar':
            # تحويل الأرقام الإنجليزية للعربية
            arabic_numbers = '٠١٢٣٤٥٦٧٨٩'
            english_numbers = '0123456789'
            trans_table = str.maketrans(english_numbers, arabic_numbers)
            text = text.translate(trans_table)
            
            # إزالة التشكيل
            text = re.sub(r'[\u064B-\u065F]', '', text)
            
            # توحيد الهمزات
            text = re.sub(r'[إأآ]', 'ا', text)
            text = re.sub(r'[ىي]', 'ي', text)
            text = re.sub(r'ة', 'ه', text)
        
        return text
    
    def get_user_stats(self, user_id: str) -> Dict:
        """إحصائيات المستخدم"""
        if user_id not in self.user_language_history:
            return {
                'total_messages': 0,
                'ar_count': 0,
                'en_count': 0,
                'preferred_lang': 'ar'
            }
        
        history = self.user_language_history[user_id]
        counter = Counter(history)
        
        return {
            'total_messages': len(history),
            'ar_count': counter.get('ar', 0),
            'en_count': counter.get('en', 0),
            'preferred_lang': counter.most_common(1)[0][0] if counter else 'ar'
        }


# Singleton instance
enhanced_language_detector = EnhancedLanguageDetector()

