"""
Cart Formatter - تنسيق عرض السلة الموحد
يعرض جميع الأسعار بالريال العماني (OMR) بدون VAT
"""

from typing import Dict, List, Optional
from app.utils.logger import app_logger as logger


class CartFormatter:
    """منسق السلة - تنسيق موحد بالعربي والإنجليزي"""
    
    DEFAULT_CURRENCY = "OMR"
    DEFAULT_DELIVERY_FEE = 0.500  # 500 بيسة = 0.5 ريال
    
    def __init__(self):
        """تهيئة المنسق"""
        pass
    
    def calculate_totals(
        self,
        items: List[Dict],
        discount: float = 0.0,
        delivery_fee: Optional[float] = None
    ) -> Dict[str, float]:
        """
        حساب الإجماليات
        
        Args:
            items: قائمة الأصناف
            discount: الخصم
            delivery_fee: رسوم التوصيل (اختياري)
            
        Returns:
            Dict مع subtotal, discount, delivery_fee, total
        """
        
        # حساب الإجمالي قبل الخصم
        subtotal = 0.0
        for item in items:
            quantity = item.get('quantity', 1)
            unit_price = float(item.get('price', 0))
            line_total = quantity * unit_price
            subtotal += line_total
        
        # رسوم التوصيل
        if delivery_fee is None:
            delivery_fee = self.DEFAULT_DELIVERY_FEE if subtotal > 0 else 0.0
        
        # الإجمالي النهائي
        total = subtotal - discount + delivery_fee
        
        # التأكد من عدم وجود قيم سالبة
        total = max(0.0, total)
        
        return {
            'subtotal': round(subtotal, 2),
            'discount': round(discount, 2),
            'delivery_fee': round(delivery_fee, 2),
            'total': round(total, 2)
        }
    
    def format_cart_ar(
        self,
        items: List[Dict],
        totals: Optional[Dict] = None,
        currency: str = None
    ) -> str:
        """
        تنسيق السلة بالعربي

        Args:
            items: قائمة الأصناف
            totals: الإجماليات (اختياري - سيتم حسابها تلقائياً)
            currency: العملة (افتراضي: OMR)

        Returns:
            نص منسق بالعربي
        """

        if currency is None:
            currency = self.DEFAULT_CURRENCY

        # التحقق من السلة الفارغة
        if not items or len(items) == 0:
            return "🛒 سلتك فارغة حالياً.\n\nاختر من المنيو لإضافة أصناف!"

        # حساب الإجماليات إذا لم تكن موجودة
        if totals is None:
            totals = self.calculate_totals(items)

        # بناء النص
        lines = ["🛒 *سلتك الحالية:*\n"]

        # الأصناف
        for idx, item in enumerate(items, 1):
            name = item.get('name_ar') or item.get('name', 'صنف')
            quantity = item.get('quantity', 1)
            unit_price = float(item.get('price', 0))
            line_total = quantity * unit_price

            # تنسيق حسب المطلوب: 1️⃣ {اسم الصنف} × {الكمية} — {السعر_الفرعي} ر.ع
            emoji_num = ['1️⃣', '2️⃣', '3️⃣', '4️⃣', '5️⃣', '6️⃣', '7️⃣', '8️⃣', '9️⃣', '🔟'][min(idx-1, 9)]
            line = f'{emoji_num} {name} × {quantity} — {line_total:.3f} ر.ع'
            lines.append(line)

        # خط فاصل
        lines.append("\n━━━━━━━━━━━━━━━━━━━━━━━")

        # الإجماليات
        lines.append(f'💰 *الإجمالي قبل الخصم:* {totals["subtotal"]:.3f} ر.ع')

        if totals['discount'] > 0:
            lines.append(f'🎁 *الخصم:* -{totals["discount"]:.3f} ر.ع')

        lines.append(f'🚚 *رسوم التوصيل:* {totals["delivery_fee"]:.3f} ر.ع')
        lines.append(f'✅ *الإجمالي النهائي:* {totals["total"]:.3f} ر.ع')
        lines.append("━━━━━━━━━━━━━━━━━━━━━━━")

        return "\n".join(lines)
    
    def format_cart_en(
        self,
        items: List[Dict],
        totals: Optional[Dict] = None,
        currency: str = None
    ) -> str:
        """
        تنسيق السلة بالإنجليزي

        Args:
            items: قائمة الأصناف
            totals: الإجماليات (اختياري - سيتم حسابها تلقائياً)
            currency: العملة (افتراضي: OMR)

        Returns:
            نص منسق بالإنجليزي
        """

        if currency is None:
            currency = self.DEFAULT_CURRENCY

        # التحقق من السلة الفارغة
        if not items or len(items) == 0:
            return "🛒 Your cart is empty.\n\nChoose from the menu to add items!"

        # حساب الإجماليات إذا لم تكن موجودة
        if totals is None:
            totals = self.calculate_totals(items)

        # بناء النص
        lines = ["🛒 *Your current cart:*\n"]

        # الأصناف
        for idx, item in enumerate(items, 1):
            name = item.get('name_en') or item.get('name', 'Item')
            quantity = item.get('quantity', 1)
            unit_price = float(item.get('price', 0))
            line_total = quantity * unit_price

            # تنسيق حسب المطلوب: 1️⃣ {Item Name} × {Quantity} — {Subtotal} OMR
            emoji_num = ['1️⃣', '2️⃣', '3️⃣', '4️⃣', '5️⃣', '6️⃣', '7️⃣', '8️⃣', '9️⃣', '🔟'][min(idx-1, 9)]
            line = f'{emoji_num} {name} × {quantity} — {line_total:.3f} OMR'
            lines.append(line)

        # خط فاصل
        lines.append("\n━━━━━━━━━━━━━━━━━━━━━━━")

        # الإجماليات
        lines.append(f'💰 *Subtotal:* {totals["subtotal"]:.3f} OMR')

        if totals['discount'] > 0:
            lines.append(f'🎁 *Discount:* -{totals["discount"]:.3f} OMR')

        lines.append(f'🚚 *Delivery Fee:* {totals["delivery_fee"]:.3f} OMR')
        lines.append(f'✅ *Total:* {totals["total"]:.3f} OMR')
        lines.append("━━━━━━━━━━━━━━━━━━━━━━━")

        return "\n".join(lines)
    
    def format_cart(
        self,
        items: List[Dict],
        lang: str = "ar",
        totals: Optional[Dict] = None,
        currency: str = None
    ) -> str:
        """
        تنسيق السلة حسب اللغة
        
        Args:
            items: قائمة الأصناف
            lang: اللغة (ar أو en)
            totals: الإجماليات (اختياري)
            currency: العملة (افتراضي: OMR)
            
        Returns:
            نص منسق
        """
        
        if lang == "en":
            return self.format_cart_en(items, totals, currency)
        else:
            return self.format_cart_ar(items, totals, currency)
    
    def format_item_added_message(
        self,
        item: Dict,
        quantity: int,
        lang: str = "ar"
    ) -> str:
        """ 
        رسالة إضافة صنف
        
        Args:
            item: الصنف المضاف
            quantity: الكمية
            lang: اللغة
            
        Returns:
            رسالة منسقة
        """
        
        name = item.get('name_ar' if lang == 'ar' else 'name_en', item.get('name', 'Item'))
        price = float(item.get('price', 0))
        total = quantity * price
        
        if lang == "ar":
            msg = f"✅ **تمت الإضافة!**\n\n"
            msg += f"📦 {name}\n"
            msg += f"🔢 الكمية: {quantity}\n"
            msg += f"💰 السعر: {price:.2f} {self.DEFAULT_CURRENCY}\n"
            msg += f"💵 الإجمالي: {total:.2f} {self.DEFAULT_CURRENCY}"
        else:
            msg = f"✅ **Added!**\n\n"
            msg += f"📦 {name}\n"
            msg += f"🔢 Quantity: {quantity}\n"
            msg += f"💰 Price: {price:.2f} {self.DEFAULT_CURRENCY}\n"
            msg += f"💵 Total: {total:.2f} {self.DEFAULT_CURRENCY}"
        
        return msg
    
    def format_item_removed_message(
        self,
        item: Dict,
        lang: str = "ar"
    ) -> str:
        """
        رسالة حذف صنف
        
        Args:
            item: الصنف المحذوف
            lang: اللغة
            
        Returns:
            رسالة منسقة
        """
        
        name = item.get('name_ar' if lang == 'ar' else 'name_en', item.get('name', 'Item'))
        
        if lang == "ar":
            msg = f"🗑️ **تم الحذف!**\n\n"
            msg += f"تم حذف: {name}"
        else:
            msg = f"🗑️ **Removed!**\n\n"
            msg += f"Removed: {name}"
        
        return msg
    
    def format_empty_cart_message(self, lang: str = "ar") -> str:
        """
        رسالة سلة فارغة
        
        Args:
            lang: اللغة
            
        Returns:
            رسالة منسقة
        """
        
        if lang == "ar":
            return "🛒 سلتك فارغة حالياً.\n\nاختر من المنيو لإضافة أصناف!"
        else:
            return "🛒 Your cart is empty.\n\nChoose from the menu to add items!"


# Singleton instance
cart_formatter = CartFormatter()

