"""
AI-Powered Intent Analyzer
يستخدم OpenAI لتحليل النية من الرسالة + السياق (آخر 3 محادثات)
"""

import os
import json
from typing import Dict, List, Optional
from openai import AsyncOpenAI
from app.utils.logger import logger

# Cache for menu items
_menu_items_cache = None

# Lazy initialization of OpenAI client
_openai_client = None

def _get_openai_client():
    """
    Lazy initialization of OpenAI client
    يتم تحميل API key من البيئة عند الحاجة فقط
    """
    global _openai_client

    if _openai_client is None:
        api_key = os.getenv("OPENAI_API_KEY")
        if not api_key:
            logger.error("❌ OPENAI_API_KEY not found in environment variables")
            raise ValueError("OPENAI_API_KEY is required for AI Intent Analyzer")

        _openai_client = AsyncOpenAI(api_key=api_key)
        logger.info("✅ OpenAI client initialized for AI Intent Analyzer")

    return _openai_client


class AIIntentAnalyzer:
    """
    محلل النية الذكي باستخدام AI
    يحلل الرسالة الحالية + آخر 3 محادثات لتحديد النية بدقة

    🎯 استراتيجية توفير التكلفة:
    - يستخدم AI فقط للحالات المعقدة (تحية + طلب، حذف سلة، إلخ)
    - يمرر أسماء الأصناف فقط (بدون تفاصيل) لتقليل التوكنات
    """

    SYSTEM_PROMPT = """أنت محلل نوايا ذكي لبوت مطعم.
مهمتك: تحليل رسالة العميل + السياق (آخر 3 رسائل) وتحديد النية بدقة.

**النوايا المتاحة:**
1. **greeting_only** - تحية فقط بدون طلب (مثل: "السلام عليكم", "مرحبا", "صباح الخير")
2. **greeting_with_order** - تحية + طلب طعام (مثل: "السلام عليكم ممكن برجر", "مرحبا عايز بيتزا")
3. **order_food** - طلب طعام مباشر (مثل: "عايز برجر", "ممكن بيتزا", "محتاج 2 بيتزا")
4. **remove_from_cart** - حذف من السلة (مثل: "امسح السلة", "احذف البرجر", "الغي الطلب")
5. **view_cart** - عرض السلة (مثل: "وش في السلة", "عايز اشوف طلبي")
6. **confirm_order** - تأكيد الطلب (مثل: "تمام", "اوكي", "نعم", "أكد")
7. **ask_question** - سؤال عام (مثل: "كم السعر؟", "عندكم توصيل؟")
8. **other** - أي شيء آخر

**قواعد مهمة:**
- إذا كانت الرسالة تحتوي على تحية + اسم صنف من المنيو → **greeting_with_order**
- إذا كانت الرسالة تحتوي على تحية فقط → **greeting_only**
- إذا كانت الرسالة تحتوي على "امسح", "احذف", "الغي" + "السلة" → **remove_from_cart**
- استخدم السياق (آخر 3 رسائل) لفهم النية بشكل أفضل

**فهم الرسائل الغامضة:**
- "محتاج 2" أو "عايز 3" → ابحث في السياق عن آخر صنف تم ذكره
- "محتاج 2 بيتزا" → النية: order_food، الصنف: "بيتزا"، الكمية: 2
- "2 برجر" → النية: order_food، الصنف: "برجر"، الكمية: 2
- إذا كانت الرسالة تحتوي على رقم + كلمة غامضة → ابحث في المنيو عن أقرب تطابق

**استخراج الأصناف (مهم جداً):**
- **تحقق أولاً**: هل الصنف موجود في قائمة المنيو المرفقة؟
- **إذا كان موجوداً**: استخرج اسم الصنف بالضبط كما هو في المنيو
- **إذا لم يكن موجوداً**:
  * ابحث عن أقرب تطابق في المنيو (مثل: "بيتزا" → "برجر كلاسيك" إذا لم توجد بيتزا)
  * إذا لم يوجد تطابق قريب، استخرج الصنف كما ذكره العميل وضع علامة `item_not_in_menu: true`
- إذا كانت الرسالة تحتوي على رقم، استخرج الكمية أيضاً

**أمثلة:**

مثال 1 - صنف موجود:
الرسالة: "محتاج 2 برجر كلاسيك"
المنيو: ["برجر كلاسيك", "حمص", "كبة مقلية"]
الرد:
{
    "intent": "order_food",
    "confidence": 0.95,
    "reasoning": "طلب طعام - الصنف موجود في المنيو",
    "extracted_items": ["برجر كلاسيك"],
    "quantity": 2,
    "item_not_in_menu": false,
    "is_compound": false
}

مثال 2 - صنف غير موجود:
الرسالة: "محتاج 2 بيتزا"
المنيو: ["برجر كلاسيك", "حمص", "كبة مقلية"]
الرد:
{
    "intent": "order_food",
    "confidence": 0.90,
    "reasoning": "طلب طعام - لكن 'بيتزا' غير موجودة في المنيو",
    "extracted_items": ["بيتزا"],
    "quantity": 2,
    "item_not_in_menu": true,
    "suggested_alternatives": ["برجر كلاسيك"],
    "is_compound": false
}

**الرد يجب أن يكون JSON فقط بهذا الشكل.**
"""

    @staticmethod
    def _load_menu_items() -> List[str]:
        """
        تحميل أسماء الأصناف فقط من المنيو (بدون تفاصيل)
        لتقليل التوكنات والتكلفة
        """
        global _menu_items_cache

        if _menu_items_cache is not None:
            return _menu_items_cache

        try:
            menu_file = os.path.join(
                os.path.dirname(os.path.dirname(os.path.dirname(__file__))),
                'data',
                'menu_items_local.json'
            )

            with open(menu_file, 'r', encoding='utf-8') as f:
                menu_data = json.load(f)

            # استخراج الأسماء فقط (عربي + إنجليزي)
            menu_names = []
            for item in menu_data:
                if item.get('is_active', True):
                    menu_names.append(item.get('name', ''))
                    if item.get('name_en'):
                        menu_names.append(item['name_en'])

            _menu_items_cache = menu_names
            logger.info(f"✅ Loaded {len(menu_names)} menu item names for AI Intent Analyzer")
            return menu_names

        except Exception as e:
            logger.error(f"❌ Error loading menu items: {e}")
            return []

    @staticmethod
    async def analyze_intent(
        current_message: str,
        conversation_history: List[Dict[str, str]] = None,
        menu_items: List[str] = None,
        use_menu: bool = True
    ) -> Dict:
        """
        تحليل النية من الرسالة + السياق

        Args:
            current_message: الرسالة الحالية
            conversation_history: آخر 3 محادثات [{"role": "user/assistant", "content": "..."}]
            menu_items: قائمة أسماء الأصناف من المنيو (اختياري)
            use_menu: هل نمرر المنيو للـ AI؟ (افتراضي: True)

        Returns:
            {
                "intent": "greeting_with_order",
                "confidence": 0.95,
                "reasoning": "...",
                "extracted_items": ["برجر كلاسيك"],
                "is_compound": true
            }
        """
        try:
            # Build context
            context_text = ""
            if conversation_history:
                context_text = "\n**آخر 3 محادثات:**\n"
                for msg in conversation_history[-3:]:
                    role = "العميل" if msg.get("role") == "user" else "البوت"
                    context_text += f"{role}: {msg.get('content', '')}\n"

            # Build menu context (only if needed)
            menu_context = ""
            if use_menu:
                if menu_items is None:
                    menu_items = AIIntentAnalyzer._load_menu_items()

                # تمرير أول 50 صنف لتحسين الدقة
                if menu_items:
                    menu_context = f"\n**أمثلة من المنيو (أول 50 صنف):**\n{', '.join(menu_items[:50])}\n"

            # Build user message
            user_message = f"""**الرسالة الحالية:**
{current_message}
{context_text}
{menu_context}

حلل النية وأرجع JSON فقط."""

            logger.info(f"🤖 AI Intent Analyzer - Analyzing: '{current_message[:50]}...'")

            # Get OpenAI client (lazy initialization)
            client = _get_openai_client()

            # Call OpenAI
            response = await client.chat.completions.create(
                model=os.getenv("OPENAI_MODEL", "gpt-4o-mini"),
                messages=[
                    {"role": "system", "content": AIIntentAnalyzer.SYSTEM_PROMPT},
                    {"role": "user", "content": user_message}
                ],
                temperature=0.1,
                max_tokens=300,  # تقليل من 500 إلى 300
                response_format={"type": "json_object"}
            )

            # Parse response
            result = json.loads(response.choices[0].message.content)

            # Log token usage for monitoring
            usage = response.usage
            logger.info(f"✅ AI Intent: {result.get('intent')} (confidence: {result.get('confidence', 0):.2f})")
            logger.info(f"📊 Tokens: {usage.prompt_tokens} input + {usage.completion_tokens} output = {usage.total_tokens} total")
            logger.info(f"📝 Reasoning: {result.get('reasoning', 'N/A')}")

            return result

        except Exception as e:
            logger.error(f"❌ Error in AI Intent Analyzer: {e}")
            return {
                "intent": "other",
                "confidence": 0.0,
                "reasoning": f"Error: {str(e)}",
                "extracted_items": [],
                "is_compound": False
            }
    
    @staticmethod
    def should_use_ai_analyzer(message: str) -> bool:
        """
        تحديد ذكي: هل نحتاج AI لتحليل هذه الرسالة؟

        نستخدم AI للحالات المعقدة والغامضة:
        - تحية + شيء آخر
        - حذف/مسح السلة
        - رسائل تحتوي على أرقام + كلمات (مثل: "محتاج 2 بيتزا")
        - رسائل غامضة

        Returns:
            True إذا كانت الرسالة تحتاج AI
        """
        message_lower = message.lower().strip()
        words = message.split()
        word_count = len(words)

        # كلمات التحية
        greeting_keywords = [
            "السلام عليكم", "السلام", "مرحبا", "صباح الخير",
            "مساء الخير", "هلا", "hello", "hi", "hey"
        ]

        # كلمات الطلب
        order_keywords = ["ممكن", "عايز", "أريد", "اريد", "أبغى", "ابغى", "محتاج", "want", "need", "بدي", "عاوز"]

        # كلمات الحذف
        delete_keywords = ["امسح", "احذف", "الغي", "حذف", "مسح"]
        cart_keywords = ["السلة", "الطلب", "الأوردر", "الكارت"]

        has_greeting = any(kw in message_lower for kw in greeting_keywords)
        has_order = any(kw in message_lower for kw in order_keywords)
        has_delete = any(kw in message_lower for kw in delete_keywords)
        has_cart = any(kw in message_lower for kw in cart_keywords)

        # تحقق من وجود أرقام في الرسالة
        import re
        has_numbers = bool(re.search(r'\d+|[٠-٩]+', message))

        # استخدم AI إذا:
        # 1. تحية + كلمة طلب (مثل: "مرحبا عايز بيتزا")
        if has_greeting and has_order:
            return True

        # 2. تحية + كلام آخر (أكثر من 3 كلمات)
        if has_greeting and word_count > 3:
            return True

        # 3. حذف + سلة
        if has_delete and has_cart:
            return True

        # 4. كلمة طلب + رقم (مثل: "محتاج 2" أو "عايز 3 بيتزا")
        if has_order and has_numbers:
            return True

        # 5. رسالة تحتوي على رقم + كلمات (مثل: "2 بيتزا" أو "تاج 2 بيتزا")
        if has_numbers and word_count >= 2:
            return True

        # 6. كلمة طلب + اسم صنف (مثل: "عايز باستا" أو "محتاج شاورما")
        # استخدم AI للتحقق من وجود الصنف في المنيو
        if has_order and word_count >= 2:
            return True

        # 7. رسالة قصيرة جداً (كلمة واحدة) - قد تكون غامضة
        if word_count == 1 and len(message) < 10:
            return False  # لا تستخدم AI للرسائل القصيرة جداً

        # الحالات الأخرى: لا تستخدم AI
        return False

    @staticmethod
    async def is_greeting_with_order(message: str, conversation_history: List[Dict] = None) -> bool:
        """
        تحقق سريع: هل الرسالة تحتوي على تحية + طلب؟
        """
        result = await AIIntentAnalyzer.analyze_intent(message, conversation_history)
        return result.get("intent") == "greeting_with_order"

    @staticmethod
    async def is_pure_greeting(message: str, conversation_history: List[Dict] = None) -> bool:
        """
        تحقق سريع: هل الرسالة تحية فقط؟
        """
        result = await AIIntentAnalyzer.analyze_intent(message, conversation_history)
        return result.get("intent") == "greeting_only"

    @staticmethod
    async def extract_items_from_message(message: str, menu_items: List[str] = None) -> List[str]:
        """
        استخراج أسماء الأصناف من الرسالة
        """
        result = await AIIntentAnalyzer.analyze_intent(message, menu_items=menu_items)
        return result.get("extracted_items", [])


# Singleton instance
ai_intent_analyzer = AIIntentAnalyzer()

