"""
State Validator - التحقق من صحة الحالة والانتقالات
طبقة حماية لمنع Hallucination والقفز غير المنطقي
"""

from typing import Dict, Optional, Tuple
from datetime import datetime, timedelta
from app.state_machine.order_states import OrderPhase, order_state_machine
from app.utils.logger import app_logger as logger


class StateValidator:
    """مدقق الحالة - يمنع الانتقالات غير الصحيحة"""
    
    def __init__(self):
        self.state_machine = order_state_machine
    
    def validate_transition(
        self,
        current_phase: OrderPhase,
        next_phase: OrderPhase,
        session_data: Dict
    ) -> Tuple[bool, Optional[str]]:
        """
        التحقق من صحة الانتقال
        
        Returns:
            (is_valid, error_message)
        """
        
        # 1. التحقق من أن الانتقال مسموح
        if not self.state_machine.is_valid_transition(current_phase, next_phase):
            valid_phases = self.state_machine.get_valid_next_phases(current_phase)
            valid_names = [p.value for p in valid_phases]
            
            error_msg = (
                f"Invalid transition: {current_phase.value} → {next_phase.value}. "
                f"Valid next phases: {', '.join(valid_names)}"
            )
            
            logger.error(f"❌ {error_msg}")
            return (False, error_msg)
        
        # 2. التحقق من البيانات المطلوبة (فقط للمراحل المتقدمة)
        # ملاحظة: لا نتحقق من البيانات عند الانتقال إلى مرحلة جمع البيانات نفسها
        # مثلاً: يمكن الانتقال إلى ADDRESS_INFO بدون عنوان (لأننا سنطلبه هناك)
        metadata = self.state_machine.get_phase_metadata(next_phase)
        if metadata and next_phase in [OrderPhase.PAYMENT, OrderPhase.CONFIRMATION, OrderPhase.COMPLETED]:
            # فقط للمراحل النهائية نتحقق من البيانات
            missing_data = []
            for required_field in metadata.required_data:
                if required_field not in session_data or not session_data[required_field]:
                    missing_data.append(required_field)

            if missing_data:
                error_msg = (
                    f"Cannot transition to {next_phase.value}: "
                    f"Missing required data: {', '.join(missing_data)}"
                )
                logger.warning(f"⚠️ {error_msg}")
                return (False, error_msg)
        
        # 3. التحقق من الحد الأدنى للوقت في المرحلة
        if metadata and metadata.min_duration_seconds > 0:
            phase_start_time = session_data.get("phase_start_time")
            if phase_start_time:
                try:
                    start_dt = datetime.fromisoformat(phase_start_time)
                    elapsed = (datetime.now() - start_dt).total_seconds()
                    
                    if elapsed < metadata.min_duration_seconds:
                        error_msg = (
                            f"Too fast transition to {next_phase.value}: "
                            f"Minimum {metadata.min_duration_seconds}s required, "
                            f"only {elapsed:.1f}s elapsed"
                        )
                        logger.warning(f"⚠️ {error_msg}")
                        # لا نمنع، فقط تحذير
                except Exception as e:
                    logger.error(f"Error checking phase duration: {e}")
        
        # ✅ الانتقال صحيح
        logger.info(f"✅ Valid transition: {current_phase.value} → {next_phase.value}")
        return (True, None)
    
    def validate_action(
        self,
        phase: OrderPhase,
        action: str
    ) -> Tuple[bool, Optional[str]]:
        """
        التحقق من أن الإجراء مسموح في هذه المرحلة
        
        Returns:
            (is_valid, error_message)
        """
        
        if not self.state_machine.is_action_allowed(phase, action):
            metadata = self.state_machine.get_phase_metadata(phase)
            allowed_actions = metadata.allowed_actions if metadata else []
            
            error_msg = (
                f"Action '{action}' not allowed in phase '{phase.value}'. "
                f"Allowed actions: {', '.join(allowed_actions)}"
            )
            
            logger.error(f"❌ {error_msg}")
            return (False, error_msg)
        
        logger.info(f"✅ Action '{action}' is valid for phase '{phase.value}'")
        return (True, None)
    
    def get_correction_message(
        self,
        current_phase: OrderPhase,
        attempted_phase: OrderPhase,
        lang: str = "ar"
    ) -> str:
        """
        الحصول على رسالة تصحيح مهذبة
        
        Args:
            current_phase: المرحلة الحالية
            attempted_phase: المرحلة التي حاول المستخدم الانتقال إليها
            lang: اللغة
            
        Returns:
            رسالة تصحيح مهذبة
        """
        
        current_meta = self.state_machine.get_phase_metadata(current_phase)
        attempted_meta = self.state_machine.get_phase_metadata(attempted_phase)
        
        if lang == "ar":
            if attempted_phase == OrderPhase.ADDRESS_INFO:
                return (
                    "عذراً، يجب عليك اختيار الأصناف أولاً قبل إدخال عنوان التوصيل. 🛒\n\n"
                    "ما الذي تريد طلبه؟"
                )
            
            elif attempted_phase == OrderPhase.PAYMENT:
                return (
                    "عذراً، يجب عليك إدخال عنوان التوصيل أولاً قبل اختيار طريقة الدفع. 📍\n\n"
                    "من فضلك أرسل عنوانك أو موقعك."
                )
            
            elif attempted_phase == OrderPhase.CONFIRMATION:
                return (
                    "عذراً، يجب إكمال جميع الخطوات قبل تأكيد الطلب:\n"
                    "1. اختيار الأصناف ✅\n"
                    "2. إدخال العنوان 📍\n"
                    "3. اختيار طريقة الدفع 💳\n\n"
                    f"أنت الآن في مرحلة: {current_meta.name_ar if current_meta else current_phase.value}"
                )
            
            else:
                return (
                    f"عذراً، لا يمكن الانتقال إلى '{attempted_meta.name_ar if attempted_meta else attempted_phase.value}' "
                    f"من '{current_meta.name_ar if current_meta else current_phase.value}'. 🚫\n\n"
                    "دعنا نكمل الخطوة الحالية أولاً."
                )
        
        else:  # English
            if attempted_phase == OrderPhase.ADDRESS_INFO:
                return (
                    "Sorry, you need to choose items first before providing delivery address. 🛒\n\n"
                    "What would you like to order?"
                )
            
            elif attempted_phase == OrderPhase.PAYMENT:
                return (
                    "Sorry, you need to provide delivery address first before choosing payment method. 📍\n\n"
                    "Please send your address or location."
                )
            
            elif attempted_phase == OrderPhase.CONFIRMATION:
                return (
                    "Sorry, you need to complete all steps before confirming:\n"
                    "1. Choose items ✅\n"
                    "2. Provide address 📍\n"
                    "3. Choose payment method 💳\n\n"
                    f"You are currently at: {current_meta.name_en if current_meta else current_phase.value}"
                )
            
            else:
                return (
                    f"Sorry, cannot transition to '{attempted_meta.name_en if attempted_meta else attempted_phase.value}' "
                    f"from '{current_meta.name_en if current_meta else current_phase.value}'. 🚫\n\n"
                    "Let's complete the current step first."
                )
    
    def auto_recover_phase(
        self,
        current_phase: OrderPhase,
        user_intent: str,
        session_data: Dict
    ) -> Tuple[OrderPhase, bool, Optional[str]]:
        """
        استرجاع تلقائي للمرحلة الصحيحة
        
        Returns:
            (recovered_phase, was_recovered, recovery_message)
        """
        
        recovered_phase = self.state_machine.get_recovery_phase(
            current_phase,
            user_intent,
            session_data
        )
        
        if recovered_phase != current_phase:
            logger.warning(
                f"🔄 Auto-recovery: {current_phase.value} → {recovered_phase.value} "
                f"(user intent: {user_intent})"
            )
            
            recovery_msg = (
                f"Automatically recovered from {current_phase.value} to {recovered_phase.value} "
                f"based on user intent: {user_intent}"
            )
            
            return (recovered_phase, True, recovery_msg)
        
        return (current_phase, False, None)
    
    def validate_llm_response(
        self,
        current_phase: OrderPhase,
        llm_response: Dict,
        session_data: Dict
    ) -> Tuple[bool, Optional[str]]:
        """
        التحقق من رد الـ LLM - طبقة الحماية
        
        Args:
            current_phase: المرحلة الحالية
            llm_response: رد الـ LLM
            session_data: بيانات الجلسة
            
        Returns:
            (is_valid, error_message)
        """
        
        # 1. التحقق من الإجراء المطلوب
        action = llm_response.get("action")
        if action:
            is_valid, error_msg = self.validate_action(current_phase, action)
            if not is_valid:
                logger.error(
                    f"❌ LLM hallucination detected: Action '{action}' not allowed in phase '{current_phase.value}'"
                )
                return (False, f"LLM suggested invalid action: {error_msg}")
        
        # 2. التحقق من المرحلة المقترحة
        suggested_phase = llm_response.get("next_phase")
        if suggested_phase:
            try:
                next_phase = OrderPhase(suggested_phase)

                # فقط نتحقق من الانتقال نفسه، لا من البيانات المطلوبة
                # (لأن validate_transition يتحقق من البيانات أيضاً)
                if not self.state_machine.is_valid_transition(current_phase, next_phase):
                    valid_phases = self.state_machine.get_valid_next_phases(current_phase)
                    valid_names = [p.value for p in valid_phases]
                    error_msg = (
                        f"Invalid transition: {current_phase.value} → {next_phase.value}. "
                        f"Valid next phases: {', '.join(valid_names)}"
                    )
                    logger.error(
                        f"❌ LLM hallucination detected: {error_msg}"
                    )
                    return (False, f"LLM suggested invalid transition: {error_msg}")

            except ValueError:
                logger.error(f"❌ LLM suggested unknown phase: {suggested_phase}")
                return (False, f"LLM suggested unknown phase: {suggested_phase}")
        
        # 3. التحقق من البيانات المطلوبة
        if action in ["confirm_order", "proceed_to_payment"]:
            metadata = self.state_machine.get_phase_metadata(current_phase)
            if metadata:
                missing_data = []
                for required_field in metadata.required_data:
                    if required_field not in session_data or not session_data[required_field]:
                        missing_data.append(required_field)
                
                if missing_data:
                    logger.error(
                        f"❌ LLM hallucination detected: Trying to proceed without required data: {missing_data}"
                    )
                    return (False, f"Missing required data: {', '.join(missing_data)}")
        
        # ✅ رد الـ LLM صحيح
        logger.info(f"✅ LLM response validated for phase '{current_phase.value}'")
        return (True, None)


# Singleton instance
state_validator = StateValidator()

