"""
Order State Machine - آلة الحالة لمراحل الطلب
يمنع Hallucination والقفز غير المنطقي بين المراحل
"""

from enum import Enum
from typing import Dict, List, Optional, Set
from dataclasses import dataclass
from app.utils.logger import app_logger as logger


class OrderPhase(str, Enum):
    """مراحل الطلب المحددة"""
    
    # المرحلة 0: البداية
    START = "start"
    
    # المرحلة 1: اختيار الأصناف
    CHOOSE_ITEM = "choose_item"
    
    # المرحلة 2: تأكيد الكمية
    CONFIRM_QUANTITY = "confirm_quantity"
    
    # المرحلة 3: إضافة ملاحظات/إضافات
    ADD_EXTRAS = "add_extras"
    
    # المرحلة 4: معلومات التوصيل
    ADDRESS_INFO = "address_info"
    
    # المرحلة 5: الدفع
    PAYMENT = "payment"
    
    # المرحلة 6: التأكيد النهائي
    CONFIRMATION = "confirmation"
    
    # المرحلة 7: مكتمل
    COMPLETED = "completed"
    
    # حالات خاصة
    CANCELLED = "cancelled"
    ERROR = "error"


@dataclass
class PhaseMetadata:
    """معلومات عن كل مرحلة"""
    phase: OrderPhase
    name_ar: str
    name_en: str
    description_ar: str
    description_en: str
    required_data: List[str]  # البيانات المطلوبة في هذه المرحلة
    allowed_actions: List[str]  # الإجراءات المسموحة
    min_duration_seconds: int = 0  # الحد الأدنى للوقت في هذه المرحلة


class OrderStateMachine:
    """آلة الحالة لإدارة مراحل الطلب"""
    
    # خريطة الانتقالات الصحيحة
    VALID_TRANSITIONS: Dict[OrderPhase, Set[OrderPhase]] = {
        OrderPhase.START: {
            OrderPhase.CHOOSE_ITEM,
            OrderPhase.CANCELLED
        },
        
        OrderPhase.CHOOSE_ITEM: {
            OrderPhase.CONFIRM_QUANTITY,
            OrderPhase.CHOOSE_ITEM,  # يمكن إضافة المزيد
            OrderPhase.ADD_EXTRAS,
            OrderPhase.ADDRESS_INFO,  # إذا كان لديه أصناف بالفعل
            OrderPhase.CANCELLED
        },
        
        OrderPhase.CONFIRM_QUANTITY: {
            OrderPhase.ADD_EXTRAS,
            OrderPhase.CHOOSE_ITEM,  # العودة لإضافة المزيد
            OrderPhase.ADDRESS_INFO,  # الانتقال للتوصيل
            OrderPhase.CANCELLED
        },
        
        OrderPhase.ADD_EXTRAS: {
            OrderPhase.CHOOSE_ITEM,  # إضافة المزيد
            OrderPhase.ADDRESS_INFO,  # الانتقال للتوصيل
            OrderPhase.CANCELLED
        },
        
        OrderPhase.ADDRESS_INFO: {
            OrderPhase.PAYMENT,
            OrderPhase.CHOOSE_ITEM,  # العودة لتعديل الطلب
            OrderPhase.CANCELLED
        },
        
        OrderPhase.PAYMENT: {
            OrderPhase.CONFIRMATION,
            OrderPhase.ADDRESS_INFO,  # العودة لتعديل العنوان
            OrderPhase.CANCELLED
        },
        
        OrderPhase.CONFIRMATION: {
            OrderPhase.COMPLETED,
            OrderPhase.PAYMENT,  # العودة لتغيير طريقة الدفع
            OrderPhase.CANCELLED
        },
        
        OrderPhase.COMPLETED: {
            OrderPhase.START  # طلب جديد
        },
        
        OrderPhase.CANCELLED: {
            OrderPhase.START  # طلب جديد
        },
        
        OrderPhase.ERROR: {
            OrderPhase.START  # إعادة البداية
        }
    }
    
    # معلومات كل مرحلة
    PHASE_METADATA: Dict[OrderPhase, PhaseMetadata] = {
        OrderPhase.START: PhaseMetadata(
            phase=OrderPhase.START,
            name_ar="البداية",
            name_en="Start",
            description_ar="بداية المحادثة",
            description_en="Start of conversation",
            required_data=[],
            allowed_actions=["greet", "show_menu", "ask_menu"]
        ),
        
        OrderPhase.CHOOSE_ITEM: PhaseMetadata(
            phase=OrderPhase.CHOOSE_ITEM,
            name_ar="اختيار الأصناف",
            name_en="Choose Items",
            description_ar="العميل يختار الأصناف من المنيو",
            description_en="Customer chooses items from menu",
            required_data=["items"],
            allowed_actions=["add_item", "search_item", "show_menu", "view_cart"]
        ),
        
        OrderPhase.CONFIRM_QUANTITY: PhaseMetadata(
            phase=OrderPhase.CONFIRM_QUANTITY,
            name_ar="تأكيد الكمية",
            name_en="Confirm Quantity",
            description_ar="تأكيد كمية الأصناف",
            description_en="Confirm item quantities",
            required_data=["items", "quantities"],
            allowed_actions=["update_quantity", "confirm_quantity", "view_cart"]
        ),
        
        OrderPhase.ADD_EXTRAS: PhaseMetadata(
            phase=OrderPhase.ADD_EXTRAS,
            name_ar="إضافة ملاحظات",
            name_en="Add Extras",
            description_ar="إضافة ملاحظات أو طلبات خاصة",
            description_en="Add notes or special requests",
            required_data=["items"],
            allowed_actions=["add_note", "add_extras", "skip_extras"]
        ),
        
        OrderPhase.ADDRESS_INFO: PhaseMetadata(
            phase=OrderPhase.ADDRESS_INFO,
            name_ar="معلومات التوصيل",
            name_en="Delivery Info",
            description_ar="إدخال عنوان التوصيل",
            description_en="Enter delivery address",
            required_data=["items", "address"],
            allowed_actions=[
                "provide_address",
                "share_location",
                "send_location",
                "add_location_notes",
                "edit_address",
                "skip_location_notes"
            ],
            min_duration_seconds=5
        ),
        
        OrderPhase.PAYMENT: PhaseMetadata(
            phase=OrderPhase.PAYMENT,
            name_ar="الدفع",
            name_en="Payment",
            description_ar="اختيار طريقة الدفع",
            description_en="Choose payment method",
            required_data=["items", "address", "payment_method"],
            allowed_actions=["choose_payment", "confirm_payment"],
            min_duration_seconds=3
        ),
        
        OrderPhase.CONFIRMATION: PhaseMetadata(
            phase=OrderPhase.CONFIRMATION,
            name_ar="التأكيد النهائي",
            name_en="Final Confirmation",
            description_ar="مراجعة وتأكيد الطلب",
            description_en="Review and confirm order",
            required_data=["items", "address", "payment_method"],
            allowed_actions=["confirm_order", "edit_order", "cancel_order"]
        ),
        
        OrderPhase.COMPLETED: PhaseMetadata(
            phase=OrderPhase.COMPLETED,
            name_ar="مكتمل",
            name_en="Completed",
            description_ar="الطلب تم بنجاح",
            description_en="Order completed successfully",
            required_data=["order_id"],
            allowed_actions=["track_order", "new_order"]
        )
    }
    
    def __init__(self):
        """تهيئة آلة الحالة"""
        pass
    
    def is_valid_transition(
        self,
        current_phase: OrderPhase,
        next_phase: OrderPhase
    ) -> bool:
        """
        التحقق من صحة الانتقال بين المراحل
        
        Args:
            current_phase: المرحلة الحالية
            next_phase: المرحلة المطلوبة
            
        Returns:
            True إذا كان الانتقال صحيح
        """
        if current_phase not in self.VALID_TRANSITIONS:
            logger.error(f"❌ Unknown current phase: {current_phase}")
            return False
        
        valid_next_phases = self.VALID_TRANSITIONS[current_phase]
        is_valid = next_phase in valid_next_phases
        
        if not is_valid:
            logger.warning(
                f"⚠️ Invalid transition: {current_phase.value} → {next_phase.value}"
            )
        
        return is_valid
    
    def get_valid_next_phases(
        self,
        current_phase: OrderPhase
    ) -> Set[OrderPhase]:
        """الحصول على المراحل الصحيحة التالية"""
        return self.VALID_TRANSITIONS.get(current_phase, set())
    
    def get_phase_metadata(
        self,
        phase: OrderPhase
    ) -> Optional[PhaseMetadata]:
        """الحصول على معلومات المرحلة"""
        return self.PHASE_METADATA.get(phase)
    
    def is_action_allowed(
        self,
        phase: OrderPhase,
        action: str
    ) -> bool:
        """التحقق من أن الإجراء مسموح في هذه المرحلة"""
        metadata = self.get_phase_metadata(phase)
        if not metadata:
            return False
        
        return action in metadata.allowed_actions
    
    def get_recovery_phase(
        self,
        current_phase: OrderPhase,
        user_intent: str,
        session_data: Dict
    ) -> OrderPhase:
        """
        تحديد المرحلة الصحيحة بناءً على نية المستخدم
        (Recovery Logic)
        
        Args:
            current_phase: المرحلة الحالية
            user_intent: نية المستخدم
            session_data: بيانات الجلسة
            
        Returns:
            المرحلة الصحيحة
        """
        
        # إذا ذكر المستخدم "عنوان" أو "توصيل"
        if user_intent in ["provide_address", "share_location"]:
            # تحقق من وجود أصناف
            if session_data.get("items") and len(session_data["items"]) > 0:
                return OrderPhase.ADDRESS_INFO
            else:
                logger.warning("⚠️ User mentioned address but no items in cart")
                return OrderPhase.CHOOSE_ITEM
        
        # إذا ذكر المستخدم "دفع" أو "payment"
        if user_intent in ["choose_payment", "confirm_payment"]:
            # تحقق من وجود عنوان
            if session_data.get("address"):
                return OrderPhase.PAYMENT
            else:
                logger.warning("⚠️ User mentioned payment but no address provided")
                return OrderPhase.ADDRESS_INFO
        
        # إذا ذكر المستخدم "تأكيد" أو "confirm"
        if user_intent in ["confirm_order"]:
            # تحقق من وجود كل البيانات
            if (session_data.get("items") and 
                session_data.get("address") and 
                session_data.get("payment_method")):
                return OrderPhase.CONFIRMATION
            else:
                logger.warning("⚠️ User wants to confirm but missing data")
                # إرجاع المرحلة المناسبة
                if not session_data.get("items"):
                    return OrderPhase.CHOOSE_ITEM
                elif not session_data.get("address"):
                    return OrderPhase.ADDRESS_INFO
                else:
                    return OrderPhase.PAYMENT
        
        # إذا ذكر المستخدم "طلب" أو "order"
        if user_intent in ["add_item", "order_food"]:
            return OrderPhase.CHOOSE_ITEM
        
        # الافتراضي: البقاء في المرحلة الحالية
        return current_phase


class OrderFlowState(str, Enum):
    """
    حالات فلو الطلب الديناميكي
    تستخدم لتتبع حالة المستخدم أثناء عملية الطلب
    """

    # حالات الطلب
    IDLE = "idle"                           # خامل (لا يوجد طلب نشط)
    SEARCHING_ITEM = "searching_item"       # يبحث عن صنف
    ITEM_FOUND = "item_found"               # تم العثور على صنف
    AWAITING_QUANTITY = "awaiting_quantity" # في انتظار الكمية
    AWAITING_CONFIRMATION = "awaiting_confirmation"  # في انتظار التأكيد
    ITEM_ADDED = "item_added"               # تمت إضافة الصنف

    # حالات إضافية
    MULTIPLE_ITEMS_FOUND = "multiple_items_found"  # عدة أصناف متطابقة
    NO_ITEM_FOUND = "no_item_found"         # لم يتم العثور على صنف
    ERROR = "error"                         # خطأ


class OrderFlowStateManager:
    """
    مدير حالات فلو الطلب الديناميكي
    يتحكم في الانتقالات بين الحالات أثناء عملية الطلب
    """

    # خريطة الانتقالات الصحيحة
    VALID_TRANSITIONS: Dict[OrderFlowState, List[OrderFlowState]] = {
        OrderFlowState.IDLE: [
            OrderFlowState.SEARCHING_ITEM,
            OrderFlowState.ERROR
        ],

        OrderFlowState.SEARCHING_ITEM: [
            OrderFlowState.ITEM_FOUND,
            OrderFlowState.MULTIPLE_ITEMS_FOUND,
            OrderFlowState.NO_ITEM_FOUND,
            OrderFlowState.ERROR
        ],

        OrderFlowState.ITEM_FOUND: [
            OrderFlowState.AWAITING_QUANTITY,
            OrderFlowState.AWAITING_CONFIRMATION,
            OrderFlowState.ITEM_ADDED,
            OrderFlowState.IDLE
        ],

        OrderFlowState.MULTIPLE_ITEMS_FOUND: [
            OrderFlowState.ITEM_FOUND,
            OrderFlowState.SEARCHING_ITEM,
            OrderFlowState.IDLE
        ],

        OrderFlowState.NO_ITEM_FOUND: [
            OrderFlowState.SEARCHING_ITEM,
            OrderFlowState.IDLE
        ],

        OrderFlowState.AWAITING_QUANTITY: [
            OrderFlowState.AWAITING_CONFIRMATION,
            OrderFlowState.ITEM_ADDED,
            OrderFlowState.IDLE
        ],

        OrderFlowState.AWAITING_CONFIRMATION: [
            OrderFlowState.ITEM_ADDED,
            OrderFlowState.AWAITING_QUANTITY,
            OrderFlowState.IDLE
        ],

        OrderFlowState.ITEM_ADDED: [
            OrderFlowState.SEARCHING_ITEM,
            OrderFlowState.IDLE
        ],

        OrderFlowState.ERROR: [
            OrderFlowState.IDLE,
            OrderFlowState.SEARCHING_ITEM
        ]
    }

    def is_valid_transition(
        self,
        current_state: OrderFlowState,
        new_state: OrderFlowState
    ) -> bool:
        """
        التحقق من صحة الانتقال بين الحالات

        Args:
            current_state: الحالة الحالية
            new_state: الحالة الجديدة

        Returns:
            True إذا كان الانتقال صحيحاً
        """
        if current_state not in self.VALID_TRANSITIONS:
            logger.warning(f"⚠️ Unknown current state: {current_state}")
            return False

        valid_next_states = self.VALID_TRANSITIONS[current_state]

        if new_state not in valid_next_states:
            logger.warning(
                f"⚠️ Invalid transition: {current_state.value} → {new_state.value}"
            )
            return False

        return True

    def get_valid_next_states(
        self,
        current_state: OrderFlowState
    ) -> List[OrderFlowState]:
        """
        الحصول على الحالات الصحيحة التالية

        Args:
            current_state: الحالة الحالية

        Returns:
            قائمة الحالات الصحيحة التالية
        """
        return self.VALID_TRANSITIONS.get(current_state, [])


# Singleton instances
order_state_machine = OrderStateMachine()
order_flow_state_manager = OrderFlowStateManager()

