"""
Order State Manager - نظام إدارة حالات الطلبات
يتحكم في دورة حياة الطلب من البداية حتى التسليم
"""

from enum import Enum
from typing import Dict, Optional, List, Tuple
from datetime import datetime
from app.utils.logger import app_logger as logger


class OrderState(str, Enum):
    """حالات الطلب"""
    CART = "cart"                          # في السلة (لم يتم التأكيد بعد)
    PENDING = "pending"                    # معلق (في انتظار التأكيد)
    CONFIRMED = "confirmed"                # مؤكد (تم قبول الطلب)
    PREPARING = "preparing"                # قيد التحضير
    READY = "ready"                        # جاهز للتوصيل
    OUT_FOR_DELIVERY = "out_for_delivery"  # في الطريق
    DELIVERED = "delivered"                # تم التسليم
    CANCELLED = "cancelled"                # ملغي


class OrderStateManager:
    """مدير حالات الطلبات"""
    
    # خريطة الانتقالات الصحيحة
    VALID_TRANSITIONS: Dict[OrderState, List[OrderState]] = {
        OrderState.CART: [
            OrderState.PENDING,      # من السلة → معلق
            OrderState.CANCELLED     # من السلة → ملغي
        ],
        OrderState.PENDING: [
            OrderState.CONFIRMED,    # من معلق → مؤكد
            OrderState.CANCELLED     # من معلق → ملغي
        ],
        OrderState.CONFIRMED: [
            OrderState.PREPARING,    # من مؤكد → قيد التحضير
            OrderState.CANCELLED     # من مؤكد → ملغي
        ],
        OrderState.PREPARING: [
            OrderState.READY,        # من قيد التحضير → جاهز
            OrderState.CANCELLED     # من قيد التحضير → ملغي
        ],
        OrderState.READY: [
            OrderState.OUT_FOR_DELIVERY,  # من جاهز → في الطريق
            OrderState.CANCELLED          # من جاهز → ملغي
        ],
        OrderState.OUT_FOR_DELIVERY: [
            OrderState.DELIVERED,    # من في الطريق → تم التسليم
            OrderState.CANCELLED     # من في الطريق → ملغي (نادر)
        ],
        OrderState.DELIVERED: [],    # لا يمكن الانتقال من تم التسليم
        OrderState.CANCELLED: []     # لا يمكن الانتقال من ملغي
    }
    
    # الرسائل العربية لكل حالة
    STATE_MESSAGES_AR: Dict[OrderState, str] = {
        OrderState.CART: "🛒 الطلب في السلة",
        OrderState.PENDING: "⏳ الطلب معلق في انتظار التأكيد",
        OrderState.CONFIRMED: "✅ تم تأكيد طلبك بنجاح!",
        OrderState.PREPARING: "👨‍🍳 نجهز طلبك الآن 🍕",
        OrderState.READY: "✅ طلبك جاهز للتوصيل 📦",
        OrderState.OUT_FOR_DELIVERY: "🚗 طلبك في الطريق إليك!",
        OrderState.DELIVERED: "🎉 تم تسليم طلبك بنجاح!",
        OrderState.CANCELLED: "❌ تم إلغاء الطلب"
    }
    
    # الرسائل الإنجليزية لكل حالة
    STATE_MESSAGES_EN: Dict[OrderState, str] = {
        OrderState.CART: "🛒 Order in cart",
        OrderState.PENDING: "⏳ Order pending confirmation",
        OrderState.CONFIRMED: "✅ Your order has been confirmed!",
        OrderState.PREPARING: "👨‍🍳 We're preparing your order 🍕",
        OrderState.READY: "✅ Your order is ready for delivery 📦",
        OrderState.OUT_FOR_DELIVERY: "🚗 Your order is on the way!",
        OrderState.DELIVERED: "🎉 Your order has been delivered!",
        OrderState.CANCELLED: "❌ Order cancelled"
    }
    
    # الرسائل التفصيلية لكل حالة (عربي)
    STATE_DETAILS_AR: Dict[OrderState, str] = {
        OrderState.CART: """🛒 **الطلب في السلة**

يمكنك الآن:
• إضافة المزيد من الأصناف
• تعديل الكميات
• تأكيد الطلب والمتابعة للدفع""",
        
        OrderState.PENDING: """⏳ **الطلب معلق**

طلبك في انتظار التأكيد من المطعم.
سنخبرك فور قبول الطلب.""",
        
        OrderState.CONFIRMED: """✅ **تم تأكيد طلبك!**

رقم الطلب: #{order_number}
الوقت المتوقع: 30-45 دقيقة

سنبدأ بتحضير طلبك الآن 👨‍🍳""",
        
        OrderState.PREPARING: """👨‍🍳 **نجهز طلبك الآن**

طلبك قيد التحضير في المطبخ 🍕
الوقت المتبقي: ~20-30 دقيقة

سنخبرك عندما يكون جاهزاً!""",
        
        OrderState.READY: """✅ **طلبك جاهز!**

تم تحضير طلبك بنجاح 📦
سيتم توصيله إليك قريباً

الوقت المتوقع للتوصيل: 15-20 دقيقة""",
        
        OrderState.OUT_FOR_DELIVERY: """🚗 **طلبك في الطريق!**

السائق في طريقه إليك الآن
الوقت المتوقع: 10-15 دقيقة

📍 يمكنك تتبع الطلب من خلال الرابط""",
        
        OrderState.DELIVERED: """🎉 **تم تسليم طلبك!**

نتمنى أن تستمتع بوجبتك! 😋

💬 نسعد بتقييمك للطلب
⭐ اكتب "تقييم" لتقييم الخدمة""",
        
        OrderState.CANCELLED: """❌ **تم إلغاء الطلب**

تم إلغاء طلبك بنجاح.

💡 يمكنك تقديم طلب جديد في أي وقت!
اكتب "المنيو" لعرض القائمة"""
    }
    
    # الرسائل التفصيلية لكل حالة (إنجليزي)
    STATE_DETAILS_EN: Dict[OrderState, str] = {
        OrderState.CART: """🛒 **Order in Cart**

You can now:
• Add more items
• Modify quantities
• Confirm order and proceed to payment""",
        
        OrderState.PENDING: """⏳ **Order Pending**

Your order is awaiting confirmation from the restaurant.
We'll notify you once it's accepted.""",
        
        OrderState.CONFIRMED: """✅ **Order Confirmed!**

Order Number: #{order_number}
Estimated Time: 30-45 minutes

We'll start preparing your order now 👨‍🍳""",
        
        OrderState.PREPARING: """👨‍🍳 **Preparing Your Order**

Your order is being prepared in the kitchen 🍕
Time Remaining: ~20-30 minutes

We'll notify you when it's ready!""",
        
        OrderState.READY: """✅ **Order Ready!**

Your order has been prepared successfully 📦
It will be delivered to you soon

Estimated Delivery: 15-20 minutes""",
        
        OrderState.OUT_FOR_DELIVERY: """🚗 **Order On The Way!**

The driver is on the way to you now
Estimated Time: 10-15 minutes

📍 You can track your order via the link""",
        
        OrderState.DELIVERED: """🎉 **Order Delivered!**

We hope you enjoy your meal! 😋

💬 We'd love your feedback
⭐ Type "rate" to rate the service""",
        
        OrderState.CANCELLED: """❌ **Order Cancelled**

Your order has been cancelled successfully.

💡 You can place a new order anytime!
Type "menu" to view the menu"""
    }
    
    def __init__(self, supabase_client=None):
        """
        تهيئة مدير حالات الطلبات
        
        Args:
            supabase_client: عميل Supabase (اختياري)
        """
        self.supabase = supabase_client
    
    def is_valid_transition(
        self,
        current_state: OrderState,
        new_state: OrderState
    ) -> bool:
        """
        التحقق من صحة الانتقال بين الحالات
        
        Args:
            current_state: الحالة الحالية
            new_state: الحالة الجديدة
            
        Returns:
            True إذا كان الانتقال صحيحاً
        """
        if current_state not in self.VALID_TRANSITIONS:
            return False
        
        return new_state in self.VALID_TRANSITIONS[current_state]
    
    def get_valid_next_states(
        self,
        current_state: OrderState
    ) -> List[OrderState]:
        """
        الحصول على الحالات الصحيحة التالية
        
        Args:
            current_state: الحالة الحالية
            
        Returns:
            قائمة الحالات الصحيحة التالية
        """
        return self.VALID_TRANSITIONS.get(current_state, [])
    
    def get_state_message(
        self,
        state: OrderState,
        lang: str = 'ar',
        detailed: bool = False,
        order_number: Optional[str] = None
    ) -> str:
        """
        الحصول على رسالة الحالة

        Args:
            state: الحالة
            lang: اللغة ('ar' أو 'en')
            detailed: رسالة تفصيلية أم مختصرة
            order_number: رقم الطلب (للرسائل التفصيلية)

        Returns:
            رسالة الحالة
        """
        if detailed:
            messages = self.STATE_DETAILS_AR if lang == 'ar' else self.STATE_DETAILS_EN
            message = messages.get(state, "")

            # استبدال رقم الطلب إذا كان موجوداً
            if order_number:
                message = message.replace("{order_number}", order_number)

            return message
        else:
            messages = self.STATE_MESSAGES_AR if lang == 'ar' else self.STATE_MESSAGES_EN
            return messages.get(state, "")

    async def change_order_state(
        self,
        order_id: str,
        new_state: OrderState,
        lang: str = 'ar'
    ) -> Dict:
        """
        تغيير حالة الطلب مع التحقق من صحة الانتقال

        Args:
            order_id: معرف الطلب
            new_state: الحالة الجديدة
            lang: اللغة

        Returns:
            dict مع النتيجة والرسالة
        """
        try:
            # 1. الحصول على الطلب الحالي
            if not self.supabase:
                logger.error("❌ Supabase client not initialized")
                return {
                    'success': False,
                    'error': 'database_not_initialized',
                    'message_ar': 'خطأ في الاتصال بقاعدة البيانات',
                    'message_en': 'Database connection error'
                }

            order_response = self.supabase.table('orders') \
                .select('id, order_id, status') \
                .eq('id', order_id) \
                .single() \
                .execute()

            if not order_response.data:
                logger.warning(f"⚠️ Order not found: {order_id}")
                return {
                    'success': False,
                    'error': 'order_not_found',
                    'message_ar': 'الطلب غير موجود',
                    'message_en': 'Order not found'
                }

            order = order_response.data
            current_state_str = order.get('status', 'cart')
            order_number = order.get('order_id', 'N/A')

            # 2. تحويل الحالة الحالية إلى OrderState
            try:
                current_state = OrderState(current_state_str)
            except ValueError:
                logger.error(f"❌ Invalid current state: {current_state_str}")
                current_state = OrderState.CART

            # 3. التحقق من صحة الانتقال
            if not self.is_valid_transition(current_state, new_state):
                valid_states = self.get_valid_next_states(current_state)
                valid_states_str = ', '.join([s.value for s in valid_states])

                logger.warning(
                    f"⚠️ Invalid transition: {current_state.value} → {new_state.value}"
                )

                if lang == 'ar':
                    message = f"""❌ **انتقال غير صحيح**

لا يمكن الانتقال من "{current_state.value}" إلى "{new_state.value}"

الحالات الصحيحة التالية: {valid_states_str}"""
                else:
                    message = f"""❌ **Invalid Transition**

Cannot transition from "{current_state.value}" to "{new_state.value}"

Valid next states: {valid_states_str}"""

                return {
                    'success': False,
                    'error': 'invalid_transition',
                    'current_state': current_state.value,
                    'attempted_state': new_state.value,
                    'valid_next_states': [s.value for s in valid_states],
                    'message_ar': message if lang == 'ar' else None,
                    'message_en': message if lang == 'en' else None
                }

            # 4. تحديث الحالة في قاعدة البيانات
            update_data = {
                'status': new_state.value,
                'updated_at': datetime.utcnow().isoformat()
            }

            # إضافة timestamps خاصة بكل حالة
            if new_state == OrderState.CONFIRMED:
                update_data['confirmed_at'] = datetime.utcnow().isoformat()
            elif new_state == OrderState.PREPARING:
                update_data['preparing_at'] = datetime.utcnow().isoformat()
            elif new_state == OrderState.READY:
                update_data['ready_at'] = datetime.utcnow().isoformat()
            elif new_state == OrderState.OUT_FOR_DELIVERY:
                update_data['out_for_delivery_at'] = datetime.utcnow().isoformat()
            elif new_state == OrderState.DELIVERED:
                update_data['delivered_at'] = datetime.utcnow().isoformat()
            elif new_state == OrderState.CANCELLED:
                update_data['cancelled_at'] = datetime.utcnow().isoformat()

            self.supabase.table('orders') \
                .update(update_data) \
                .eq('id', order_id) \
                .execute()

            logger.info(
                f"✅ Order state changed: {order_number} | "
                f"{current_state.value} → {new_state.value}"
            )

            # 5. الحصول على الرسالة المناسبة
            message = self.get_state_message(
                new_state,
                lang=lang,
                detailed=True,
                order_number=order_number
            )

            return {
                'success': True,
                'order_id': order_id,
                'order_number': order_number,
                'previous_state': current_state.value,
                'new_state': new_state.value,
                'message_ar': message if lang == 'ar' else self.get_state_message(new_state, 'ar', True, order_number),
                'message_en': message if lang == 'en' else self.get_state_message(new_state, 'en', True, order_number),
                'timestamp': datetime.utcnow().isoformat()
            }

        except Exception as e:
            logger.error(f"❌ Error changing order state: {str(e)}")
            return {
                'success': False,
                'error': 'exception',
                'message_ar': f'حدث خطأ: {str(e)}',
                'message_en': f'Error: {str(e)}'
            }

    def get_state_color(self, state: OrderState) -> str:
        """
        الحصول على لون الحالة (للواجهات)

        Args:
            state: الحالة

        Returns:
            كود اللون
        """
        colors = {
            OrderState.CART: "#9E9E9E",           # رمادي
            OrderState.PENDING: "#FF9800",        # برتقالي
            OrderState.CONFIRMED: "#4CAF50",      # أخضر
            OrderState.PREPARING: "#2196F3",      # أزرق
            OrderState.READY: "#8BC34A",          # أخضر فاتح
            OrderState.OUT_FOR_DELIVERY: "#00BCD4", # سماوي
            OrderState.DELIVERED: "#4CAF50",      # أخضر
            OrderState.CANCELLED: "#F44336"       # أحمر
        }
        return colors.get(state, "#9E9E9E")


# Singleton instance
order_state_manager = OrderStateManager()

