"""
Word Correction Dictionary
===========================
Maps common misspellings and variations to correct menu item names.
Helps improve search accuracy for user queries.
"""

from typing import Dict, List, Optional
from loguru import logger


class WordCorrectionDict:
    """Dictionary for correcting common misspellings and variations"""
    
    def __init__(self):
        """
        Initialize word correction dictionary with common misspellings.

        Note: This dictionary works AFTER normalize_arabic_spelling() has been applied,
        so all entries should use normalized forms (ي not ى, ة not ه, etc.)
        """
        # Arabic corrections (after spelling normalization)
        self.arabic_corrections = {
            # ===== Harees variations =====
            'هرس': 'هريس',
            'هريس': 'هريس',  # already correct
            'حريس': 'هريس',  # ح ↔ ه confusion
            'حرس': 'هريس',
            'هرص': 'هريس',  # ص ↔ س confusion
            'حرص': 'هريس',

            # ===== Masloona variations =====
            'مصلونة': 'مصلونة',  # already correct
            'مسلونة': 'مصلونة',  # ص ↔ س confusion
            'مضلونة': 'مصلونة',  # ض ↔ ص confusion
            'مظلونة': 'مصلونة',  # ظ ↔ ص confusion

            # ===== Kabsa variations =====
            'كبسة': 'كبسة',  # already correct
            'كبصة': 'كبسة',  # ص ↔ س confusion
            'قبسة': 'كبسة',  # ق ↔ ك confusion
            'قبصة': 'كبسة',
            'كابسة': 'كبسة',  # extra alif

            # ===== Soup variations =====
            'شوربة': 'شوربة',  # already correct
            'شربة': 'شوربة',  # missing و
            'شورية': 'شوربة',  # ي ↔ ب confusion
            'صوربة': 'شوربة',  # ص ↔ ش confusion

            # ===== Salad variations =====
            'سلطة': 'سلطة',  # already correct
            'صلطة': 'سلطة',  # ص ↔ س confusion
            'سلاطة': 'سلطة',  # extra alif

            # ===== Meat variations =====
            'لحم': 'لحم',  # already correct
            'لحمة': 'لحم',  # with ta marbuta
            'لاحم': 'لحم',  # extra alif

            # ===== Chicken variations =====
            'دجاج': 'دجاج',  # already correct
            'دجاجة': 'دجاج',  # with ta marbuta
            'فراخ': 'دجاج',  # Egyptian dialect
            'فرخة': 'دجاج',
            'فرخ': 'دجاج',
            'دياي': 'دجاج',  # Gulf dialect

            # ===== Fish variations =====
            'سمك': 'سمك',  # already correct
            'سمكة': 'سمك',  # with ta marbuta
            'صمك': 'سمك',  # ص ↔ س confusion

            # ===== Rice variations =====
            'رز': 'رز',  # already correct
            'رزة': 'رز',  # with ta marbuta
            'ارز': 'رز',  # with alif
            'ريز': 'رز',  # with ya
            'عيش': 'رز',  # Egyptian dialect

            # ===== Burger variations =====
            'برجر': 'برجر',  # already correct
            'برقر': 'برجر',  # ق ↔ ج confusion
            'برغر': 'برجر',  # غ ↔ ج confusion
            'بيرجر': 'برجر',  # extra ya
            'بورجر': 'برجر',  # extra waw
            'برغير': 'برجر',

            # ===== Camel variations =====
            'جمل': 'جمل',  # already correct
            'جمال': 'جمل',  # plural form
            'جمله': 'جمل',

            # ===== Lentil variations =====
            'عدس': 'عدس',  # already correct
            'عدسة': 'عدس',  # with ta marbuta

            # ===== Hummus variations =====
            'حمص': 'حمص',  # already correct
            'حموص': 'حمص',  # with waw
            'حمصة': 'حمص',  # with ta marbuta
            'حماص': 'حمص',  # with alif

            # ===== Tabbouleh variations =====
            'تبولة': 'تبولة',  # already correct
            'تبوولة': 'تبولة',  # double waw
            'طبولة': 'تبولة',  # ط ↔ ت confusion

            # ===== Fattoush variations =====
            'فتوش': 'فتوش',  # already correct
            'فتوشة': 'فتوش',  # with ta marbuta
            'فطوش': 'فتوش',  # ط ↔ ت confusion

            # ===== Kibbeh variations =====
            'كبة': 'كبة',  # already correct
            'كبية': 'كبة',  # with ya
            'قبة': 'كبة',  # ق ↔ ك confusion
            'كبية': 'كبة',

            # ===== Delivery variations =====
            'توصيل': 'توصيل',  # already correct
            'توصيلة': 'توصيل',
            'ديليفري': 'توصيل',  # English
            'دليفري': 'توصيل',

            # ===== Address variations =====
            'عنوان': 'عنوان',  # already correct
            'عنواني': 'عنوان',
            'موقع': 'موقع',
            'موقعي': 'موقع',

            # ===== Payment variations =====
            'دفع': 'دفع',  # already correct
            'دفعة': 'دفع',
            'كاش': 'نقد',  # English
            'كاشي': 'نقد',

            # ===== Confirmation variations =====
            'اكيد': 'أكيد',
            'تمام': 'نعم',
            'ماشي': 'نعم',
            'اوكي': 'نعم',
            'اوك': 'نعم',
        }
        
        # English corrections
        self.english_corrections = {
            # Harees variations
            'hares': 'harees',
            'haris': 'harees',
            'hariss': 'harees',
            'harres': 'harees',
            
            # Masloona variations
            'maslouna': 'masloona',
            'maslona': 'masloona',
            'maslouna': 'masloona',
            
            # Kabsa variations
            'kabsah': 'kabsa',
            'kapsa': 'kabsa',
            'kapsah': 'kabsa',
            
            # Soup variations
            'soupe': 'soup',
            'sup': 'soup',
            
            # Salad variations
            'salat': 'salad',
            
            # Meat variations
            'meet': 'meat',
            'mete': 'meat',
            
            # Chicken variations
            'chiken': 'chicken',
            'chikken': 'chicken',
            'chickin': 'chicken',
            
            # Fish variations
            'fisch': 'fish',
            'fsh': 'fish',
            
            # Rice variations
            'rize': 'rice',
            'ryce': 'rice',
            
            # Camel variations
            'camle': 'camel',
            'cammel': 'camel',
            
            # Lentil variations
            'lentile': 'lentil',
            'lentils': 'lentil',
            
            # Delivery variations
            'delivry': 'delivery',
            'delivary': 'delivery',
            
            # Payment variations
            'payement': 'payment',
            'paymant': 'payment',
        }
        
        # Synonyms (words that mean the same thing)
        self.arabic_synonyms = {
            'أريد': ['ابغى', 'ابي', 'بدي', 'عايز'],
            'عندكم': ['عندك', 'موجود', 'متوفر'],
            'كم': ['بكم', 'سعر', 'ثمن'],
            'توصيل': ['ديليفري', 'إيصال'],
            'نعم': ['اي', 'ايوه', 'اوكي', 'تمام', 'ماشي'],
            'لا': ['لأ', 'مو', 'ما'],
        }
        
        self.english_synonyms = {
            'want': ['need', 'like', 'order'],
            'have': ['got', 'available'],
            'price': ['cost', 'how much'],
            'delivery': ['deliver', 'shipping'],
            'yes': ['yeah', 'yep', 'ok', 'okay'],
            'no': ['nope', 'nah'],
        }
        
        logger.info(f"✅ Word correction dictionary loaded: "
                   f"{len(self.arabic_corrections)} AR + {len(self.english_corrections)} EN corrections")
    
    def correct_word(self, word: str, language: str = 'auto') -> str:
        """
        Correct a single word
        
        Args:
            word: Word to correct
            language: 'ar', 'en', or 'auto' (auto-detect)
            
        Returns:
            Corrected word (or original if no correction found)
        """
        word_lower = word.lower().strip()
        
        if language == 'auto':
            # Auto-detect language
            if self._is_arabic(word):
                language = 'ar'
            else:
                language = 'en'
        
        if language == 'ar':
            return self.arabic_corrections.get(word_lower, word)
        else:
            return self.english_corrections.get(word_lower, word)
    
    def correct_query(self, query: str) -> str:
        """
        Correct a full query (multiple words)
        
        Args:
            query: Query to correct
            
        Returns:
            Corrected query
        """
        words = query.split()
        corrected_words = []
        
        for word in words:
            corrected = self.correct_word(word)
            corrected_words.append(corrected)
        
        corrected_query = ' '.join(corrected_words)
        
        if corrected_query != query:
            logger.info(f"🔧 Corrected query: '{query}' → '{corrected_query}'")
        
        return corrected_query
    
    def expand_with_synonyms(self, query: str) -> List[str]:
        """
        Expand query with synonyms
        
        Args:
            query: Original query
            
        Returns:
            List of queries including original and synonym variations
        """
        queries = [query]
        words = query.split()
        
        # Check if any word has synonyms
        for word in words:
            word_lower = word.lower()
            
            # Check Arabic synonyms
            for main_word, synonyms in self.arabic_synonyms.items():
                if word_lower == main_word.lower():
                    # Add queries with each synonym
                    for synonym in synonyms:
                        new_query = query.replace(word, synonym)
                        if new_query not in queries:
                            queries.append(new_query)
                elif word_lower in [s.lower() for s in synonyms]:
                    # Replace synonym with main word
                    new_query = query.replace(word, main_word)
                    if new_query not in queries:
                        queries.append(new_query)
            
            # Check English synonyms
            for main_word, synonyms in self.english_synonyms.items():
                if word_lower == main_word.lower():
                    for synonym in synonyms:
                        new_query = query.replace(word, synonym)
                        if new_query not in queries:
                            queries.append(new_query)
                elif word_lower in [s.lower() for s in synonyms]:
                    new_query = query.replace(word, main_word)
                    if new_query not in queries:
                        queries.append(new_query)
        
        if len(queries) > 1:
            logger.info(f"🔍 Expanded query to {len(queries)} variations")
        
        return queries
    
    def _is_arabic(self, text: str) -> bool:
        """Check if text contains Arabic characters"""
        return any('\u0600' <= char <= '\u06FF' for char in text)


# Global instance
word_correction_dict = WordCorrectionDict()

