"""
Unified Menu Search - البحث الموحد في المنيو
توحيد جميع طرق البحث في المنيو

🟩 LAYER 3: ACTION LAYER
"""

from typing import List, Dict, Optional, Tuple
from dataclasses import dataclass
from enum import Enum
from app.services.faiss_menu_search import faiss_menu_search
from app.services.word_correction_dict import word_correction_dict
from app.utils.text_helpers import normalize_text
from app.utils.logger import app_logger as logger


class SearchMethod(str, Enum):
    """طرق البحث"""
    EXACT_MATCH = "exact_match"
    PARTIAL_MATCH = "partial_match"
    SEMANTIC_SEARCH = "semantic_search"
    CATEGORY_FILTER = "category_filter"
    TAG_SEARCH = "tag_search"
    PRICE_FILTER = "price_filter"


@dataclass
class SearchResult:
    """نتيجة البحث"""
    
    # الأصناف المطابقة
    items: List[Dict]
    
    # طريقة البحث المستخدمة
    method: SearchMethod
    
    # معلومات البحث
    query: str
    corrected_query: Optional[str]
    
    # الإحصائيات
    total_results: int
    search_time_ms: float
    confidence: float
    
    # فلاتر مطبقة
    filters_applied: Dict = None
    
    def to_dict(self) -> Dict:
        """تحويل إلى قاموس"""
        return {
            'items': self.items,
            'method': self.method.value,
            'query': self.query,
            'corrected_query': self.corrected_query,
            'total_results': self.total_results,
            'search_time_ms': self.search_time_ms,
            'confidence': self.confidence,
            'filters_applied': self.filters_applied
        }


class UnifiedMenuSearch:
    """
    البحث الموحد في المنيو
    
    المسؤوليات:
    1. البحث الدلالي (FAISS)
    2. البحث بالمطابقة الجزئية
    3. الفلترة حسب الفئة
    4. الفلترة حسب السعر
    5. البحث بالوسوم (tags)
    6. تصحيح الأخطاء الإملائية
    
    الاستخدام:
        search = UnifiedMenuSearch()
        result = await search.search(
            query="هريس دجاج",
            top_k=5,
            category="main_dishes"
        )
    """
    
    def __init__(self):
        self.faiss_search = faiss_menu_search
    
    async def search(
        self,
        query: str,
        top_k: int = 5,
        category: Optional[str] = None,
        min_price: Optional[float] = None,
        max_price: Optional[float] = None,
        tags: Optional[List[str]] = None,
        language: str = "ar"
    ) -> SearchResult:
        """
        البحث الموحد
        
        Args:
            query: استعلام البحث
            top_k: عدد النتائج
            category: فلترة حسب الفئة
            min_price: الحد الأدنى للسعر
            max_price: الحد الأقصى للسعر
            tags: فلترة حسب الوسوم
            language: اللغة
            
        Returns:
            SearchResult
        """
        import time
        start_time = time.time()
        
        try:
            logger.info(f"🟩 ACTION LAYER: Searching menu for '{query}'")
            
            # 1. تصحيح الاستعلام
            corrected_query = self._correct_query(query)
            
            if corrected_query != query:
                logger.info(f"   Query corrected: '{query}' → '{corrected_query}'")
            
            # 2. تحديد طريقة البحث
            search_method = self._determine_search_method(
                corrected_query, category, tags, min_price, max_price
            )
            
            logger.info(f"   Search method: {search_method.value}")
            
            # 3. تنفيذ البحث
            items, confidence = await self._execute_search(
                corrected_query,
                search_method,
                top_k,
                category,
                min_price,
                max_price,
                tags
            )
            
            # 4. حساب الوقت
            search_time = (time.time() - start_time) * 1000
            
            logger.info(f"   ✅ Found {len(items)} items in {search_time:.1f}ms")
            
            # 5. بناء النتيجة
            result = SearchResult(
                items=items,
                method=search_method,
                query=query,
                corrected_query=corrected_query if corrected_query != query else None,
                total_results=len(items),
                search_time_ms=search_time,
                confidence=confidence,
                filters_applied={
                    'category': category,
                    'min_price': min_price,
                    'max_price': max_price,
                    'tags': tags
                }
            )
            
            return result
            
        except Exception as e:
            logger.error(f"❌ Error in unified search: {str(e)}")
            
            # إرجاع نتيجة فارغة
            search_time = (time.time() - start_time) * 1000
            
            return SearchResult(
                items=[],
                method=SearchMethod.SEMANTIC_SEARCH,
                query=query,
                corrected_query=None,
                total_results=0,
                search_time_ms=search_time,
                confidence=0.0
            )
    
    def _correct_query(self, query: str) -> str:
        """تصحيح الاستعلام"""
        try:
            # تصحيح الأخطاء الإملائية
            corrected = word_correction_dict.correct_query(query)

            # تطبيع النص
            normalized = normalize_text(corrected)

            return normalized

        except Exception as e:
            logger.warning(f"Error correcting query: {str(e)}")
            return query
    
    def _determine_search_method(
        self,
        query: str,
        category: Optional[str],
        tags: Optional[List[str]],
        min_price: Optional[float],
        max_price: Optional[float]
    ) -> SearchMethod:
        """تحديد طريقة البحث المناسبة"""
        
        # إذا كان هناك فلترة حسب الفئة فقط
        if category and not query:
            return SearchMethod.CATEGORY_FILTER
        
        # إذا كان هناك فلترة حسب الوسوم
        if tags:
            return SearchMethod.TAG_SEARCH
        
        # إذا كان هناك فلترة حسب السعر فقط
        if (min_price or max_price) and not query:
            return SearchMethod.PRICE_FILTER
        
        # إذا كان الاستعلام قصير جداً (كلمة واحدة)
        if len(query.split()) == 1:
            return SearchMethod.PARTIAL_MATCH
        
        # الافتراضي: بحث دلالي
        return SearchMethod.SEMANTIC_SEARCH
    
    async def _execute_search(
        self,
        query: str,
        method: SearchMethod,
        top_k: int,
        category: Optional[str],
        min_price: Optional[float],
        max_price: Optional[float],
        tags: Optional[List[str]]
    ) -> Tuple[List[Dict], float]:
        """
        تنفيذ البحث
        
        Returns:
            (items, confidence)
        """
        
        if method == SearchMethod.SEMANTIC_SEARCH:
            return await self._semantic_search(query, top_k, category, min_price, max_price)
        
        elif method == SearchMethod.PARTIAL_MATCH:
            return await self._partial_match_search(query, top_k, category)
        
        elif method == SearchMethod.CATEGORY_FILTER:
            return await self._category_filter(category, top_k)
        
        elif method == SearchMethod.TAG_SEARCH:
            return await self._tag_search(tags, top_k)
        
        elif method == SearchMethod.PRICE_FILTER:
            return await self._price_filter(min_price, max_price, top_k)
        
        else:
            # الافتراضي
            return await self._semantic_search(query, top_k, category, min_price, max_price)
    
    async def _semantic_search(
        self,
        query: str,
        top_k: int,
        category: Optional[str],
        min_price: Optional[float],
        max_price: Optional[float]
    ) -> Tuple[List[Dict], float]:
        """البحث الدلالي باستخدام FAISS"""
        
        try:
            # البحث في FAISS
            results = await self.faiss_search.search_menu(query, top_k=top_k * 2)
            
            if not results:
                return [], 0.0
            
            # تطبيق الفلاتر
            filtered_results = self._apply_filters(
                results,
                category=category,
                min_price=min_price,
                max_price=max_price
            )
            
            # أخذ أفضل top_k نتائج
            final_results = filtered_results[:top_k]
            
            # حساب متوسط الثقة
            avg_confidence = sum(r.get('score', 0) for r in final_results) / len(final_results) if final_results else 0.0
            
            return final_results, avg_confidence
            
        except Exception as e:
            logger.error(f"Error in semantic search: {str(e)}")
            return [], 0.0
    
    async def _partial_match_search(
        self,
        query: str,
        top_k: int,
        category: Optional[str]
    ) -> Tuple[List[Dict], float]:
        """البحث بالمطابقة الجزئية"""
        
        # استخدام FAISS للبحث الدلالي
        return await self._semantic_search(query, top_k, category, None, None)
    
    async def _category_filter(
        self,
        category: str,
        top_k: int
    ) -> Tuple[List[Dict], float]:
        """الفلترة حسب الفئة"""
        
        # جلب جميع الأصناف من الفئة
        # TODO: تنفيذ جلب من cache أو database
        return [], 0.8
    
    async def _tag_search(
        self,
        tags: List[str],
        top_k: int
    ) -> Tuple[List[Dict], float]:
        """البحث بالوسوم"""
        
        # TODO: تنفيذ البحث بالوسوم
        return [], 0.7
    
    async def _price_filter(
        self,
        min_price: Optional[float],
        max_price: Optional[float],
        top_k: int
    ) -> Tuple[List[Dict], float]:
        """الفلترة حسب السعر"""
        
        # TODO: تنفيذ الفلترة حسب السعر
        return [], 0.9
    
    def _apply_filters(
        self,
        items: List[Dict],
        category: Optional[str] = None,
        min_price: Optional[float] = None,
        max_price: Optional[float] = None
    ) -> List[Dict]:
        """تطبيق الفلاتر على النتائج"""
        
        filtered = items
        
        # فلترة حسب الفئة
        if category:
            filtered = [
                item for item in filtered
                if item.get('category') == category
            ]
        
        # فلترة حسب السعر
        if min_price is not None:
            filtered = [
                item for item in filtered
                if item.get('price_omr', 0) >= min_price
            ]
        
        if max_price is not None:
            filtered = [
                item for item in filtered
                if item.get('price_omr', 0) <= max_price
            ]
        
        return filtered


# Singleton instance
unified_menu_search = UnifiedMenuSearch()

