"""
Session Manager Service
Handles customer sessions and draft orders
"""
from typing import Optional, Dict, Any
from datetime import datetime
from app.utils.logger import app_logger as logger
from app.state_machine.order_states import OrderPhase


class SessionManager:
    """Service for managing customer sessions"""
    
    def __init__(self, supabase_client):
        self.supabase = supabase_client
    
    async def get_or_create_session(self, chat_id: str, customer_id: Optional[str] = None) -> Dict:
        """
        Get existing session or create new one
        
        Args:
            chat_id: WhatsApp chat ID
            customer_id: Optional customer ID
            
        Returns:
            Session dictionary
        """
        try:
            # Try to get existing ACTIVE session
            response = self.supabase.table('draft_sessions') \
                .select('*') \
                .eq('chat_id', chat_id) \
                .eq('status', 'active') \
                .order('updated_at', desc=True) \
                .limit(1) \
                .execute()

            if response.data and len(response.data) > 0:
                session = response.data[0]
                logger.info(f"Found existing session for chat_id: {chat_id}")
                return session
            
            # Create new session
            new_session = {
                'chat_id': chat_id,
                'customer_id': customer_id,
                'stage': 'items',
                # 🗑️ 'items': [] removed - now using draft_cart_items table
                'status': 'active',
                # 🗑️ 'subtotal': 0 removed - calculated from cart_service
                'order_phase': OrderPhase.START.value,  # إضافة المرحلة
                'phase_start_time': datetime.utcnow().isoformat(),  # وقت بداية المرحلة
                'created_at': datetime.utcnow().isoformat(),
                'updated_at': datetime.utcnow().isoformat()
            }
            
            response = self.supabase.table('draft_sessions') \
                .insert(new_session) \
                .execute()
            
            session = response.data[0]
            logger.info(f"Created new session for chat_id: {chat_id}")
            return session
            
        except Exception as e:
            logger.error(f"Error getting/creating session: {e}")
            raise
    
    async def update_session(self, session_id: str, updates: Dict[str, Any]) -> Dict:
        """
        Update session with new data

        Args:
            session_id: Session UUID
            updates: Dictionary of fields to update

        Returns:
            Updated session dictionary
        """
        try:
            # Filter out fields that don't exist in the database
            # pending_quantity is stored in memory only (in session object)
            db_updates = {k: v for k, v in updates.items() if k != 'pending_quantity'}

            db_updates['updated_at'] = datetime.utcnow().isoformat()

            response = self.supabase.table('draft_sessions') \
                .update(db_updates) \
                .eq('id', session_id) \
                .execute()

            session = response.data[0]
            logger.info(f"Updated session {session_id}: {list(db_updates.keys())}")
            return session

        except Exception as e:
            logger.error(f"Error updating session: {e}")
            raise
    
    # 🗑️ DEPRECATED: add_items_to_session removed - now using cart_service instead
    
    async def update_address(
        self,
        session_id: str,
        address: str = None,
        latitude: float = None,
        longitude: float = None,
        location_name: str = None,
        address_type: str = 'text',
        notes: Optional[str] = None
    ) -> Dict:
        """
        Update session address (text or location)

        Args:
            session_id: Session UUID
            address: Delivery address text (optional if location provided)
            latitude: Location latitude (optional)
            longitude: Location longitude (optional)
            location_name: Location name (optional)
            address_type: 'text' or 'location'
            notes: Optional delivery notes

        Returns:
            Updated session dictionary
        """
        try:
            updates = {
                'stage': 'payment',
                'address_type': address_type
            }

            if address:
                updates['address'] = address

            if latitude is not None:
                updates['latitude'] = latitude

            if longitude is not None:
                updates['longitude'] = longitude

            if location_name:
                updates['location_name'] = location_name

            if notes:
                updates['notes'] = notes

            return await self.update_session(session_id, updates)

        except Exception as e:
            logger.error(f"Error updating address: {e}")
            raise
    
    async def update_payment_method(self, session_id: str, payment_method: str) -> Dict:
        """
        Update session payment method
        
        Args:
            session_id: Session UUID
            payment_method: Payment method (نقدي or بطاقة)
            
        Returns:
            Updated session dictionary
        """
        try:
            updates = {
                'payment_method': payment_method,
                'stage': 'confirmation'
            }
            
            return await self.update_session(session_id, updates)
            
        except Exception as e:
            logger.error(f"Error updating payment method: {e}")
            raise
    
    async def cancel_session(self, session_id: str) -> Dict:
        """
        Cancel/delete session
        
        Args:
            session_id: Session UUID
            
        Returns:
            Deleted session dictionary
        """
        try:
            response = self.supabase.table('draft_sessions') \
                .delete() \
                .eq('id', session_id) \
                .execute()
            
            logger.info(f"Cancelled session {session_id}")
            return response.data[0] if response.data else {}
            
        except Exception as e:
            logger.error(f"Error cancelling session: {e}")
            raise
    
    async def get_session_context(self, session: Dict) -> Dict:
        """
        Extract comprehensive context from session for prompt building

        Args:
            session: Session dictionary

        Returns:
            Comprehensive context dictionary with all session details
        """
        # Get items with full details
        items = session.get('items', [])
        items_count = len(items)

        # Calculate total quantity
        total_quantity = sum(item.get('quantity', 0) for item in items)

        # Get last completed order info
        last_order_info = None
        try:
            chat_id = session.get('chat_id')
            logger.info(f"🔍 Fetching last order for chat_id: {chat_id}")
            if chat_id:
                # Get chat UUID
                chat_response = self.supabase.table('chats') \
                    .select('id') \
                    .eq('chat_id', chat_id) \
                    .single() \
                    .execute()

                if chat_response.data:
                    chat_uuid = chat_response.data.get('id')
                    logger.info(f"✅ Found chat UUID: {chat_uuid}")

                    # Get latest completed order
                    order_response = self.supabase.table('orders') \
                        .select('id, order_id, created_at, status') \
                        .eq('chat_id', chat_uuid) \
                        .order('created_at', desc=True) \
                        .limit(1) \
                        .execute()

                    if order_response.data:
                        order = order_response.data[0]
                        logger.info(f"✅ Found last order: {order['order_id']}")

                        # Get order items count
                        items_response = self.supabase.table('order_items') \
                            .select('id') \
                            .eq('order_id', order['id']) \
                            .execute()

                        last_order_info = {
                            'order_id': order['order_id'],
                            'created_at': order['created_at'],
                            'status': order['status'],
                            'items_count': len(items_response.data) if items_response.data else 0
                        }
                        logger.info(f"✅ Last order info: {last_order_info}")
                    else:
                        logger.warning(f"❌ No orders found for chat UUID: {chat_uuid}")
                else:
                    logger.warning(f"❌ Chat not found for chat_id: {chat_id}")
        except Exception as e:
            logger.error(f"❌ Error fetching last order info: {e}")

        return {
            # Basic identifiers
            'session_id': session.get('id'),
            'customer_id': session.get('customer_id'),
            'chat_id': session.get('chat_id'),

            # Session state
            'stage': session.get('stage', 'items'),
            'status': session.get('status', 'active'),
            'active_intent': session.get('active_intent'),
            'lang': session.get('lang'),

            # Order details
            'items': items,
            'items_count': items_count,
            'total_quantity': total_quantity,
            'subtotal': session.get('subtotal', 0),

            # Delivery information
            'service_type': session.get('service_type'),
            'address': session.get('address'),

            # Payment
            'payment_method': session.get('payment_method'),

            # Additional information
            'notes': session.get('notes'),
            'attachments': session.get('attachments', []),

            # Last completed order
            'last_order': last_order_info,

            # Metadata
            'created_at': session.get('created_at'),
            'updated_at': session.get('updated_at'),
            'last_message_id': session.get('last_message_id'),
            'promoted_order_id': session.get('promoted_order_id'),

            # Order Phase (State Machine)
            'order_phase': session.get('order_phase', OrderPhase.START.value),
            'phase_start_time': session.get('phase_start_time'),

            # ✅ Order food handler states (for multi-step interactions)
            'waiting_for_item_choice': session.get('waiting_for_item_choice', False),
            'waiting_for_quantity': session.get('waiting_for_quantity', False),
            'pending_item_choices': session.get('pending_item_choices', []),
            'selected_item_code': session.get('selected_item_code'),

            # ✅ Include full session for handlers that need it
            'session': session
        }

    async def update_order_phase(
        self,
        session_id: str,
        new_phase: OrderPhase,
        reset_timer: bool = True
    ) -> Dict:
        """
        تحديث مرحلة الطلب

        Args:
            session_id: Session UUID
            new_phase: المرحلة الجديدة
            reset_timer: إعادة تعيين وقت بداية المرحلة

        Returns:
            Updated session dictionary
        """
        try:
            updates = {
                'order_phase': new_phase.value,
                'updated_at': datetime.utcnow().isoformat()
            }

            if reset_timer:
                updates['phase_start_time'] = datetime.utcnow().isoformat()

            response = self.supabase.table('draft_sessions') \
                .update(updates) \
                .eq('id', session_id) \
                .execute()

            session = response.data[0]
            logger.info(f"✅ Updated order phase for session {session_id}: {new_phase.value}")
            return session

        except Exception as e:
            logger.error(f"❌ Error updating order phase: {e}")
            raise

