"""
Response Formatter - منسق الردود الموحد
تنسيق موحد لجميع أنواع الردود

🟥 LAYER 4: OUTPUT LAYER
"""

from typing import Dict, List, Optional, Any
from dataclasses import dataclass
from enum import Enum
from app.utils.logger import app_logger as logger


class ResponseType(str, Enum):
    """أنواع الردود"""
    TEXT = "text"
    MENU_LIST = "menu_list"
    CART_SUMMARY = "cart_summary"
    ORDER_CONFIRMATION = "order_confirmation"
    ORDER_STATUS = "order_status"
    ERROR = "error"
    QUESTION = "question"
    OPTIONS = "options"


@dataclass
class FormattedResponse:
    """رد منسق"""
    
    # المحتوى
    text: str
    response_type: ResponseType
    
    # اللغة
    language: str
    
    # بيانات إضافية
    data: Optional[Dict] = None
    
    # إجراءات
    action: Optional[str] = None
    buttons: Optional[List[Dict]] = None
    
    # وسائط
    media_url: Optional[str] = None
    media_type: Optional[str] = None
    
    # إحصائيات
    formatting_time_ms: float = 0.0
    
    def to_dict(self) -> Dict:
        """تحويل إلى قاموس"""
        return {
            'text': self.text,
            'response_type': self.response_type.value,
            'language': self.language,
            'data': self.data,
            'action': self.action,
            'buttons': self.buttons,
            'media_url': self.media_url,
            'media_type': self.media_type,
            'formatting_time_ms': self.formatting_time_ms
        }


class ResponseFormatter:
    """
    منسق الردود الموحد
    
    المسؤوليات:
    1. تنسيق الردود النصية
    2. إضافة emojis مناسبة
    3. تنسيق القوائم والجداول
    4. تنسيق الأزرار والخيارات
    5. دعم متعدد اللغات
    
    الاستخدام:
        formatter = ResponseFormatter()
        response = formatter.format_menu_list(
            items=[...],
            language="ar"
        )
    """
    
    # Emojis
    EMOJIS = {
        'food': '🍽️',
        'cart': '🛒',
        'check': '✅',
        'cross': '❌',
        'money': '💰',
        'location': '📍',
        'phone': '📱',
        'clock': '🕐',
        'delivery': '🚚',
        'cooking': '👨‍🍳',
        'question': '❓',
        'info': 'ℹ️',
        'warning': '⚠️',
        'star': '⭐',
        'fire': '🔥',
        'new': '🆕'
    }
    
    def __init__(self):
        pass
    
    def format_menu_list(
        self,
        items: List[Dict],
        language: str = "ar",
        show_prices: bool = True,
        show_descriptions: bool = False,
        numbered: bool = True
    ) -> FormattedResponse:
        """
        تنسيق قائمة أصناف المنيو
        
        Args:
            items: قائمة الأصناف
            language: اللغة
            show_prices: عرض الأسعار
            show_descriptions: عرض الأوصاف
            numbered: ترقيم الأصناف
            
        Returns:
            FormattedResponse
        """
        import time
        start_time = time.time()
        
        try:
            is_arabic = language == "ar"
            lines = []
            
            # العنوان
            if is_arabic:
                lines.append(f"{self.EMOJIS['food']} **قائمة الطعام:**\n")
            else:
                lines.append(f"{self.EMOJIS['food']} **Menu:**\n")
            
            # الأصناف
            for idx, item in enumerate(items, 1):
                name = item.get('name_ar' if is_arabic else 'name_en', item.get('name', ''))
                price = item.get('price_omr', 0)
                
                # الرقم والاسم
                if numbered:
                    line = f"{idx}. **{name}**"
                else:
                    line = f"• **{name}**"
                
                # السعر
                if show_prices:
                    if is_arabic:
                        line += f" - {price} ر.ع"
                    else:
                        line += f" - {price} OMR"
                
                lines.append(line)
                
                # الوصف
                if show_descriptions and item.get('description'):
                    desc = item.get('description_ar' if is_arabic else 'description_en', '')
                    if desc:
                        lines.append(f"   _{desc}_")
                
                lines.append("")  # سطر فارغ
            
            # السؤال النهائي
            if is_arabic:
                lines.append(f"\n{self.EMOJIS['question']} **هل تريد طلب شيء؟**")
            else:
                lines.append(f"\n{self.EMOJIS['question']} **Would you like to order something?**")
            
            text = "\n".join(lines)
            
            processing_time = (time.time() - start_time) * 1000
            
            return FormattedResponse(
                text=text,
                response_type=ResponseType.MENU_LIST,
                language=language,
                data={'items': items, 'count': len(items)},
                formatting_time_ms=processing_time
            )
            
        except Exception as e:
            logger.error(f"Error formatting menu list: {str(e)}")
            return self._error_response(language)
    
    def format_cart_summary(
        self,
        items: List[Dict],
        subtotal: float,
        language: str = "ar"
    ) -> FormattedResponse:
        """
        تنسيق ملخص السلة
        
        Args:
            items: أصناف السلة
            subtotal: الإجمالي
            language: اللغة
            
        Returns:
            FormattedResponse
        """
        import time
        start_time = time.time()
        
        try:
            is_arabic = language == "ar"
            lines = []
            
            # العنوان
            if is_arabic:
                lines.append(f"{self.EMOJIS['cart']} **سلتك الحالية:**\n")
            else:
                lines.append(f"{self.EMOJIS['cart']} **Your Cart:**\n")
            
            # الأصناف
            for item in items:
                name = item.get('name_ar' if is_arabic else 'name_en', item.get('name', ''))
                quantity = item.get('quantity', 1)
                price = item.get('price', 0)
                total = quantity * price
                
                if is_arabic:
                    lines.append(f"• {name} × {quantity} = {total:.2f} ر.ع")
                else:
                    lines.append(f"• {name} × {quantity} = {total:.2f} OMR")
            
            # الإجمالي
            lines.append("\n" + "─" * 30)
            if is_arabic:
                lines.append(f"{self.EMOJIS['money']} **الإجمالي:** {subtotal:.2f} ر.ع")
            else:
                lines.append(f"{self.EMOJIS['money']} **Total:** {subtotal:.2f} OMR")
            
            text = "\n".join(lines)
            
            processing_time = (time.time() - start_time) * 1000
            
            return FormattedResponse(
                text=text,
                response_type=ResponseType.CART_SUMMARY,
                language=language,
                data={'items': items, 'subtotal': subtotal, 'count': len(items)},
                formatting_time_ms=processing_time
            )
            
        except Exception as e:
            logger.error(f"Error formatting cart summary: {str(e)}")
            return self._error_response(language)
    
    def format_order_confirmation(
        self,
        order_number: str,
        items: List[Dict],
        total: float,
        address: str,
        payment_method: str,
        language: str = "ar"
    ) -> FormattedResponse:
        """
        تنسيق تأكيد الطلب
        """
        import time
        start_time = time.time()
        
        try:
            is_arabic = language == "ar"
            lines = []
            
            # العنوان
            if is_arabic:
                lines.append(f"{self.EMOJIS['check']} **تم تأكيد طلبك!**\n")
                lines.append(f"رقم الطلب: **{order_number}**\n")
            else:
                lines.append(f"{self.EMOJIS['check']} **Order Confirmed!**\n")
                lines.append(f"Order Number: **{order_number}**\n")
            
            # الأصناف
            if is_arabic:
                lines.append(f"{self.EMOJIS['food']} **الأصناف:**")
            else:
                lines.append(f"{self.EMOJIS['food']} **Items:**")
            
            for item in items:
                name = item.get('name_ar' if is_arabic else 'name_en', '')
                quantity = item.get('quantity', 1)
                lines.append(f"• {name} × {quantity}")
            
            # الإجمالي
            if is_arabic:
                lines.append(f"\n{self.EMOJIS['money']} **الإجمالي:** {total:.2f} ر.ع")
            else:
                lines.append(f"\n{self.EMOJIS['money']} **Total:** {total:.2f} OMR")
            
            # العنوان
            if is_arabic:
                lines.append(f"{self.EMOJIS['location']} **العنوان:** {address}")
            else:
                lines.append(f"{self.EMOJIS['location']} **Address:** {address}")
            
            # طريقة الدفع
            if is_arabic:
                lines.append(f"{self.EMOJIS['money']} **الدفع:** {payment_method}")
            else:
                lines.append(f"{self.EMOJIS['money']} **Payment:** {payment_method}")
            
            # رسالة الشكر
            if is_arabic:
                lines.append(f"\n{self.EMOJIS['cooking']} جاري تحضير طلبك...")
                lines.append("شكراً لك! 🙏")
            else:
                lines.append(f"\n{self.EMOJIS['cooking']} Preparing your order...")
                lines.append("Thank you! 🙏")
            
            text = "\n".join(lines)
            
            processing_time = (time.time() - start_time) * 1000
            
            return FormattedResponse(
                text=text,
                response_type=ResponseType.ORDER_CONFIRMATION,
                language=language,
                data={
                    'order_number': order_number,
                    'items': items,
                    'total': total,
                    'address': address,
                    'payment_method': payment_method
                },
                formatting_time_ms=processing_time
            )
            
        except Exception as e:
            logger.error(f"Error formatting order confirmation: {str(e)}")
            return self._error_response(language)
    
    def _error_response(self, language: str = "ar") -> FormattedResponse:
        """رد خطأ"""
        is_arabic = language == "ar"
        
        if is_arabic:
            text = f"{self.EMOJIS['warning']} عذراً، حدث خطأ في تنسيق الرد."
        else:
            text = f"{self.EMOJIS['warning']} Sorry, an error occurred while formatting the response."
        
        return FormattedResponse(
            text=text,
            response_type=ResponseType.ERROR,
            language=language
        )


# Singleton instance
response_formatter = ResponseFormatter()

