#!/usr/bin/env python3
# -*- coding: utf-8 -*-
"""
Reminder Service - خدمة التذكير للطلبات غير المكتملة
ترسل رسائل تذكير تلقائية بعد فترة من عدم النشاط

الميزات:
- كشف الطلبات غير المكتملة
- إرسال تذكير بعد 5 دقائق من عدم النشاط
- تتبع التذكيرات المرسلة (لتجنب الإزعاج)
- دعم ثنائي اللغة (عربي/إنجليزي)
"""

import asyncio
from datetime import datetime, timedelta
from typing import Dict, Optional, List
from app.utils.logger import app_logger as logger
from app.config import settings
from supabase import create_client

# إنشاء عميل Supabase
supabase = create_client(settings.SUPABASE_URL, settings.SUPABASE_KEY)


class ReminderService:
    """خدمة التذكير للطلبات غير المكتملة"""
    
    def __init__(self):
        """تهيئة الخدمة"""
        self.reminder_delay_minutes = 5  # التأخير قبل إرسال التذكير
        self.max_reminders_per_session = 1  # عدد التذكيرات المسموح لكل جلسة
        self.reminders_sent = {}  # تتبع التذكيرات المرسلة {chat_id: count}
    
    async def check_incomplete_orders(self):
        """
        فحص الطلبات غير المكتملة وإرسال تذكيرات
        
        يتم استدعاء هذه الدالة دورياً (كل دقيقة مثلاً)
        """
        try:
            logger.info("🔍 Checking for incomplete orders...")
            
            # حساب الوقت الحد (5 دقائق مضت)
            cutoff_time = datetime.utcnow() - timedelta(minutes=self.reminder_delay_minutes)
            
            # البحث عن الجلسات غير المكتملة
            result = supabase.table('draft_sessions').select('*').execute()
            
            if not result.data:
                logger.info("✅ No sessions found")
                return
            
            incomplete_count = 0
            reminders_sent_count = 0
            
            for session in result.data:
                chat_id = session.get('chat_id')
                
                # تحقق من حالة الجلسة
                is_incomplete = self._is_incomplete_order(session)
                
                if not is_incomplete:
                    continue
                
                incomplete_count += 1
                
                # تحقق من آخر نشاط
                last_activity = session.get('updated_at')
                if not last_activity:
                    continue
                
                # تحويل النص إلى datetime (بدون timezone)
                last_activity_dt = datetime.fromisoformat(last_activity.replace('Z', '').replace('+00:00', ''))
                
                # إذا كان آخر نشاط قبل 5 دقائق أو أكثر
                if last_activity_dt < cutoff_time:
                    # تحقق من عدد التذكيرات المرسلة
                    reminders_count = self.reminders_sent.get(chat_id, 0)
                    
                    if reminders_count < self.max_reminders_per_session:
                        # إرسال تذكير
                        await self._send_reminder(session)
                        self.reminders_sent[chat_id] = reminders_count + 1
                        reminders_sent_count += 1
            
            logger.info(f"✅ Found {incomplete_count} incomplete orders, sent {reminders_sent_count} reminders")
            
        except Exception as e:
            logger.error(f"❌ Error checking incomplete orders: {e}")
    
    def _is_incomplete_order(self, session: Dict) -> bool:
        """
        تحديد إذا كان الطلب غير مكتمل
        
        الطلب غير مكتمل إذا:
        - هناك أصناف في السلة ولكن لم يتم تأكيد الطلب
        - المستخدم في حالة انتظار (waiting_for_quantity, waiting_for_choice)
        - لم يتم إدخال العنوان بعد
        
        Args:
            session: بيانات الجلسة
            
        Returns:
            True إذا كان الطلب غير مكتمل
        """
        # فحص السلة
        cart_items = session.get('cart_items', [])
        has_items = len(cart_items) > 0
        
        # فحص حالات الانتظار
        waiting_for_quantity = session.get('waiting_for_quantity', False)
        waiting_for_choice = session.get('waiting_for_item_choice', False)
        waiting_for_more = session.get('waiting_for_more_items', False)
        
        # فحص العنوان
        delivery_address = session.get('delivery_address')
        has_address = delivery_address is not None and delivery_address != ''
        
        # الطلب غير مكتمل إذا:
        # 1. هناك أصناف في السلة ولكن لا يوجد عنوان
        # 2. المستخدم في حالة انتظار
        is_incomplete = (
            (has_items and not has_address) or
            waiting_for_quantity or
            waiting_for_choice or
            waiting_for_more
        )
        
        return is_incomplete
    
    async def _send_reminder(self, session: Dict):
        """
        إرسال رسالة تذكير للمستخدم
        
        Args:
            session: بيانات الجلسة
        """
        try:
            chat_id = session.get('chat_id')
            phone_number = session.get('phone_number')
            
            if not chat_id or not phone_number:
                logger.warning(f"⚠️ Missing chat_id or phone_number for session")
                return
            
            # تحديد اللغة
            language = session.get('language', 'ar')
            
            # تحديد نوع التذكير بناءً على حالة الجلسة
            reminder_message = self._get_reminder_message(session, language)
            
            # إرسال الرسالة عبر Evolution API
            await self._send_whatsapp_message(phone_number, reminder_message)
            
            logger.info(f"✅ Reminder sent to {chat_id}")
            
        except Exception as e:
            logger.error(f"❌ Error sending reminder: {e}")
    
    def _get_reminder_message(self, session: Dict, language: str = 'ar') -> str:
        """
        توليد رسالة التذكير بناءً على حالة الجلسة
        
        Args:
            session: بيانات الجلسة
            language: اللغة (ar أو en)
            
        Returns:
            رسالة التذكير
        """
        cart_items = session.get('cart_items', [])
        has_items = len(cart_items) > 0
        
        waiting_for_quantity = session.get('waiting_for_quantity', False)
        waiting_for_choice = session.get('waiting_for_item_choice', False)
        
        if language == 'ar':
            if waiting_for_quantity or waiting_for_choice:
                # المستخدم في حالة انتظار اختيار
                return "👋 مرحباً!\n\nلاحظت أنك لم تكمل اختيارك.\n\nهل تريد المتابعة؟"
            elif has_items:
                # هناك أصناف في السلة
                items_count = len(cart_items)
                return f"👋 مرحباً!\n\nلديك {items_count} صنف في سلتك.\n\nهل تريد إكمال الطلب؟"
            else:
                # رسالة عامة
                return "👋 مرحباً!\n\nهل تريد طلب شيء؟\n\nاكتب اسم الصنف أو \"المنيو\""
        else:
            # English
            if waiting_for_quantity or waiting_for_choice:
                return "👋 Hi!\n\nI noticed you didn't complete your selection.\n\nWould you like to continue?"
            elif has_items:
                items_count = len(cart_items)
                return f"👋 Hi!\n\nYou have {items_count} item(s) in your cart.\n\nWould you like to complete your order?"
            else:
                return "👋 Hi!\n\nWould you like to order something?\n\nType item name or \"menu\""
    
    async def _send_whatsapp_message(self, phone_number: str, message: str):
        """
        إرسال رسالة WhatsApp عبر Evolution API
        
        Args:
            phone_number: رقم الهاتف
            message: الرسالة
        """
        try:
            import httpx
            
            # تنسيق رقم الهاتف
            formatted_number = phone_number.replace('+', '').replace(' ', '')
            if not formatted_number.endswith('@s.whatsapp.net'):
                formatted_number = f"{formatted_number}@s.whatsapp.net"
            
            # إعداد الطلب
            url = f"{settings.EVOLUTION_API_URL}/message/sendText/{settings.EVOLUTION_INSTANCE_NAME}"
            headers = {
                "Content-Type": "application/json",
                "apikey": settings.EVOLUTION_API_KEY
            }
            payload = {
                "number": formatted_number,
                "text": message
            }
            
            # إرسال الطلب
            async with httpx.AsyncClient() as client:
                response = await client.post(url, json=payload, headers=headers, timeout=10.0)
                
                if response.status_code == 200 or response.status_code == 201:
                    logger.info(f"✅ WhatsApp message sent to {phone_number}")
                else:
                    logger.error(f"❌ Failed to send WhatsApp message: {response.status_code} - {response.text}")
                    
        except Exception as e:
            logger.error(f"❌ Error sending WhatsApp message: {e}")
    
    def clear_reminder_count(self, chat_id: str):
        """
        مسح عداد التذكيرات لجلسة معينة
        
        يتم استدعاؤها عندما يكمل المستخدم الطلب
        
        Args:
            chat_id: معرف المحادثة
        """
        if chat_id in self.reminders_sent:
            del self.reminders_sent[chat_id]
            logger.info(f"✅ Cleared reminder count for {chat_id}")
    
    async def start_background_task(self):
        """
        بدء مهمة الخلفية للفحص الدوري
        
        يتم استدعاؤها عند بدء تشغيل التطبيق
        """
        logger.info("🚀 Starting reminder service background task...")
        
        while True:
            try:
                await self.check_incomplete_orders()
                # الانتظار دقيقة واحدة قبل الفحص التالي
                await asyncio.sleep(60)
            except Exception as e:
                logger.error(f"❌ Error in reminder service background task: {e}")
                # الانتظار 5 دقائق قبل المحاولة مرة أخرى
                await asyncio.sleep(300)


# Singleton instance
reminder_service = ReminderService()

