"""
Prompt Builder V2 - Uses server-side prompts instead of database
This is faster and ensures strict, unchanging rules
"""
from typing import Dict, List
from app.utils.logger import app_logger as logger
from app.config import settings
from app.prompts import (
    SYSTEM_ROLE,
    ORDER_MANAGEMENT_PROMPT,
    SMALL_TALK_PROMPT,
    RESPONSE_FORMAT_PROMPT
)


class PromptBuilderV2:
    """Build prompts from server-side files instead of database"""
    
    def __init__(self, supabase):
        """
        Initialize prompt builder
        
        Args:
            supabase: Supabase client (still needed for menu items)
        """
        self.supabase = supabase
    
    async def fetch_menu_items(self) -> List[Dict]:
        """
        Fetch menu items from local file or database

        Returns:
            List of menu items
        """
        try:
            if settings.USE_LOCAL_MENU:
                # Use local JSON file
                from app.services.local_menu_loader import load_local_menu
                menu_items = load_local_menu()
                logger.info(f"✅ Loaded {len(menu_items)} menu items from local file")
            else:
                # Use Supabase database
                result = self.supabase.table('menu_items') \
                    .select('code, name_ar, name_en, price_omr, category') \
                    .order('category') \
                    .order('name_ar') \
                    .execute()
                menu_items = result.data if result.data else []
                logger.info(f"✅ Fetched {len(menu_items)} menu items from database")

            return menu_items

        except Exception as e:
            logger.error(f"❌ Error fetching menu items: {str(e)}")
            return []
    
    def build_menu_section(self, menu_items: List[Dict]) -> str:
        """
        Build menu section from menu items
        
        Args:
            menu_items: List of menu items
            
        Returns:
            Formatted menu string
        """
        if not menu_items:
            return "## 📋 **القائمة**\n\nلا توجد أصناف متاحة حالياً."
        
        menu_text = "## 📋 **القائمة**\n\n"
        
        # Group by category
        categories = {}
        for item in menu_items:
            category = item.get('category', 'أخرى')
            if category not in categories:
                categories[category] = []
            categories[category].append(item)
        
        # Format each category
        for category, items in categories.items():
            menu_text += f"### **{category}:**\n"
            for item in items:
                name_ar = item.get('name_ar', '')
                name_en = item.get('name_en', '')
                # Support both price_omr (from DB) and price (from local file)
                price = item.get('price_omr', item.get('price', 0))
                code = item.get('code', '')

                menu_text += f"- {name_ar}"
                if name_en:
                    menu_text += f" ({name_en})"
                menu_text += f" - {price:.3f} ر.ع - [{code}]\n"
            menu_text += "\n"
        
        return menu_text

    def _build_cart_info(self, context: Dict) -> str:
        """
        Build current cart information section

        Args:
            context: Session context

        Returns:
            Formatted cart info string (empty if no items)
        """
        items = context.get('items', [])
        subtotal = context.get('subtotal', 0)

        # Only show cart if there are items
        if not items or len(items) == 0:
            return ""

        cart_text = "## 🛒 **السلة الحالية (CURRENT CART)**\n\n"
        cart_text += "**⚠️ IMPORTANT: The customer already has items in their cart!**\n\n"

        # List items
        for idx, item in enumerate(items, 1):
            name = item.get('name', 'Unknown')
            qty = item.get('quantity', 1)
            price = item.get('price', 0)
            item_total = float(price) * int(qty)
            cart_text += f"{idx}. {name} × {qty} = {item_total:.3f} OMR\n"

        cart_text += f"\n**المجموع (Subtotal): {subtotal:.3f} OMR**\n\n"

        # Add instructions
        cart_text += """**📌 WHEN USER ASKS "ما طلبي" or "what's my order":**
→ Use `ViewCartResponse` with:
  - intent: "View Cart"
  - reply: Show the cart items above with total
  - action: null (NO ACTION!)
  - data: null

**Example Response:**
```json
{
  "intent": "View Cart",
  "reply": "لديك في السلة:\\n1. سمك مقلي × 1 = 3.500 ر.ع\\n\\nالمجموع: 3.500 ر.ع\\n\\nهل تريد إضافة صنف آخر؟",
  "action": null,
  "data": null
}
```

**IMPORTANT: Single Question System**
- Ask ONLY ONE question per message
- Never combine multiple questions like "هل تريد إضافة المزيد أم إكمال الطلب؟"
- Instead ask: "هل تريد إضافة صنف آخر؟"
- Wait for user response before asking next question

**📌 WHEN USER ADDS MORE ITEMS:**
- Use `AddItemResponse` (NOT ViewCartResponse!)
- Add to existing cart (don't replace!)
- Update subtotal

**📌 WHEN USER WANTS TO CHECKOUT:**
- Move to address stage
- Keep all items in cart
"""

        return cart_text

    async def build_system_prompt(self, context: Dict) -> str:
        """
        Build complete system prompt

        Args:
            context: Session context

        Returns:
            Complete system prompt
        """
        stage = context.get('stage', 'items')

        logger.info(f"🔨 Building system prompt for stage: {stage}")

        # Start with system role (always included)
        prompt_parts = [
            SYSTEM_ROLE,
            "\n---\n"
        ]

        # 🛒 ADD CURRENT CART INFO (if exists)
        cart_info = self._build_cart_info(context)
        if cart_info:
            prompt_parts.append(cart_info)
            prompt_parts.append("\n---\n")

        # Add stage-specific prompts
        if stage == 'items' or not stage:
            prompt_parts.extend([
                SMALL_TALK_PROMPT,
                "\n---\n",
                ORDER_MANAGEMENT_PROMPT,
                "\n---\n"
            ])
        elif stage == 'address':
            prompt_parts.append("""## 📍 **مرحلة العنوان**

**المطلوب:**
- استلام عنوان التوصيل من العميل
- يمكن أن يكون نص أو موقع GPS

**عند استلام العنوان:**
- استخدم `AddressResponse`
- احفظ العنوان
- انتقل لمرحلة الدفع

**مثال:**
```
👤 العميل: "مسقط، الخوير، شارع السلطان قابوس"
🤖 أنت: استخدم AddressResponse
{
  "intent": "New Order - Address",
  "reply": "تم حفظ العنوان ✅\n\nاختر طريقة الدفع:\n1. نقداً 💵\n2. بطاقة 💳",
  "action": "update_session",
  "address": "مسقط، الخوير، شارع السلطان قابوس"
}
```
""")
        elif stage == 'payment':
            prompt_parts.append("""## 💳 **مرحلة الدفع**

**المطلوب:**
- اختيار طريقة الدفع (نقداً أو بطاقة)

**عند اختيار الدفع:**
- استخدم `PaymentResponse`
- payment_method: "cash" أو "card"
- **stage: "confirmation"** (انتقل لمرحلة التأكيد تلقائياً)
- اسأل: "هل تريد تأكيد الطلب؟"

**مثال:**
```
👤 العميل: "نقداً"
🤖 أنت: استخدم PaymentResponse
{
  "intent": "New Order - Payment",
  "reply": "تمام! سيتم الدفع نقداً عند الاستلام 💵\n\nهل تريد تأكيد الطلب؟",
  "action": "update_session",
  "payment_method": "cash",
  "stage": "confirmation"
}
```

**ملاحظة مهمة:** لا تسأل "ماذا تريد أن تفعل؟" - اسأل مباشرة "هل تريد تأكيد الطلب؟"
""")
        elif stage == 'confirmation':
            prompt_parts.append("""## ✅ **مرحلة التأكيد**

**المطلوب:**
- عرض ملخص الطلب
- طلب التأكيد النهائي

**عند التأكيد:**
- استخدم `ConfirmOrderResponse`
- اعرض ملخص كامل
- أعط رقم الطلب

**مثال:**
```
👤 العميل: "نعم، أكد"
🤖 أنت: استخدم ConfirmOrderResponse
{
  "intent": "Confirm Order",
  "reply": "تم تأكيد طلبك! 🎉\n\nرقم الطلب: #10001\n\nسيصلك خلال 30-45 دقيقة",
  "action": "confirm_order"
}
```
""")
        
        # Add response format (always included)
        prompt_parts.extend([
            "\n---\n",
            RESPONSE_FORMAT_PROMPT
        ])
        
        # Fetch and add menu
        menu_items = await self.fetch_menu_items()
        menu_section = self.build_menu_section(menu_items)
        prompt_parts.extend([
            "\n---\n",
            menu_section
        ])
        
        # Combine all parts
        full_prompt = "".join(prompt_parts)
        
        logger.info(f"✅ Built system prompt: {len(full_prompt)} chars")
        
        return full_prompt
    
    def build_user_message(self, message_text: str, context: Dict) -> str:
        """
        Build user message with session context
        
        Args:
            message_text: Customer message
            context: Session context
            
        Returns:
            Formatted user message
        """
        # Build session context
        session_id = context.get('session_id', 'N/A')
        chat_id = context.get('chat_id', 'N/A')
        stage = context.get('stage', 'items')
        items = context.get('items', [])
        subtotal = context.get('subtotal', 0)
        address = context.get('address')
        payment_method = context.get('payment_method')
        
        # Format items
        items_text = ""
        if items:
            items_text = f"Order Items ({len(items)} items):\n"
            for i, item in enumerate(items, 1):
                name = item.get('name', 'Unknown')
                qty = item.get('quantity', 1)
                price = item.get('price', 0)
                items_text += f"  {i}. {name} x {qty} @ {price:.3f} OMR\n"
            items_text += f"Subtotal: {subtotal:.3f} OMR\n"
        else:
            items_text = "Order Items: (no items yet)\n"
        
        # Build user message
        user_message = f"""Customer Message: {message_text}

=== SESSION CONTEXT ===

Session ID: {session_id}
Chat ID: {chat_id}
Stage: {stage}

{items_text}
Delivery Address: {address or 'Not provided'}
Payment Method: {payment_method or 'Not selected'}

=== END SESSION CONTEXT ===
"""
        
        logger.info(f"✅ Built user message: {len(user_message)} chars")
        
        return user_message
    
    async def build_dynamic_prompt(
        self,
        message_text: str,
        context: Dict,
        conversation_history: List[Dict] = None
    ) -> tuple[str, str]:
        """
        Build complete prompt (system + user)
        
        Args:
            message_text: Customer message
            context: Session context
            conversation_history: Optional conversation history
            
        Returns:
            Tuple of (system_prompt, user_message)
        """
        # Build system prompt
        system_prompt = await self.build_system_prompt(context)
        
        # Build user message
        user_message = self.build_user_message(message_text, context)
        
        # Add conversation history to user message if provided
        if conversation_history:
            history_text = "\n=== RECENT CONVERSATION HISTORY ===\n\n"
            for msg in conversation_history:
                try:
                    # Handle both dict and string formats
                    if isinstance(msg, dict):
                        role = "🤖 Assistant" if msg.get('role') == 'assistant' else "👤 Customer"
                        content = str(msg.get('content', ''))[:200]  # Limit length
                    elif isinstance(msg, str):
                        role = "👤 Customer"
                        content = msg[:200]
                    else:
                        logger.warning(f"⚠️ Unexpected message format: {type(msg)}")
                        continue

                    history_text += f"{role}: {content}\n\n"
                except Exception as e:
                    logger.error(f"❌ Error processing history message: {e}")
                    continue

            history_text += "=== END CONVERSATION HISTORY ===\n\n"

            user_message = history_text + user_message
        
        logger.info("✅ Dynamic prompt built successfully")
        
        return system_prompt, user_message

