"""
Order Manager Service
Handles order creation and management
"""
from typing import Dict, Optional
from datetime import datetime
from app.utils.logger import app_logger as logger


class OrderManager:
    """Service for managing orders"""
    
    def __init__(self, supabase_client):
        self.supabase = supabase_client
    
    async def create_order_from_session(self, session: Dict) -> Dict:
        """
        Create order from draft session

        Args:
            session: Session dictionary

        Returns:
            Created order dictionary with order_id
        """
        try:
            # Get or create chat UUID from chats table using chat_id (TEXT)
            chat_id_text = session.get('chat_id')  # e.g., "201556307784@s.whatsapp.net"
            chat_uuid = None

            if chat_id_text:
                try:
                    # Try to find existing chat
                    chat_response = self.supabase.table('chats') \
                        .select('id') \
                        .eq('chat_id', chat_id_text) \
                        .execute()

                    if chat_response.data and len(chat_response.data) > 0:
                        chat_uuid = chat_response.data[0].get('id')
                        logger.info(f"✅ Found existing chat UUID: {chat_uuid} for chat_id: {chat_id_text}")
                    else:
                        # Create new chat record
                        logger.info(f"📝 Creating new chat record for: {chat_id_text}")
                        new_chat = self.supabase.table('chats').insert({
                            'chat_id': chat_id_text,
                            'instance_name': 'yasser',  # Evolution API instance name
                            'last_seen_at': datetime.utcnow().isoformat()
                        }).execute()

                        if new_chat.data and len(new_chat.data) > 0:
                            chat_uuid = new_chat.data[0].get('id')
                            logger.info(f"✅ Created new chat UUID: {chat_uuid} for chat_id: {chat_id_text}")
                        else:
                            logger.error(f"❌ Failed to create chat record for {chat_id_text}")

                except Exception as e:
                    logger.error(f"❌ Error getting/creating chat UUID for {chat_id_text}: {e}")
                    import traceback
                    logger.error(traceback.format_exc())

            # 🆕 Calculate subtotal from cart items (don't rely on session.subtotal)
            from app.services.cart_service import CartService
            cart_totals = CartService.calculate_totals(session['id'])
            subtotal = cart_totals.get('subtotal', 0)

            logger.info(f"💰 Calculated subtotal from cart: {subtotal} ر.ع")

            # Prepare order data (order_id will be auto-generated)
            order_data = {
                'chat_id': chat_uuid,  # UUID foreign key to chats table
                'subtotal_omr': subtotal,  # Correct column name
                'delivery_fee_omr': 0.500,  # Correct column name
                'total_omr': float(subtotal) + 0.500,  # Correct column name
                'delivery_address': session.get('address'),
                'notes': session.get('notes'),
                'payment_method': session.get('payment_method'),
                'status': 'open',  # ✅ Correct: valid enum value (open, confirmed, preparing, out_for_delivery, completed, cancelled)
                'created_at': datetime.utcnow().isoformat(),
                'updated_at': datetime.utcnow().isoformat()
            }

            # Add location data if available
            if session.get('latitude'):
                order_data['delivery_latitude'] = session.get('latitude')
                order_data['delivery_longitude'] = session.get('longitude')
                order_data['delivery_location_name'] = session.get('location_name')
                order_data['address_type'] = session.get('address_type')

            # Insert order (order_id will be auto-generated)
            response = self.supabase.table('orders') \
                .insert(order_data) \
                .execute()

            if not response.data or len(response.data) == 0:
                raise Exception("Failed to create order - no data returned from database")

            order = response.data[0]
            order_uuid = order.get('id')  # UUID of the order (primary key)
            order_id_text = order.get('order_id')  # Order ID (e.g., "10025")

            if not order_uuid:
                raise Exception("Order created but UUID is missing from response")

            if not order_id_text:
                raise Exception("Order created but order_id is missing from response")

            logger.info(f"✅ Created order {order_id_text} (UUID: {order_uuid}) from session {session.get('id')}")

            # Insert order items - Get from draft_cart_items table
            items = CartService.get_cart_items(session.get('id'))
            logger.info(f"📦 Retrieved {len(items)} items from cart for order {order_id_text}")

            if items:
                order_items = []
                for item in items:
                    # Get item details from draft_cart_items (correct column names)
                    item_code = item.get('item_code')  # Correct column name
                    item_name = item.get('item_name_ar') or item.get('item_name_en') or item.get('item_name', '')  # Fallback chain
                    item_qty = item.get('qty', 1)  # Correct column name
                    item_price = item.get('unit_price', 0)  # Correct column name

                    # If code is not provided, try to find it by name (fallback for old sessions)
                    if not item_code:
                        # Try exact match first
                        menu_item_response = self.supabase.table('menu_items') \
                            .select('code, name_ar, name_en') \
                            .or_(f'name_ar.eq.{item_name},name_en.eq.{item_name}') \
                            .execute()

                        # If no exact match, try partial match
                        if not menu_item_response.data:
                            menu_item_response = self.supabase.table('menu_items') \
                                .select('code, name_ar, name_en') \
                                .or_(f'name_ar.ilike.%{item_name}%,name_en.ilike.%{item_name}%') \
                                .limit(1) \
                                .execute()

                        if not menu_item_response.data:
                            logger.error(f"Menu item not found for name: {item_name}")
                            continue

                        menu_item = menu_item_response.data[0]
                        item_code = menu_item['code']
                        logger.warning(f"Used fallback search for item: {item_name} -> {item_code}")

                    order_items.append({
                        'order_id': order_uuid,  # Use UUID for foreign key
                        'item_code': item_code,  # ✅ Correct: matches order_items.item_code
                        'item_name': item_name,  # ✅ Correct: matches order_items.item_name
                        'qty': item_qty,  # ✅ Correct: matches order_items.qty
                        'unit_price_omr': item_price,  # ✅ Correct: matches order_items.unit_price_omr
                        'subtotal_omr': item_qty * item_price  # ✅ Correct: matches order_items.subtotal_omr
                    })

                if order_items:
                    self.supabase.table('order_items') \
                        .insert(order_items) \
                        .execute()

                    logger.info(f"Created {len(order_items)} order items for order {order_id_text}")

            # Update session status
            await self._update_session_after_order(session.get('id'), order_uuid)

            # Return order with order_number
            return order

        except Exception as e:
            logger.error(f"Error creating order: {e}")
            raise

    
    async def _update_session_after_order(self, session_id: str, order_id: str):
        """
        Update session after order creation
        
        Args:
            session_id: Session UUID
            order_id: Created order UUID
        """
        try:
            self.supabase.table('draft_sessions') \
                .update({
                    'status': 'completed',
                    'promoted_order_id': order_id,
                    'updated_at': datetime.utcnow().isoformat()
                }) \
                .eq('id', session_id) \
                .execute()
            
            logger.info(f"Updated session {session_id} after order creation")
            
        except Exception as e:
            logger.error(f"Error updating session after order: {e}")
            # Don't raise - order was created successfully
    
    async def get_order_by_number(self, order_number: str) -> Optional[Dict]:
        """
        Get order by order number

        Args:
            order_number: Order number (order_id)

        Returns:
            Order dictionary or None
        """
        try:
            response = self.supabase.table('orders') \
                .select('*') \
                .eq('order_id', order_number) \
                .single() \
                .execute()

            return response.data

        except Exception as e:
            logger.error(f"Error getting order by number: {e}")
            return None
    
    async def get_customer_orders(
        self,
        customer_id: str,
        limit: int = 10,
        status: Optional[str] = None
    ) -> list:
        """
        Get customer's orders
        
        Args:
            customer_id: Customer UUID
            limit: Maximum number of orders to return
            status: Optional status filter
            
        Returns:
            List of order dictionaries
        """
        try:
            query = self.supabase.table('orders') \
                .select('*') \
                .eq('customer_id', customer_id) \
                .order('created_at', desc=True) \
                .limit(limit)
            
            if status:
                query = query.eq('status', status)
            
            response = query.execute()
            
            return response.data
            
        except Exception as e:
            logger.error(f"Error getting customer orders: {e}")
            return []
    
    async def update_order_status(
        self,
        order_id: str,
        new_status: str,
        notes: Optional[str] = None
    ) -> Dict:
        """
        Update order status

        Args:
            order_id: Order UUID
            new_status: New status (open, confirmed, preparing, out_for_delivery, completed, cancelled)
            notes: Optional status notes

        Returns:
            Updated order dictionary
        """
        try:
            updates = {
                'status': new_status,
                'updated_at': datetime.utcnow().isoformat()
            }
            
            if notes:
                updates['status_notes'] = notes
            
            response = self.supabase.table('orders') \
                .update(updates) \
                .eq('id', order_id) \
                .execute()
            
            order = response.data[0]
            logger.info(f"Updated order {order.get('order_number')} status to {new_status}")
            return order
            
        except Exception as e:
            logger.error(f"Error updating order status: {e}")
            raise
    
    async def cancel_order(self, order_id: str, reason: Optional[str] = None) -> Dict:
        """
        Cancel order
        
        Args:
            order_id: Order UUID
            reason: Optional cancellation reason
            
        Returns:
            Updated order dictionary
        """
        try:
            return await self.update_order_status(
                order_id,
                'cancelled',
                notes=reason
            )
            
        except Exception as e:
            logger.error(f"Error cancelling order: {e}")
            raise

