"""
OpenAI Service with Structured Outputs Support
Uses Pydantic models to enforce strict response schemas
"""
from openai import AsyncOpenAI
import json
import asyncio
import re
from typing import Dict, Optional, Type, Union
from pydantic import BaseModel
from app.config import settings
from app.utils.logger import app_logger as logger
from app.models.ai_responses import (
    AddItemResponse,
    AskInformationResponse,
    SmallTalkResponse,
    ViewCartResponse,
    AddressResponse,
    PaymentResponse,
    ConfirmOrderResponse,
    CancelOrderResponse,
    ModifyOrderResponse,
    TrackOrderResponse,
    ContinueOrderResponse,
    RepeatOrderResponse,
    MenuResponse,
    BusinessHoursResponse,
    LocationResponse,
    OffersResponse,
    ContactRequestResponse,
    get_response_model_for_stage
)


class OpenAIServiceStructured:
    """Service for interacting with OpenAI API using Structured Outputs"""

    def __init__(self, credits_manager=None):
        self.client = AsyncOpenAI(
            api_key=settings.OPENAI_API_KEY,
            timeout=60.0  # 60 seconds timeout
        )
        self.model = settings.OPENAI_MODEL
        self.temperature = settings.OPENAI_TEMPERATURE
        self.max_tokens = settings.OPENAI_MAX_TOKENS
        self.credits_manager = credits_manager
    
    async def get_structured_response(
        self,
        system_prompt: str,
        user_message: str,
        response_model: Type[BaseModel],
        conversation_history: Optional[list] = None,
        temperature: Optional[float] = None
    ) -> BaseModel:
        """
        Get structured response from OpenAI using Pydantic models
        
        Args:
            system_prompt: System instructions
            user_message: User message
            response_model: Pydantic model for response structure
            conversation_history: Optional conversation history
            temperature: Optional temperature override
            
        Returns:
            Parsed response matching the model
        """
        try:
            logger.info(f"Sending structured request to OpenAI (model: {self.model})")

            # Get model name safely (Union types don't have __name__)
            try:
                model_name = response_model.__name__
            except AttributeError:
                model_name = str(response_model)
            logger.debug(f"Response model: {model_name}")

            if conversation_history:
                logger.info(f"Including {len(conversation_history)} messages from conversation history")
            
            # Build messages array
            messages = [{"role": "system", "content": system_prompt}]
            
            # Add conversation history if provided
            if conversation_history:
                for msg in conversation_history:
                    messages.append({
                        "role": msg["role"],
                        "content": msg["content"]
                    })
            
            # Add current user message
            messages.append({"role": "user", "content": user_message})

            # Call OpenAI with structured outputs with timeout
            try:
                response = await asyncio.wait_for(
                    self.client.beta.chat.completions.parse(
                        model=self.model,
                        messages=messages,
                        response_format=response_model,
                        temperature=temperature or self.temperature
                    ),
                    timeout=45.0  # 45 seconds timeout
                )
            except asyncio.TimeoutError:
                logger.error("❌ OpenAI API timeout after 45 seconds!")
                raise TimeoutError("OpenAI API request timed out")
            
            # Log token usage
            tokens_used = response.usage.total_tokens if hasattr(response, 'usage') else 0
            logger.info(f"💰 Tokens used: {tokens_used}")
            
            # Log cost to database if credits_manager is available
            if self.credits_manager and tokens_used > 0:
                try:
                    await self.credits_manager.log_message_cost(
                        message_type='ai_request',
                        tokens_used=tokens_used,
                        model_used=self.model
                    )
                except Exception as e:
                    logger.error(f"❌ Failed to log message cost: {str(e)}")
            
            # Extract parsed response
            parsed = response.choices[0].message.parsed
            
            if parsed is None:
                logger.error("❌ Parsed response is None!")
                raise ValueError("Failed to parse response")
            
            logger.info(f"✅ Structured response: {parsed.intent}")
            logger.debug(f"Response: {parsed.model_dump()}")
            
            return parsed
            
        except Exception as e:
            logger.error(f"Error in get_structured_response: {str(e)}")
            raise
    
    def determine_response_model(self, context: Dict, message_text: str) -> Type[BaseModel]:
        """
        Determine which response model to use based on context and message

        Args:
            context: Session context
            message_text: Customer message

        Returns:
            Appropriate Pydantic model class

        Note: OpenAI Structured Outputs doesn't support Union types directly.
        We use a single flexible model instead.
        """
        # Import the unified response model
        from app.models.ai_responses import AIResponse

        # Always return the unified model
        # The AI will choose the appropriate intent and structure
        return AIResponse
    
    async def process_message_structured(
        self,
        message_text: str,
        system_prompt: str,
        context: Dict,
        conversation_history: Optional[list] = None
    ) -> Dict:
        """
        Process message and return structured response
        
        Args:
            message_text: Customer message
            system_prompt: System prompt
            context: Session context
            conversation_history: Optional conversation history
            
        Returns:
            Response as dictionary (for compatibility with existing code)
        """
        try:
            # Determine appropriate response model
            response_model = self.determine_response_model(context, message_text)
            
            # Get structured response
            parsed = await self.get_structured_response(
                system_prompt=system_prompt,
                user_message=message_text,
                response_model=response_model,
                conversation_history=conversation_history
            )
            
            # Convert to dict for compatibility
            response_dict = parsed.model_dump()
            
            # Add intent and reply to top level (for compatibility)
            result = {
                'intent': response_dict.get('intent'),
                'reply': response_dict.get('reply'),
                'action': response_dict.get('action'),
                'data': None
            }
            
            # Handle different response types
            if isinstance(parsed, AddItemResponse):
                result['data'] = {
                    'items': [item.model_dump() for item in parsed.items],
                    'subtotal': parsed.subtotal
                }
            elif isinstance(parsed, AddressResponse):
                result['data'] = {'address': parsed.address}
            elif isinstance(parsed, PaymentResponse):
                result['data'] = {'payment_method': parsed.payment_method}
            elif isinstance(parsed, TrackOrderResponse):
                result['data'] = {'order_id': parsed.order_id}
            elif isinstance(parsed, ContinueOrderResponse):
                result['data'] = {'next_stage': parsed.next_stage}
            elif isinstance(parsed, ModifyOrderResponse):
                result['data'] = {'modification': parsed.modification}
            elif isinstance(parsed, CancelOrderResponse):
                if parsed.order_id:
                    result['data'] = {'order_id': parsed.order_id}
            elif isinstance(parsed, AskInformationResponse):
                result['reply'] = self._format_options_reply(
                    original_reply=parsed.reply,
                    options=parsed.options,
                    lang=context.get('lang')
                )
            
            logger.info(f"✅ Processed structured response: {result['intent']}")
            return result
            
        except Exception as e:
            logger.error(f"Error in process_message_structured: {str(e)}")
            # Fallback to simple response
            return {
                'intent': 'Small Talk',
                'reply': 'عذراً، حدث خطأ. يرجى المحاولة مرة أخرى.',
                'action': None,
                'data': None
            }

    @staticmethod
    def _looks_like_option_line(line: str) -> bool:
        """Check if a line already contains a numbered option."""
        if not line:
            return False
        pattern = r"^\s*([0-9]+|[٠-٩]+)\s*[\.\-\)]\s*"
        return bool(re.match(pattern, line.strip()))

    def _format_options_reply(self, original_reply: str, options: list[str], lang: Optional[str]) -> str:
        """
        Format AskInformationResponse replies with a consistent, professional layout.

        Args:
            original_reply: Reply generated by the model
            options: List of option strings provided by the model
            lang: Customer language code (ar/en)

        Returns:
            Formatted reply string
        """
        if not options:
            return original_reply

        lang = (lang or 'ar').lower()
        is_english = lang == 'en'

        lines = [line.strip() for line in (original_reply or "").splitlines() if line.strip()]

        header_line = None
        footer_line = None

        for line in lines:
            if not header_line and not self._looks_like_option_line(line):
                header_line = line
                break

        for line in reversed(lines):
            if self._looks_like_option_line(line):
                continue
            if line.endswith(('؟', '?')):
                footer_line = line
                break

        default_header = "Here are the available options:" if is_english else "إليك الخيارات المتاحة:"
        default_footer = (
            "Which option would you like? Please reply with the number so I can add it to your order."
            if is_english
            else "ما الخيار الذي تفضله؟ يرجى الرد برقم الصنف لنعمل على إضافته إلى طلبك."
        )

        header = header_line or default_header
        if header and not header.endswith((':', '؟', '?', '!', '.')):
            header = f"{header}:"

        option_lines = []
        for index, option in enumerate(options, start=1):
            clean_option = re.sub(r"^\s*([0-9]+|[٠-٩]+)\s*[\.\-\)]\s*", "", option or "").strip()
            option_lines.append(f"{index}. {clean_option}")

        footer = footer_line or default_footer

        formatted_parts = [
            header,
            "",
            *option_lines,
            "",
            footer
        ]

        return "\n".join(part for part in formatted_parts if part is not None)
