"""
Metrics Tracker - نظام تتبع مقاييس الأداء
يتتبع أداء التصنيف والاستجابة
"""

from typing import Dict, List, Optional
from datetime import datetime, timedelta
from collections import defaultdict, Counter
from dataclasses import dataclass, asdict
import json
from pathlib import Path
from app.utils.logger import app_logger as logger


@dataclass
class ClassificationMetric:
    """مقياس تصنيف واحد"""
    timestamp: str
    message: str
    intent: str
    classification_method: str  # 'rule_based', 'ai', 'fallback'
    confidence: float
    response_time_ms: float
    success: bool
    error: Optional[str] = None


class MetricsTracker:
    """
    نظام تتبع مقاييس الأداء
    
    يتتبع:
    1. نسبة نجاح القواعد vs AI
    2. وقت الاستجابة
    3. دقة التصنيف
    4. الأخطاء الشائعة
    5. النوايا الأكثر استخداماً
    """
    
    def __init__(self, metrics_dir: str = "restaurant-chatbot-server/data/metrics"):
        self.metrics_dir = Path(metrics_dir)
        self.metrics_dir.mkdir(parents=True, exist_ok=True)
        
        # In-memory metrics (last 1000 entries)
        self.recent_metrics: List[ClassificationMetric] = []
        self.max_recent = 1000
        
        # Aggregated stats
        self.stats = {
            'total_classifications': 0,
            'rule_based_count': 0,
            'ai_count': 0,
            'fallback_count': 0,
            'success_count': 0,
            'error_count': 0,
            'total_response_time_ms': 0.0,
            'intent_counts': Counter(),
            'error_types': Counter(),
            'hourly_stats': defaultdict(lambda: {'count': 0, 'rule_based': 0, 'ai': 0})
        }
        
        # Load existing metrics
        self._load_daily_metrics()
    
    def track_classification(
        self,
        message: str,
        intent: str,
        classification_method: str,
        confidence: float,
        response_time_ms: float,
        success: bool = True,
        error: Optional[str] = None
    ):
        """
        تتبع عملية تصنيف
        
        Args:
            message: رسالة المستخدم
            intent: النية المكتشفة
            classification_method: 'rule_based', 'ai', 'fallback'
            confidence: درجة الثقة
            response_time_ms: وقت الاستجابة بالميلي ثانية
            success: هل نجحت العملية؟
            error: رسالة الخطأ (إن وجدت)
        """
        try:
            # Create metric
            metric = ClassificationMetric(
                timestamp=datetime.now().isoformat(),
                message=message[:100],  # Truncate for privacy
                intent=intent,
                classification_method=classification_method,
                confidence=confidence,
                response_time_ms=response_time_ms,
                success=success,
                error=error
            )
            
            # Add to recent metrics
            self.recent_metrics.append(metric)
            if len(self.recent_metrics) > self.max_recent:
                self.recent_metrics.pop(0)
            
            # Update aggregated stats
            self._update_stats(metric)
            
            # Log
            logger.debug(
                f"📊 Metric tracked: {classification_method} | "
                f"{intent} | {response_time_ms:.0f}ms | "
                f"confidence={confidence:.2f}"
            )
            
        except Exception as e:
            logger.error(f"❌ Error tracking metric: {str(e)}")
    
    def _update_stats(self, metric: ClassificationMetric):
        """تحديث الإحصائيات المجمعة"""
        self.stats['total_classifications'] += 1
        
        # Count by method
        if metric.classification_method == 'rule_based':
            self.stats['rule_based_count'] += 1
        elif metric.classification_method == 'ai':
            self.stats['ai_count'] += 1
        elif metric.classification_method == 'fallback':
            self.stats['fallback_count'] += 1
        
        # Success/error
        if metric.success:
            self.stats['success_count'] += 1
        else:
            self.stats['error_count'] += 1
            if metric.error:
                self.stats['error_types'][metric.error] += 1
        
        # Response time
        self.stats['total_response_time_ms'] += metric.response_time_ms
        
        # Intent counts
        self.stats['intent_counts'][metric.intent] += 1
        
        # Hourly stats
        hour = datetime.fromisoformat(metric.timestamp).hour
        self.stats['hourly_stats'][hour]['count'] += 1
        if metric.classification_method == 'rule_based':
            self.stats['hourly_stats'][hour]['rule_based'] += 1
        elif metric.classification_method == 'ai':
            self.stats['hourly_stats'][hour]['ai'] += 1
    
    def get_summary(self) -> Dict:
        """
        الحصول على ملخص الإحصائيات
        """
        total = self.stats['total_classifications']
        
        if total == 0:
            return {
                'total_classifications': 0,
                'rule_based_percentage': 0.0,
                'ai_percentage': 0.0,
                'fallback_percentage': 0.0,
                'success_rate': 0.0,
                'avg_response_time_ms': 0.0,
                'top_intents': [],
                'top_errors': []
            }
        
        return {
            'total_classifications': total,
            'rule_based_percentage': (self.stats['rule_based_count'] / total) * 100,
            'ai_percentage': (self.stats['ai_count'] / total) * 100,
            'fallback_percentage': (self.stats['fallback_count'] / total) * 100,
            'success_rate': (self.stats['success_count'] / total) * 100,
            'avg_response_time_ms': self.stats['total_response_time_ms'] / total,
            'top_intents': self.stats['intent_counts'].most_common(10),
            'top_errors': self.stats['error_types'].most_common(5)
        }
    
    def get_detailed_stats(self) -> Dict:
        """
        إحصائيات مفصلة
        """
        summary = self.get_summary()
        
        # Calculate hourly distribution
        hourly_distribution = {}
        for hour, data in self.stats['hourly_stats'].items():
            total_hour = data['count']
            if total_hour > 0:
                hourly_distribution[hour] = {
                    'total': total_hour,
                    'rule_based_percentage': (data['rule_based'] / total_hour) * 100,
                    'ai_percentage': (data['ai'] / total_hour) * 100
                }
        
        # Recent performance (last 100 classifications)
        recent_100 = self.recent_metrics[-100:]
        recent_rule_based = sum(1 for m in recent_100 if m.classification_method == 'rule_based')
        recent_ai = sum(1 for m in recent_100 if m.classification_method == 'ai')
        recent_success = sum(1 for m in recent_100 if m.success)
        
        return {
            **summary,
            'hourly_distribution': hourly_distribution,
            'recent_performance': {
                'sample_size': len(recent_100),
                'rule_based_percentage': (recent_rule_based / len(recent_100) * 100) if recent_100 else 0,
                'ai_percentage': (recent_ai / len(recent_100) * 100) if recent_100 else 0,
                'success_rate': (recent_success / len(recent_100) * 100) if recent_100 else 0
            }
        }
    
    def get_performance_comparison(self) -> Dict:
        """
        مقارنة أداء القواعد vs AI
        """
        rule_based_metrics = [m for m in self.recent_metrics if m.classification_method == 'rule_based']
        ai_metrics = [m for m in self.recent_metrics if m.classification_method == 'ai']
        
        def calc_avg_time(metrics):
            if not metrics:
                return 0.0
            return sum(m.response_time_ms for m in metrics) / len(metrics)
        
        def calc_success_rate(metrics):
            if not metrics:
                return 0.0
            return (sum(1 for m in metrics if m.success) / len(metrics)) * 100
        
        return {
            'rule_based': {
                'count': len(rule_based_metrics),
                'avg_response_time_ms': calc_avg_time(rule_based_metrics),
                'success_rate': calc_success_rate(rule_based_metrics),
                'avg_confidence': sum(m.confidence for m in rule_based_metrics) / len(rule_based_metrics) if rule_based_metrics else 0
            },
            'ai': {
                'count': len(ai_metrics),
                'avg_response_time_ms': calc_avg_time(ai_metrics),
                'success_rate': calc_success_rate(ai_metrics),
                'avg_confidence': sum(m.confidence for m in ai_metrics) / len(ai_metrics) if ai_metrics else 0
            }
        }
    
    def save_daily_metrics(self):
        """
        حفظ المقاييس اليومية
        """
        try:
            today = datetime.now().strftime('%Y-%m-%d')
            metrics_file = self.metrics_dir / f"metrics_{today}.json"
            
            data = {
                'date': today,
                'summary': self.get_summary(),
                'detailed_stats': self.get_detailed_stats(),
                'performance_comparison': self.get_performance_comparison(),
                'recent_metrics': [asdict(m) for m in self.recent_metrics[-100:]]  # Save last 100
            }
            
            with open(metrics_file, 'w', encoding='utf-8') as f:
                json.dump(data, f, ensure_ascii=False, indent=2)
            
            logger.info(f"✅ Saved daily metrics to {metrics_file}")
            
        except Exception as e:
            logger.error(f"❌ Error saving daily metrics: {str(e)}")
    
    def _load_daily_metrics(self):
        """
        تحميل المقاييس اليومية
        """
        try:
            today = datetime.now().strftime('%Y-%m-%d')
            metrics_file = self.metrics_dir / f"metrics_{today}.json"
            
            if not metrics_file.exists():
                logger.info("📊 No existing metrics for today, starting fresh")
                return
            
            with open(metrics_file, 'r', encoding='utf-8') as f:
                data = json.load(f)
            
            # Restore recent metrics
            if 'recent_metrics' in data:
                self.recent_metrics = [
                    ClassificationMetric(**m) for m in data['recent_metrics']
                ]
            
            logger.info(f"✅ Loaded {len(self.recent_metrics)} metrics from {metrics_file}")
            
        except Exception as e:
            logger.error(f"❌ Error loading daily metrics: {str(e)}")
    
    def reset_stats(self):
        """إعادة تعيين الإحصائيات"""
        self.stats = {
            'total_classifications': 0,
            'rule_based_count': 0,
            'ai_count': 0,
            'fallback_count': 0,
            'success_count': 0,
            'error_count': 0,
            'total_response_time_ms': 0.0,
            'intent_counts': Counter(),
            'error_types': Counter(),
            'hourly_stats': defaultdict(lambda: {'count': 0, 'rule_based': 0, 'ai': 0})
        }
        self.recent_metrics.clear()
        logger.info("✅ Stats reset")


# Singleton instance
metrics_tracker = MetricsTracker()

