"""
Menu Sync Service
Manages synchronization between Supabase → Cache → FAISS
"""
from typing import List, Dict, Any, Optional
from datetime import datetime
import numpy as np
from app.database.supabase_client import get_supabase_client
from app.cache.menu_cache import get_menu_cache
from app.utils.logger import logger
import os


class MenuSyncService:
    """
    Service to sync menu data from Supabase to local cache and FAISS index
    """
    
    def __init__(self):
        """Initialize menu sync service"""
        self.supabase = get_supabase_client()
        self.cache = get_menu_cache()
        self.faiss_index_path = "data/faiss_menu_index.bin"
        self.embeddings_path = "data/menu_embeddings.npy"
        
        logger.info("🔄 Menu sync service initialized")
    
    def sync_full(self, force: bool = False) -> Dict[str, Any]:
        """
        Full synchronization: Supabase → Cache → FAISS
        
        Args:
            force: Force sync even if cache is fresh
            
        Returns:
            Sync result dictionary
        """
        logger.info("="*80)
        logger.info("🔄 Starting full menu synchronization...")
        logger.info("="*80)
        
        result = {
            'success': False,
            'steps': {},
            'errors': [],
            'timestamp': datetime.now().isoformat()
        }
        
        try:
            # Step 1: Check if sync is needed
            if not force and not self.cache.is_stale():
                logger.info("✅ Cache is fresh, skipping sync")
                result['success'] = True
                result['steps']['check'] = 'Cache is fresh'
                return result
            
            # Step 2: Fetch from Supabase
            logger.info("\n📥 Step 1/3: Fetching menu from Supabase...")
            menu_items = self.supabase.get_all_menu_items(active_only=True)
            
            if not menu_items:
                error = "No menu items fetched from Supabase"
                logger.error(f"❌ {error}")
                result['errors'].append(error)
                return result
            
            logger.info(f"✅ Fetched {len(menu_items)} items from Supabase")
            result['steps']['fetch'] = f"{len(menu_items)} items fetched"
            
            # Step 3: Update cache
            logger.info("\n💾 Step 2/3: Updating local cache...")
            cache_success = self.cache.save_to_file(menu_items)
            
            if not cache_success:
                error = "Failed to save cache"
                logger.error(f"❌ {error}")
                result['errors'].append(error)
                return result
            
            logger.info(f"✅ Cache updated with {len(menu_items)} items")
            result['steps']['cache'] = 'Cache updated'
            
            # Step 4: Rebuild FAISS index
            logger.info("\n🔍 Step 3/3: Rebuilding FAISS index...")
            faiss_success = self._rebuild_faiss_index(menu_items)
            
            if not faiss_success:
                error = "Failed to rebuild FAISS index"
                logger.warning(f"⚠️  {error}")
                result['errors'].append(error)
                # Don't fail the whole sync if FAISS fails
            else:
                logger.info("✅ FAISS index rebuilt")
                result['steps']['faiss'] = 'FAISS index rebuilt'
            
            # Success!
            result['success'] = True
            result['item_count'] = len(menu_items)
            
            logger.info("\n" + "="*80)
            logger.info("✅ Full synchronization completed successfully!")
            logger.info(f"📊 Total items: {len(menu_items)}")
            logger.info("="*80)
            
            return result
            
        except Exception as e:
            error = f"Sync failed: {str(e)}"
            logger.error(f"❌ {error}")
            result['errors'].append(error)
            return result
    
    def sync_cache_only(self) -> bool:
        """
        Sync only cache (Supabase → Cache)
        
        Returns:
            True if successful
        """
        logger.info("🔄 Syncing cache only...")
        
        try:
            menu_items = self.supabase.get_all_menu_items(active_only=True)
            
            if not menu_items:
                logger.error("❌ No menu items fetched")
                return False
            
            success = self.cache.save_to_file(menu_items)
            
            if success:
                logger.info(f"✅ Cache synced with {len(menu_items)} items")
            
            return success
            
        except Exception as e:
            logger.error(f"❌ Cache sync failed: {str(e)}")
            return False
    
    def _rebuild_faiss_index(self, menu_items: List[Dict[str, Any]]) -> bool:
        """
        Rebuild FAISS index from menu items
        
        Args:
            menu_items: List of menu items
            
        Returns:
            True if successful
        """
        try:
            # Check if OpenAI is available
            openai_key = os.getenv('OPENAI_API_KEY')
            if not openai_key:
                logger.warning("⚠️  OpenAI API key not found, skipping FAISS rebuild")
                return False
            
            from openai import OpenAI
            import faiss
            
            client = OpenAI(api_key=openai_key)
            
            # Generate embeddings
            logger.info("🔮 Generating embeddings...")
            embeddings = []
            
            for item in menu_items:
                # Create text for embedding
                text_parts = [
                    item.get('name_ar', ''),
                    item.get('name_en', ''),
                    item.get('description', ''),
                ]
                
                # Add tags if available
                tags = item.get('tags', [])
                if tags:
                    text_parts.append(' '.join(tags))
                
                text = ' '.join(filter(None, text_parts))
                
                # Get embedding
                response = client.embeddings.create(
                    model="text-embedding-3-small",
                    input=text
                )
                
                embeddings.append(response.data[0].embedding)
            
            # Convert to numpy array
            embeddings_array = np.array(embeddings, dtype='float32')
            
            # Create FAISS index
            dimension = embeddings_array.shape[1]
            index = faiss.IndexFlatL2(dimension)
            index.add(embeddings_array)
            
            # Save index
            os.makedirs(os.path.dirname(self.faiss_index_path), exist_ok=True)
            faiss.write_index(index, self.faiss_index_path)
            
            # Save embeddings
            np.save(self.embeddings_path, embeddings_array)
            
            logger.info(f"✅ FAISS index created with {len(embeddings)} items")
            
            return True
            
        except ImportError as e:
            logger.warning(f"⚠️  FAISS or OpenAI not available: {str(e)}")
            return False
        except Exception as e:
            logger.error(f"❌ FAISS rebuild failed: {str(e)}")
            return False
    
    def get_sync_status(self) -> Dict[str, Any]:
        """
        Get current sync status
        
        Returns:
            Status dictionary
        """
        cache_stats = self.cache.get_stats()
        
        # Check FAISS index
        faiss_exists = os.path.exists(self.faiss_index_path)
        faiss_size = os.path.getsize(self.faiss_index_path) if faiss_exists else 0
        
        # Check Supabase
        supabase_healthy = self.supabase.health_check()
        supabase_count = self.supabase.get_table_count('menu_items') if supabase_healthy else 0
        
        return {
            'supabase': {
                'healthy': supabase_healthy,
                'item_count': supabase_count
            },
            'cache': cache_stats,
            'faiss': {
                'exists': faiss_exists,
                'size_bytes': faiss_size,
                'path': self.faiss_index_path
            },
            'sync_needed': cache_stats.get('is_stale', True)
        }
    
    def validate_sync(self) -> Dict[str, Any]:
        """
        Validate that all components are in sync
        
        Returns:
            Validation result
        """
        logger.info("🔍 Validating sync status...")
        
        result = {
            'valid': True,
            'issues': [],
            'warnings': []
        }
        
        try:
            # Check Supabase
            supabase_items = self.supabase.get_all_menu_items(active_only=True)
            supabase_count = len(supabase_items)
            
            # Check Cache
            cache_items = self.cache.get_all_items()
            cache_count = len(cache_items)
            
            # Compare counts
            if supabase_count != cache_count:
                issue = f"Count mismatch: Supabase={supabase_count}, Cache={cache_count}"
                result['issues'].append(issue)
                result['valid'] = False
                logger.warning(f"⚠️  {issue}")
            
            # Check if cache is stale
            if self.cache.is_stale():
                warning = "Cache is stale (>24 hours old)"
                result['warnings'].append(warning)
                logger.warning(f"⚠️  {warning}")
            
            # Check FAISS
            if not os.path.exists(self.faiss_index_path):
                warning = "FAISS index not found"
                result['warnings'].append(warning)
                logger.warning(f"⚠️  {warning}")
            
            if result['valid'] and not result['warnings']:
                logger.info("✅ All components are in sync")
            
            return result
            
        except Exception as e:
            result['valid'] = False
            result['issues'].append(f"Validation error: {str(e)}")
            logger.error(f"❌ Validation failed: {str(e)}")
            return result


# Singleton instance
_menu_sync_service: Optional[MenuSyncService] = None


def get_menu_sync_service() -> MenuSyncService:
    """
    Get singleton menu sync service instance
    
    Returns:
        MenuSyncService instance
    """
    global _menu_sync_service
    
    if _menu_sync_service is None:
        _menu_sync_service = MenuSyncService()
        logger.info("🔄 Menu sync service singleton created")
    
    return _menu_sync_service

