"""
Unified Menu Search Service
البحث الموحد في المنيو: Exact → Partial → Semantic (FAISS)
يدمج 3 طرق بحث بالأولوية ويرتب النتائج

Uses in-memory cache (menu_cache.json) instead of database queries for ultra-fast search.
"""

from typing import List, Dict, Optional, Tuple, Set
import json
import os
from app.services.faiss_menu_search import faiss_menu_search
from app.utils.word_correction_dict import word_correction_dict
from app.utils.text_helpers import clean_user_query
from app.utils.logger import app_logger as logger
import re


class MenuSearchService:
    """
    خدمة البحث الموحد في المنيو

    التدفق:
    1. تصحيح الأخطاء الإملائية (Word Correction)
    2. البحث بالتطابق التام (Exact Match)
    3. البحث بالتطابق الجزئي (Partial Match)
    4. البحث الدلالي (Semantic FAISS)
    5. دمج وترتيب النتائج بالأولوية

    Uses in-memory cache loaded from menu_cache.json
    """

    def __init__(self):
        """
        تهيئة خدمة البحث

        Loads menu from cache file instead of database
        """
        self.faiss_search = faiss_menu_search
        self.word_corrector = word_correction_dict
        self._menu_cache = []
        self._load_menu_cache()

    def _load_menu_cache(self):
        """
        تحميل المنيو من ملف الكاش
        """
        try:
            cache_path = os.path.join(
                os.path.dirname(os.path.dirname(os.path.dirname(__file__))),
                'data', 'cache', 'menu_cache.json'
            )

            if os.path.exists(cache_path):
                with open(cache_path, 'r', encoding='utf-8') as f:
                    data = json.load(f)
                    self._menu_cache = data.get('items', [])
                    logger.info(f"✅ Loaded {len(self._menu_cache)} menu items from cache")
            else:
                logger.error(f"❌ Menu cache file not found: {cache_path}")
                self._menu_cache = []
        except Exception as e:
            logger.error(f"❌ Error loading menu cache: {str(e)}")
            self._menu_cache = []

    def search_menu(
        self,
        query: str,
        limit: int = 10,
        min_similarity: float = 0.3
    ) -> Dict[str, any]:
        """
        البحث الموحد في المنيو

        التدفق:
        1. تصحيح الأخطاء الإملائية
        2. البحث بالتطابق التام (Exact)
        3. البحث بالتطابق الجزئي (Partial)
        4. البحث الدلالي (Semantic FAISS)
        5. دمج وترتيب النتائج

        Args:
            query: استعلام البحث
            limit: الحد الأقصى للنتائج
            min_similarity: الحد الأدنى للتشابه (للبحث الدلالي)

        Returns:
            Dict مع:
                - results: قائمة النتائج المرتبة
                - search_methods: طرق البحث المستخدمة
                - corrected_query: الاستعلام بعد التصحيح
                - total_found: عدد النتائج
        """
        logger.info(f"🔍 Unified search for: '{query}'")

        # 1. تنظيف النص (إزالة كلمات استفهامية) واستخراج المصطلح الأساسي
        core_term = clean_user_query(query)

        if core_term != query:
            logger.info(f"🧹 Cleaned query: '{query}' → '{core_term}'")

        # 1.5. إزالة "ال" التعريف من بداية الكلمات
        core_term_without_al = self._remove_al_prefix(core_term)

        if core_term_without_al != core_term:
            logger.info(f"🧹 Removed 'ال' prefix: '{core_term}' → '{core_term_without_al}'")
            core_term = core_term_without_al

        # 2. تصحيح الأخطاء الإملائية
        corrected_term, corrections = self.word_corrector.correct_text(core_term)

        if corrections:
            logger.info(f"🔧 Corrected query: '{core_term}' → '{corrected_term}'")
            logger.info(f"   Corrections: {[c['original'] + '→' + c['corrected'] for c in corrections]}")

        # ========================================================================
        # A. EXACT GATE (إلزامي) - البحث بالتطابق التام
        # ========================================================================
        exact_results = self._exact_match_search(corrected_term)

        if exact_results:
            logger.info(f"🎯 Exact Gate: Found {len(exact_results)} exact matches")
            # ترتيب النتائج الدقيقة حسب الأولوية
            exact_results = self._rank_exact_results(corrected_term, exact_results)

            # تطبيق الحد الأقصى
            final_results = exact_results[:limit]

            return {
                'results': final_results,
                'search_methods': [f"exact({len(exact_results)})"],
                'corrected_query': corrected_term if corrections else query,
                'total_found': len(final_results),
                'exact_count': len(exact_results),
                'partial_count': 0,
                'semantic_count': 0
            }

        # ========================================================================
        # B. CATEGORY GUARD - فلتر الفئات
        # ========================================================================
        category_keywords = self._detect_category_keywords(corrected_term)

        # 4. البحث بالتطابق الجزئي (Partial Match)
        partial_results = self._partial_match_search(corrected_term)

        # تطبيق Category Guard
        if partial_results and not category_keywords:
            partial_results = self._apply_category_guard(corrected_term, partial_results)

        if partial_results:
            logger.info(f"🔍 Partial Match: Found {len(partial_results)} results")
            final_results = partial_results[:limit]

            return {
                'results': final_results,
                'search_methods': [f"partial({len(partial_results)})"],
                'corrected_query': corrected_term if corrections else query,
                'total_found': len(final_results),
                'exact_count': 0,
                'partial_count': len(partial_results),
                'semantic_count': 0
            }

        # ========================================================================
        # C. SEMANTIC FALLBACK - البحث الدلالي (آخر خيار)
        # ========================================================================
        semantic_results = self._semantic_search(corrected_term, min_similarity)

        # فلتر النتائج الدلالية بالمصطلح الأساسي
        if semantic_results:
            filtered_semantic = self._filter_semantic_by_core_term(corrected_term, semantic_results)

            if filtered_semantic:
                logger.info(f"🧠 Semantic Fallback (filtered): Found {len(filtered_semantic)} results")
                final_results = filtered_semantic[:limit]
            else:
                logger.warning(f"⚠️ Semantic results filtered out, returning unfiltered")
                final_results = semantic_results[:limit]

            return {
                'results': final_results,
                'search_methods': [f"semantic({len(semantic_results)})"],
                'corrected_query': corrected_term if corrections else query,
                'total_found': len(final_results),
                'exact_count': 0,
                'partial_count': 0,
                'semantic_count': len(semantic_results)
            }

        # لم يتم العثور على أي نتائج
        logger.warning(f"⚠️ No results found for: '{query}'")

        return {
            'results': [],
            'search_methods': [],
            'corrected_query': corrected_term if corrections else query,
            'total_found': 0,
            'exact_count': 0,
            'partial_count': 0,
            'semantic_count': 0
        }

    def _remove_al_prefix(self, text: str) -> str:
        """
        إزالة "ال" التعريف من بداية الكلمات

        Args:
            text: النص المراد معالجته

        Returns:
            النص بعد إزالة "ال" التعريف
        """
        if not text:
            return text

        words = text.split()
        cleaned_words = []

        for word in words:
            # إزالة "ال" من بداية الكلمة إذا كانت أطول من حرفين
            if word.startswith('ال') and len(word) > 2:
                cleaned_words.append(word[2:])
            else:
                cleaned_words.append(word)

        return ' '.join(cleaned_words)

    def _exact_match_search(self, query: str) -> List[Dict]:
        """
        البحث بالتطابق التام (Exact Match)
        يبحث عن تطابق تام في أسماء الأصناف

        Args:
            query: استعلام البحث

        Returns:
            قائمة الأصناف المطابقة تماماً
        """
        try:
            query_lower = query.lower().strip()

            # البحث في الكاش بدلاً من قاعدة البيانات
            results = []
            for item in self._menu_cache:
                if not item.get('active', True):
                    continue

                name_ar = item.get('name_ar', '').lower()
                name_en = item.get('name_en', '').lower()

                if query_lower == name_ar or query_lower == name_en:
                    item_copy = item.copy()
                    item_copy['match_type'] = 'exact'
                    item_copy['match_score'] = 1.0
                    results.append(item_copy)

            if results:
                logger.info(f"🎯 Exact match found: {len(results)} item(s)")
                for item in results:
                    logger.debug(f"   - {item['name_ar']} / {item['name_en']}")

            return results

        except Exception as e:
            logger.error(f"❌ Error in exact match search: {e}")
            return []

    def _partial_match_search(self, query: str) -> List[Dict]:
        """
        البحث بالتطابق الجزئي (Partial Match)
        يبحث عن الأصناف التي تحتوي الكلمة جزئياً

        Args:
            query: استعلام البحث

        Returns:
            قائمة الأصناف المطابقة جزئياً
        """
        try:
            query_lower = query.lower().strip()

            # البحث في الكاش بدلاً من قاعدة البيانات
            results = []
            for item in self._menu_cache:
                if not item.get('active', True):
                    continue

                name_ar = item.get('name_ar', '').lower()
                name_en = item.get('name_en', '').lower()
                description = item.get('description', '').lower()
                tags = [tag.lower() for tag in item.get('tags', [])]

                # البحث في الاسم أو الوصف أو الوسوم
                if (query_lower in name_ar or
                    query_lower in name_en or
                    query_lower in description or
                    any(query_lower in tag for tag in tags)):

                    item_copy = item.copy()
                    item_copy['match_type'] = 'partial'
                    item_copy['match_score'] = self._calculate_partial_score(query_lower, item)
                    results.append(item_copy)

            if results:
                logger.info(f"🔍 Partial match found: {len(results)} item(s)")
                for item in results:
                    logger.debug(f"   - {item['name_ar']} / {item['name_en']} (score: {item['match_score']:.2f})")

            return results

        except Exception as e:
            logger.error(f"❌ Error in partial match search: {e}")
            return []

    def _calculate_partial_score(self, query: str, item: Dict) -> float:
        """
        حساب نسبة التطابق الجزئي

        Args:
            query: استعلام البحث
            item: الصنف

        Returns:
            نسبة التطابق (0.3-0.9)
        """
        score = 0.3  # النقاط الأساسية (مخفضة من 0.5)

        name_ar = item.get('name_ar', '').lower()
        name_en = item.get('name_en', '').lower()
        name_ar_words = name_ar.split()
        name_en_words = name_en.split()

        # أعلى أولوية: الكلمة في بداية الاسم
        if name_ar.startswith(query) or name_en.startswith(query):
            score += 0.6  # زيادة من 0.3 إلى 0.6
        # أولوية عالية: الكلمة كاملة في الاسم
        elif query in name_ar_words or query in name_en_words:
            score += 0.5  # جديد
        # أولوية متوسطة: الكلمة جزء من كلمة في الاسم
        elif query in name_ar or query in name_en:
            score += 0.3  # زيادة من 0.2 إلى 0.3

        # أولوية منخفضة: الكلمة في الوصف فقط
        description_ar = (item.get('description_ar') or '').lower()
        description_en = (item.get('description_en') or '').lower()
        if query in description_ar or query in description_en:
            # لا نضيف نقاط إضافية، فقط النقاط الأساسية
            pass

        # أولوية منخفضة: الكلمة في الوسوم
        tags = item.get('tags', [])
        if tags and query in [tag.lower() for tag in tags]:
            score += 0.1

        return min(score, 0.9)  # الحد الأقصى 0.9
    
    def _semantic_search(self, query: str, min_similarity: float = 0.3) -> List[Dict]:
        """
        البحث الدلالي باستخدام FAISS
        يبحث عن أصناف مشابهة بالمعنى

        Args:
            query: استعلام البحث
            min_similarity: الحد الأدنى للتشابه

        Returns:
            قائمة الأصناف المشابهة دلالياً
        """
        try:
            # استخدام FAISS للبحث الدلالي (sync wrapper for async function)
            import asyncio

            # Check if there's a running event loop
            try:
                asyncio.get_running_loop()
                # If we're in an async context, we can't use asyncio.run()
                # For now, skip FAISS search in sync context
                logger.warning("⚠️ Skipping FAISS search in sync context")
                return []
            except RuntimeError:
                # No running loop, we can use asyncio.run()
                faiss_results = asyncio.run(
                    self.faiss_search.search(
                        query=query,
                        top_k=10,
                        min_score=min_similarity,
                        keyword_filter=True,  # ← تفعيل فلتر الكلمات المفتاحية
                        exact_match_boost=True  # ← تفعيل تعزيز التطابق التام
                    )
                )

            if not faiss_results:
                logger.info(f"🤷 No semantic matches found")
                return []

            # تصفية النتائج حسب الحد الأدنى للتشابه
            filtered_results = []
            for result in faiss_results:
                if result.get('similarity', 0) >= min_similarity:
                    result['match_type'] = 'semantic'
                    result['match_score'] = result.get('similarity', 0)
                    filtered_results.append(result)

            if filtered_results:
                logger.info(f"🧠 Semantic match found: {len(filtered_results)} item(s)")
                for item in filtered_results:
                    logger.debug(f"   - {item.get('name_ar')} / {item.get('name_en')} (score: {item['match_score']:.2f})")

            return filtered_results

        except Exception as e:
            logger.error(f"❌ Error in semantic search: {e}")
            return []

    def _merge_and_rank_results(
        self,
        exact_results: List[Dict],
        partial_results: List[Dict],
        semantic_results: List[Dict],
        limit: int
    ) -> List[Dict]:
        """
        دمج وترتيب النتائج بالأولوية

        الأولوية:
        🥇 Exact Match (1.0)
        🥈 Partial Match (0.5-0.9)
        🥉 Semantic Match (0.3-0.9)

        Args:
            exact_results: نتائج التطابق التام
            partial_results: نتائج التطابق الجزئي
            semantic_results: نتائج البحث الدلالي
            limit: الحد الأقصى للنتائج

        Returns:
            قائمة النتائج المدمجة والمرتبة
        """
        # استخدام dict لتجنب التكرار (بناءً على code)
        merged_dict = {}

        # 1. إضافة Exact matches (أعلى أولوية)
        for item in exact_results:
            code = item.get('code')
            if code and code not in merged_dict:
                merged_dict[code] = item

        # 2. إضافة Partial matches (أولوية متوسطة)
        for item in partial_results:
            code = item.get('code')
            if code and code not in merged_dict:
                merged_dict[code] = item

        # 3. إضافة Semantic matches (أولوية منخفضة)
        for item in semantic_results:
            code = item.get('code')
            if code and code not in merged_dict:
                merged_dict[code] = item

        # تحويل إلى قائمة وترتيب حسب match_score
        merged_list = list(merged_dict.values())
        merged_list.sort(key=lambda x: x.get('match_score', 0), reverse=True)

        # تطبيق الحد الأقصى
        final_results = merged_list[:limit]

        logger.info(f"📊 Merged results: {len(final_results)} items")
        logger.info(f"   🥇 Exact: {len(exact_results)}")
        logger.info(f"   🥈 Partial: {len(partial_results)}")
        logger.info(f"   🥉 Semantic: {len(semantic_results)}")

        return final_results

    def _rank_exact_results(self, query: str, results: List[Dict]) -> List[Dict]:
        """
        ترتيب النتائج الدقيقة حسب الأولوية

        الأولوية:
        1. startswith(core_term)
        2. word_boundary_contains(core_term)
        3. alias match
        """
        query_lower = query.lower().strip()
        ranked = []

        for item in results:
            name_ar = (item.get('name_ar') or '').lower()
            name_en = (item.get('name_en') or '').lower()

            # Priority 1: Starts with query
            if name_ar.startswith(query_lower) or name_en.startswith(query_lower):
                item['match_score'] = 1.0
                item['match_priority'] = 1
            # Priority 2: Word boundary contains
            elif self._word_boundary_contains(name_ar, query_lower) or self._word_boundary_contains(name_en, query_lower):
                item['match_score'] = 0.95
                item['match_priority'] = 2
            # Priority 3: Contains anywhere
            else:
                item['match_score'] = 0.90
                item['match_priority'] = 3

            ranked.append(item)

        # Sort by priority then score
        ranked.sort(key=lambda x: (x.get('match_priority', 99), -x.get('match_score', 0)))

        return ranked

    def _word_boundary_contains(self, text: str, term: str) -> bool:
        """
        التحقق من وجود المصطلح كحدود كلمة كاملة
        """
        words = text.split()
        return term in words

    def _detect_category_keywords(self, query: str) -> Set[str]:
        """
        اكتشاف كلمات الفئات في الاستعلام
        """
        category_keywords = {
            'ar': ['شوربة', 'شوربات', 'سلطة', 'سلطات', 'مشروب', 'مشروبات', 'عصير', 'عصائر',
                   'مقبلات', 'حلويات', 'حلى', 'عروض', 'عرض'],
            'en': ['soup', 'soups', 'salad', 'salads', 'drink', 'drinks', 'juice', 'juices',
                   'appetizer', 'appetizers', 'dessert', 'desserts', 'offer', 'offers']
        }

        query_lower = query.lower()
        detected = set()

        for lang, keywords in category_keywords.items():
            for keyword in keywords:
                if keyword in query_lower:
                    detected.add(keyword)

        return detected

    def _apply_category_guard(self, query: str, results: List[Dict]) -> List[Dict]:
        """
        تطبيق Category Guard - استبعاد فئات غير متوقعة

        إذا كان الاستعلام لا يحتوي على كلمات فئة، استبعد الفئات غير المتوقعة
        """
        query_lower = query.lower()
        filtered = []

        # الفئات التي يجب استبعادها إذا لم تُذكر صراحة
        unexpected_categories = ['شوربات', 'سلطات', 'مشروبات', 'مقبلات', 'حلويات', 'عروض']

        for item in results:
            category = (item.get('category') or '').lower()
            name_ar = (item.get('name_ar') or '').lower()
            name_ar_words = name_ar.split()
            tags = [tag.lower() for tag in (item.get('tags') or [])]
            aliases = [alias.lower() for alias in (item.get('aliases') or [])]

            # إذا كان الصنف يبدأ بالمصطلح، احتفظ به (أولوية عالية)
            if name_ar.startswith(query_lower):
                filtered.append(item)
                continue

            # إذا كان المصطلح أول كلمة في الاسم، احتفظ به
            if name_ar_words and name_ar_words[0] == query_lower:
                filtered.append(item)
                continue

            # إذا كان المصطلح موجود في الوسوم أو المرادفات، احتفظ به
            if query_lower in tags or query_lower in aliases:
                filtered.append(item)
                continue

            # إذا كان المصطلح موجود في أي كلمة من اسم الصنف، احتفظ به
            if query_lower in name_ar_words:
                filtered.append(item)
                continue

            # إذا كانت الفئة غير متوقعة، استبعده
            if any(cat in category for cat in unexpected_categories):
                logger.debug(f"   Category Guard: Excluded '{item.get('name_ar')}' (category: {category})")
                continue

            filtered.append(item)

        return filtered

    def _filter_semantic_by_core_term(self, core_term: str, results: List[Dict]) -> List[Dict]:
        """
        فلتر النتائج الدلالية بالمصطلح الأساسي

        يحتفظ فقط بالنتائج التي تحتوي على المصطلح في:
        - الاسم (name_ar/name_en)
        - الوسوم (tags)
        """
        core_term_lower = core_term.lower().strip()
        filtered = []

        for item in results:
            name_ar = (item.get('name_ar') or '').lower()
            name_en = (item.get('name_en') or '').lower()
            tags = [tag.lower() for tag in (item.get('tags') or [])]

            # Check if core term appears in name or tags
            if (self._word_boundary_contains(name_ar, core_term_lower) or
                self._word_boundary_contains(name_en, core_term_lower) or
                core_term_lower in tags or
                core_term_lower in name_ar or
                core_term_lower in name_en):
                filtered.append(item)
            else:
                logger.debug(f"   Semantic Filter: Excluded '{item.get('name_ar')}' (no core term match)")

        return filtered

    def search_menu_simple(self, query: str) -> List[Dict]:
        """
        بحث بسيط يعيد قائمة النتائج فقط (للتوافق مع الكود القديم)

        Args:
            query: استعلام البحث

        Returns:
            قائمة النتائج
        """
        result = self.search_menu(query)
        return result.get('results', [])

    def format_results_for_display(self, results: List[Dict]) -> str:
        """
        تنسيق النتائج للعرض

        Args:
            results: قائمة النتائج

        Returns:
            نص منسق للعرض
        """
        if not results:
            return "لم يتم العثور على نتائج"

        output = []
        for i, item in enumerate(results, 1):
            name_ar = item.get('name_ar', 'N/A')
            name_en = item.get('name_en', 'N/A')
            price = item.get('price', 0)
            match_type = item.get('match_type', 'unknown')
            match_score = item.get('match_score', 0)

            # رمز نوع التطابق
            if match_type == 'exact':
                icon = '🎯'
            elif match_type == 'partial':
                icon = '🔍'
            else:
                icon = '🧠'

            output.append(
                f"{icon} {i}. {name_ar} / {name_en}\n"
                f"   💰 {price} ريال | {match_type} ({match_score:.2f})"
            )

        return "\n\n".join(output)


# Singleton instance
menu_search_service = MenuSearchService()

