"""
Local Menu Loader Service
Loads menu items from local JSON file instead of Supabase
"""
import json
import os
from typing import List, Dict, Optional
from functools import lru_cache
from app.utils.logger import app_logger as logger

# Path to local menu file
MENU_FILE_PATH = os.path.join(
    os.path.dirname(os.path.dirname(os.path.dirname(os.path.abspath(__file__)))),
    'data',
    'menu_items_local.json'
)

@lru_cache(maxsize=1)
def load_local_menu() -> List[Dict]:
    """
    Load menu items from local JSON file
    
    Returns:
        List of active menu items
        
    Raises:
        FileNotFoundError: If menu file doesn't exist
        json.JSONDecodeError: If menu file is invalid JSON
    """
    if not os.path.exists(MENU_FILE_PATH):
        raise FileNotFoundError(f"❌ ملف المنيو غير موجود في: {MENU_FILE_PATH}")
    
    try:
        with open(MENU_FILE_PATH, "r", encoding="utf-8") as f:
            data = json.load(f)
        
        # Filter only active items
        active_items = [item for item in data if item.get("active", True)]
        
        logger.info(f"✅ Loaded {len(active_items)} active menu items from local file")
        logger.info(f"📁 File path: {MENU_FILE_PATH}")
        
        return active_items
        
    except json.JSONDecodeError as e:
        logger.error(f"❌ Invalid JSON in menu file: {e}")
        raise
    except Exception as e:
        logger.error(f"❌ Error loading menu file: {e}")
        raise


def reload_menu() -> List[Dict]:
    """
    Force reload menu from file (clears cache)
    
    Returns:
        List of active menu items
    """
    load_local_menu.cache_clear()
    return load_local_menu()


def get_menu_item_by_code(code: str) -> Optional[Dict]:
    """
    Get a single menu item by code
    
    Args:
        code: Item code (e.g., "APP_HUMMUS")
        
    Returns:
        Menu item dict or None if not found
    """
    menu = load_local_menu()
    for item in menu:
        if item.get('code') == code:
            return item
    return None


def search_menu_items(query: str, lang: str = 'ar') -> List[Dict]:
    """
    Search menu items by name
    
    Args:
        query: Search query
        lang: Language ('ar' or 'en')
        
    Returns:
        List of matching menu items
    """
    menu = load_local_menu()
    query_lower = query.strip().lower()
    results = []
    
    name_field = 'name_ar' if lang == 'ar' else 'name_en'
    
    for item in menu:
        if query_lower in item.get(name_field, '').lower():
            results.append(item)
    
    return results


def get_menu_by_category(category: str = None) -> Dict[str, List[Dict]]:
    """
    Get menu items grouped by category
    
    Args:
        category: Optional category filter
        
    Returns:
        Dict with categories as keys and item lists as values
    """
    menu = load_local_menu()
    
    if category:
        # Filter by specific category
        items = [item for item in menu if item.get('category') == category]
        return {category: items}
    
    # Group by all categories
    categories = {}
    for item in menu:
        cat = item.get('category', 'غير مصنف')
        if cat not in categories:
            categories[cat] = []
        categories[cat].append(item)
    
    return categories


def get_menu_stats() -> Dict:
    """
    Get statistics about the menu
    
    Returns:
        Dict with menu statistics
    """
    menu = load_local_menu()
    categories = get_menu_by_category()
    
    total_items = len(menu)
    total_categories = len(categories)
    
    # Calculate price range
    prices = [item.get('price', 0) for item in menu]
    min_price = min(prices) if prices else 0
    max_price = max(prices) if prices else 0
    avg_price = sum(prices) / len(prices) if prices else 0
    
    return {
        'total_items': total_items,
        'total_categories': total_categories,
        'categories': list(categories.keys()),
        'price_range': {
            'min': min_price,
            'max': max_price,
            'avg': round(avg_price, 3)
        }
    }


# Test function
if __name__ == "__main__":
    print("🧪 Testing Local Menu Loader...")
    print("="*60)
    
    try:
        # Load menu
        menu = load_local_menu()
        print(f"✅ Loaded {len(menu)} items")
        
        # Show stats
        stats = get_menu_stats()
        print(f"\n📊 Menu Statistics:")
        print(f"   - Total Items: {stats['total_items']}")
        print(f"   - Categories: {stats['total_categories']}")
        print(f"   - Price Range: {stats['price_range']['min']:.3f} - {stats['price_range']['max']:.3f} OMR")
        print(f"   - Average Price: {stats['price_range']['avg']:.3f} OMR")
        
        # Show categories
        print(f"\n📋 Categories:")
        for cat in stats['categories']:
            print(f"   - {cat}")
        
        # Test search
        print(f"\n🔍 Search Test (حمص):")
        results = search_menu_items("حمص", "ar")
        for item in results:
            print(f"   - {item['name_ar']} ({item['code']}) - {item['price']:.3f} OMR")
        
        # Test get by code
        print(f"\n🔍 Get by Code Test (APP_HUMMUS):")
        item = get_menu_item_by_code("APP_HUMMUS")
        if item:
            print(f"   - {item['name_ar']} - {item['price']:.3f} OMR")
        
        print("\n" + "="*60)
        print("✅ All tests passed!")
        
    except Exception as e:
        print(f"❌ Error: {e}")
        import traceback
        traceback.print_exc()

