"""
Intent Service - الخدمة الرئيسية لمعالجة النوايا
"""

from typing import Dict, Optional, List
from app.intents.classifier import intent_classifier
from app.handlers.registry import handler_registry
from app.intents.definitions import IntentType
from app.utils.logger import app_logger as logger


class IntentService:
    """
    خدمة معالجة النوايا
    
    تدفق العمل:
    1. استخلاص النية من رسالة المستخدم
    2. اختيار المعالج المناسب
    3. تنفيذ المعالج
    4. إرجاع الرد
    """
    
    def __init__(self):
        self.classifier = intent_classifier
        self.registry = handler_registry
    
    async def process_message(
        self,
        user_message: str,
        context: Dict,
        conversation_history: Optional[List[Dict]] = None
    ) -> Dict:
        """
        معالجة رسالة المستخدم
        
        Args:
            user_message: رسالة المستخدم
            context: السياق (السلة، المرحلة، إلخ)
            conversation_history: تاريخ المحادثة
            
        Returns:
            Dict: الرد (intent, reply, action, data)
        """
        try:
            logger.info(f"📨 Processing message: '{user_message[:50]}...'")
            
            # Step 1: استخلاص النية
            intent_type = await self.classifier.classify_intent(
                user_message,
                context,
                conversation_history
            )
            
            logger.info(f"🎯 Detected intent: {intent_type.value}")
            
            # Step 2: الحصول على المعالج
            handler = self.registry.get_handler(intent_type)
            
            if not handler:
                logger.warning(f"⚠️ No handler found for intent: {intent_type.value}")
                return self._fallback_response(user_message, context)
            
            # Step 3: تنفيذ المعالج
            response = await handler.handle(
                user_message,
                context,
                conversation_history
            )
            
            logger.info(f"✅ Response generated: {response.get('intent')}")
            
            return response
            
        except Exception as e:
            logger.error(f"❌ Error processing message: {str(e)}")
            return self._error_response(context)
    
    def _fallback_response(self, user_message: str, context: Dict) -> Dict:
        """رد احتياطي عندما لا يوجد معالج"""
        
        lang = context.get('lang', 'ar')
        
        if lang == 'ar':
            reply = "عذراً، لم أفهم طلبك. هل يمكنك إعادة الصياغة؟"
        else:
            reply = "Sorry, I didn't understand your request. Can you rephrase?"
        
        return {
            "intent": "Ask Information",
            "reply": reply,
            "action": None,
            "data": None
        }
    
    def _error_response(self, context: Dict) -> Dict:
        """رد عند حدوث خطأ"""
        
        lang = context.get('lang', 'ar')
        
        if lang == 'ar':
            reply = "عذراً، حدث خطأ. من فضلك حاول مرة أخرى."
        else:
            reply = "Sorry, an error occurred. Please try again."
        
        return {
            "intent": "Ask Information",
            "reply": reply,
            "action": None,
            "data": None
        }


# Singleton instance
intent_service = IntentService()

