"""
Input Processor - معالج الإدخال الموحد
نقطة دخول موحدة لمعالجة جميع أنواع المدخلات

🟦 LAYER 1: INPUT LAYER
"""

from typing import Dict, Optional, Tuple
from dataclasses import dataclass
from app.utils.enhanced_language_detector import enhanced_language_detector
from app.services.word_correction_dict import word_correction_dict
from app.utils.text_helpers import normalize_text
from app.utils.logger import app_logger as logger


@dataclass
class ProcessedInput:
    """نتيجة معالجة الإدخال"""
    
    # النص الأصلي
    original_text: str
    
    # النص المعالج
    cleaned_text: str
    corrected_text: str
    normalized_text: str
    
    # معلومات اللغة
    language: str
    dialect: Optional[str]
    language_confidence: float
    
    # معلومات المستخدم
    chat_id: str
    phone_number: str
    
    # معلومات الرسالة
    message_type: str  # text, audio, location
    message_id: str
    
    # إحصائيات
    processing_time_ms: float
    corrections_made: int
    
    def to_dict(self) -> Dict:
        """تحويل إلى قاموس"""
        return {
            'original_text': self.original_text,
            'cleaned_text': self.cleaned_text,
            'corrected_text': self.corrected_text,
            'normalized_text': self.normalized_text,
            'language': self.language,
            'dialect': self.dialect,
            'language_confidence': self.language_confidence,
            'chat_id': self.chat_id,
            'phone_number': self.phone_number,
            'message_type': self.message_type,
            'message_id': self.message_id,
            'processing_time_ms': self.processing_time_ms,
            'corrections_made': self.corrections_made
        }


class InputProcessor:
    """
    معالج الإدخال الموحد
    
    المسؤوليات:
    1. استقبال الرسالة الخام
    2. كشف اللغة واللهجة
    3. تصحيح الأخطاء الإملائية
    4. تنظيف وتطبيع النص
    5. استخراج المعلومات الأساسية
    
    الاستخدام:
        processor = InputProcessor()
        result = await processor.process(
            message_text="عندكم هرس دجاج؟",
            chat_id="96812345678@s.whatsapp.net",
            phone_number="96812345678",
            message_id="msg_123",
            session_lang="ar"
        )
    """
    
    def __init__(self):
        self.language_detector = enhanced_language_detector
    
    async def process(
        self,
        message_text: str,
        chat_id: str,
        phone_number: str,
        message_id: str,
        message_type: str = "text",
        session_lang: Optional[str] = None
    ) -> ProcessedInput:
        """
        معالجة الإدخال الكامل
        
        Args:
            message_text: نص الرسالة
            chat_id: معرف المحادثة
            phone_number: رقم الهاتف
            message_id: معرف الرسالة
            message_type: نوع الرسالة (text, audio, location)
            session_lang: لغة الجلسة (اختياري)
            
        Returns:
            ProcessedInput: نتيجة المعالجة
        """
        import time
        start_time = time.time()
        
        try:
            logger.info(f"🟦 INPUT LAYER: Processing message from {phone_number}")
            logger.debug(f"   Original: '{message_text[:50]}...'")
            
            # 1. كشف اللغة واللهجة
            language, confidence = self.language_detector.detect(
                message_text,
                user_id=chat_id,
                session_lang=session_lang
            )
            
            dialect = self.language_detector.detect_dialect(message_text)
            
            logger.info(f"   Language: {language} ({dialect}) - {confidence:.2f}")
            
            # 2. تصحيح الأخطاء الإملائية
            corrected_text, corrections_count = self._correct_text(message_text)
            
            if corrections_count > 0:
                logger.info(f"   Corrections: {corrections_count} words fixed")
                logger.debug(f"   Corrected: '{corrected_text[:50]}...'")
            
            # 3. تنظيف وتطبيع النص
            cleaned_text = self._clean_text(corrected_text)
            normalized_text = self._normalize_text(cleaned_text)
            
            logger.debug(f"   Normalized: '{normalized_text[:50]}...'")
            
            # 4. حساب الوقت
            processing_time = (time.time() - start_time) * 1000  # ms
            
            logger.info(f"   ✅ Processed in {processing_time:.1f}ms")
            
            # 5. بناء النتيجة
            result = ProcessedInput(
                original_text=message_text,
                cleaned_text=cleaned_text,
                corrected_text=corrected_text,
                normalized_text=normalized_text,
                language=language,
                dialect=dialect,
                language_confidence=confidence,
                chat_id=chat_id,
                phone_number=phone_number,
                message_type=message_type,
                message_id=message_id,
                processing_time_ms=processing_time,
                corrections_made=corrections_count
            )
            
            return result
            
        except Exception as e:
            logger.error(f"❌ Error processing input: {str(e)}")
            
            # إرجاع نتيجة أساسية في حالة الخطأ
            processing_time = (time.time() - start_time) * 1000
            
            return ProcessedInput(
                original_text=message_text,
                cleaned_text=message_text,
                corrected_text=message_text,
                normalized_text=message_text,
                language=session_lang or "ar",
                dialect=None,
                language_confidence=0.5,
                chat_id=chat_id,
                phone_number=phone_number,
                message_type=message_type,
                message_id=message_id,
                processing_time_ms=processing_time,
                corrections_made=0
            )
    
    def _correct_text(self, text: str) -> Tuple[str, int]:
        """
        تصحيح الأخطاء الإملائية

        Returns:
            (corrected_text, corrections_count)
        """
        try:
            original_words = text.split()
            corrected_text = word_correction_dict.correct_query(text)
            corrected_words = corrected_text.split()

            # حساب عدد التصحيحات
            corrections = sum(
                1 for orig, corr in zip(original_words, corrected_words)
                if orig != corr
            )

            return corrected_text, corrections

        except Exception as e:
            logger.warning(f"Error in text correction: {str(e)}")
            return text, 0
    
    def _clean_text(self, text: str) -> str:
        """
        تنظيف النص الأساسي
        
        - إزالة المسافات الزائدة
        - إزالة الرموز الخاصة غير المرغوبة
        - الحفاظ على علامات الترقيم المهمة
        """
        import re
        
        # إزالة المسافات الزائدة
        text = ' '.join(text.split())
        
        # إزالة الرموز الخاصة (ماعدا علامات الترقيم المهمة)
        # نحتفظ بـ: ؟ ! . , - ( )
        text = re.sub(r'[^\w\s؟!.,\-()أ-ي]', '', text)
        
        return text.strip()
    
    def _normalize_text(self, text: str) -> str:
        """
        تطبيع النص العربي

        - إزالة التشكيل
        - توحيد الأحرف (أ، إ، آ → ا)
        - توحيد التاء المربوطة (ة → ه)
        """
        try:
            return normalize_text(text)
        except Exception as e:
            logger.warning(f"Error in text normalization: {str(e)}")
            return text
    
    def extract_phone_number(self, chat_id: str) -> str:
        """
        استخراج رقم الهاتف من chat_id
        
        Args:
            chat_id: "96812345678@s.whatsapp.net"
            
        Returns:
            "96812345678"
        """
        return chat_id.split('@')[0] if '@' in chat_id else chat_id
    
    def is_valid_message(self, message_text: str) -> bool:
        """
        التحقق من صحة الرسالة
        
        Returns:
            True إذا كانت الرسالة صالحة للمعالجة
        """
        if not message_text:
            return False
        
        # الرسالة يجب أن تحتوي على حرف واحد على الأقل
        if not any(c.isalnum() for c in message_text):
            return False
        
        # الرسالة لا يجب أن تكون طويلة جداً (>1000 حرف)
        if len(message_text) > 1000:
            logger.warning(f"Message too long: {len(message_text)} chars")
            return False
        
        return True


# Singleton instance
input_processor = InputProcessor()


# ============================================================================
# Helper Functions
# ============================================================================

async def process_user_input(
    message_text: str,
    chat_id: str,
    phone_number: str,
    message_id: str,
    message_type: str = "text",
    session_lang: Optional[str] = None
) -> ProcessedInput:
    """
    دالة مساعدة لمعالجة الإدخال
    
    الاستخدام:
        result = await process_user_input(
            message_text="عندكم هريس؟",
            chat_id="96812345678@s.whatsapp.net",
            phone_number="96812345678",
            message_id="msg_123"
        )
    """
    return await input_processor.process(
        message_text=message_text,
        chat_id=chat_id,
        phone_number=phone_number,
        message_id=message_id,
        message_type=message_type,
        session_lang=session_lang
    )

