"""
Evolution API Service
Handles communication with Evolution API for WhatsApp
"""
import httpx
from typing import Dict, Optional
from app.config import settings
from app.utils.logger import app_logger as logger


class EvolutionAPIService:
    """Service for interacting with Evolution API"""

    # قائمة الأرقام المحظورة (لن يتم إرسال رسائل لها)
    BLOCKED_NUMBERS = [
        # "201002051593",  # تم إلغاء الحظر للاختبار
        # "201002051593@s.whatsapp.net",
        # "01002051593"
    ]

    def __init__(self):
        self.base_url = settings.EVOLUTION_API_URL
        self.api_key = settings.EVOLUTION_API_KEY
        self.instance_name = settings.EVOLUTION_INSTANCE_NAME
        self.headers = {
            "apikey": self.api_key,
            "Content-Type": "application/json"
        }

    def _is_blocked(self, phone_number: str) -> bool:
        """
        التحقق من أن الرقم محظور

        Args:
            phone_number: رقم الهاتف

        Returns:
            True إذا كان الرقم محظور
        """
        # تنظيف الرقم
        clean_number = phone_number.replace("@s.whatsapp.net", "").replace("+", "").replace(" ", "")

        for blocked in self.BLOCKED_NUMBERS:
            blocked_clean = blocked.replace("@s.whatsapp.net", "").replace("+", "").replace(" ", "")
            if clean_number == blocked_clean:
                logger.warning(f"🚫 Blocked number: {phone_number} - Message will not be sent")
                return True

        return False
    
    async def send_text_message(
        self,
        phone_number: str,
        message: str
    ) -> Dict:
        """
        Send text message via Evolution API

        Args:
            phone_number: Recipient phone number (with country code)
            message: Message text

        Returns:
            API response dictionary
        """
        # التحقق من الحظر
        if self._is_blocked(phone_number):
            logger.warning(f"🚫 Message blocked for {phone_number}")
            return {"status": "blocked", "message": "Number is blocked"}

        try:
            url = f"{self.base_url}/message/sendText/{self.instance_name}"

            payload = {
                "number": phone_number,
                "text": message
            }

            # 🔍 Debug: Log API call details
            logger.debug(f"📡 Calling Evolution API:")
            logger.debug(f"   - URL: {url}")
            logger.debug(f"   - Phone: {phone_number}")
            logger.debug(f"   - Message preview: {message[:100]}...")

            async with httpx.AsyncClient() as client:
                response = await client.post(
                    url,
                    json=payload,
                    headers=self.headers,
                    timeout=30.0
                )

                response.raise_for_status()
                result = response.json()

                logger.info(f"✅ Message sent to {phone_number}")
                logger.debug(f"   - Response: {result}")
                return result
                
        except httpx.HTTPError as e:
            logger.error(f"HTTP error sending message: {str(e)}")
            logger.error(f"Response status: {e.response.status_code if hasattr(e, 'response') else 'N/A'}")
            logger.error(f"Response body: {e.response.text if hasattr(e, 'response') else 'N/A'}")
            raise
        except Exception as e:
            logger.error(f"Error sending message: {str(e)}")
            logger.error(f"Error type: {type(e).__name__}")
            raise
    
    async def send_media_message(
        self,
        phone_number: str,
        media_url: str,
        caption: Optional[str] = None,
        media_type: str = "image"
    ) -> Dict:
        """
        Send media message (image, video, document)

        Args:
            phone_number: Recipient phone number
            media_url: URL of the media file
            caption: Optional caption
            media_type: Type of media (image, video, document)

        Returns:
            API response dictionary
        """
        # التحقق من الحظر
        if self._is_blocked(phone_number):
            logger.warning(f"🚫 Media message blocked for {phone_number}")
            return {"status": "blocked", "message": "Number is blocked"}

        try:
            url = f"{self.base_url}/message/sendMedia/{self.instance_name}"
            
            payload = {
                "number": phone_number,
                "mediatype": media_type,
                "media": media_url
            }
            
            if caption:
                payload["caption"] = caption
            
            async with httpx.AsyncClient() as client:
                response = await client.post(
                    url,
                    json=payload,
                    headers=self.headers,
                    timeout=30.0
                )
                
                response.raise_for_status()
                result = response.json()
                
                logger.info(f"Media message sent to {phone_number}")
                return result
                
        except Exception as e:
            logger.error(f"Error sending media message: {e}")
            raise
    
    async def send_buttons_message(
        self,
        phone_number: str,
        message: str,
        buttons: list
    ) -> Dict:
        """
        Send message with buttons

        Args:
            phone_number: Recipient phone number
            message: Message text
            buttons: List of button dictionaries

        Returns:
            API response dictionary
        """
        # التحقق من الحظر
        if self._is_blocked(phone_number):
            logger.warning(f"🚫 Buttons message blocked for {phone_number}")
            return {"status": "blocked", "message": "Number is blocked"}

        try:
            url = f"{self.base_url}/message/sendButtons/{self.instance_name}"
            
            payload = {
                "number": phone_number,
                "text": message,
                "buttons": buttons
            }
            
            async with httpx.AsyncClient() as client:
                response = await client.post(
                    url,
                    json=payload,
                    headers=self.headers,
                    timeout=30.0
                )
                
                response.raise_for_status()
                result = response.json()
                
                logger.info(f"Buttons message sent to {phone_number}")
                return result
                
        except Exception as e:
            logger.error(f"Error sending buttons message: {e}")
            raise
    
    async def get_instance_status(self) -> Dict:
        """
        Get instance connection status
        
        Returns:
            Instance status dictionary
        """
        try:
            url = f"{self.base_url}/instance/connectionState/{self.instance_name}"
            
            async with httpx.AsyncClient() as client:
                response = await client.get(
                    url,
                    headers=self.headers,
                    timeout=10.0
                )
                
                response.raise_for_status()
                result = response.json()
                
                logger.info(f"Instance status: {result.get('state')}")
                return result
                
        except Exception as e:
            logger.error(f"Error getting instance status: {e}")
            raise
    
    async def mark_message_as_read(
        self,
        phone_number: str,
        message_id: str
    ) -> Dict:
        """
        Mark message as read
        
        Args:
            phone_number: Sender phone number
            message_id: Message ID to mark as read
            
        Returns:
            API response dictionary
        """
        try:
            url = f"{self.base_url}/chat/markMessageAsRead/{self.instance_name}"
            
            payload = {
                "remoteJid": phone_number,
                "id": message_id
            }
            
            async with httpx.AsyncClient() as client:
                response = await client.post(
                    url,
                    json=payload,
                    headers=self.headers,
                    timeout=10.0
                )
                
                response.raise_for_status()
                result = response.json()
                
                logger.debug(f"Message marked as read: {message_id}")
                return result
                
        except Exception as e:
            logger.error(f"Error marking message as read: {e}")
            # Don't raise - this is not critical
            return {}
    
    async def send_poll(
        self,
        phone_number: str,
        name: str,
        options: list,
        selectable_count: int = 1
    ) -> Dict:
        """
        Send poll message (works on web and mobile)

        Args:
            phone_number: Recipient phone number
            name: Poll question/title
            options: List of option strings (max 12 options)
            selectable_count: Number of options user can select (default: 1)

        Returns:
            API response dictionary
        """
        # التحقق من الحظر
        if self._is_blocked(phone_number):
            logger.warning(f"🚫 Poll blocked for {phone_number}")
            return {"status": "blocked", "message": "Number is blocked"}

        try:
            url = f"{self.base_url}/message/sendPoll/{self.instance_name}"

            payload = {
                "number": phone_number,
                "name": name,
                "selectableCount": selectable_count,
                "values": options
            }

            async with httpx.AsyncClient() as client:
                response = await client.post(
                    url,
                    json=payload,
                    headers=self.headers,
                    timeout=30.0
                )

                response.raise_for_status()
                result = response.json()

                logger.info(f"Poll sent to {phone_number}")
                return result

        except httpx.HTTPError as e:
            logger.error(f"HTTP error sending poll: {e}")
            raise
        except Exception as e:
            logger.error(f"Error sending poll: {e}")
            raise

    async def send_list_message(
        self,
        phone_number: str,
        title: str,
        description: str,
        button_text: str,
        sections: list,
        footer: str = ""
    ) -> Dict:
        """
        Send message with list (works on web and mobile)

        Args:
            phone_number: Recipient phone number
            title: List message title
            description: List message description
            button_text: Text on the button to open list
            sections: List of sections with rows
                Each section: {
                    "title": "Section Title",
                    "rows": [
                        {
                            "title": "Row Title",
                            "description": "Row Description",
                            "rowId": "row_id"
                        }
                    ]
                }
            footer: Optional footer text

        Returns:
            API response dictionary
        """
        # التحقق من الحظر
        if self._is_blocked(phone_number):
            logger.warning(f"🚫 List blocked for {phone_number}")
            return {"status": "blocked", "message": "Number is blocked"}

        try:
            url = f"{self.base_url}/message/sendList/{self.instance_name}"

            payload = {
                "number": phone_number,
                "title": title,
                "description": description,
                "buttonText": button_text,
                "footerText": footer,
                "sections": sections
            }

            async with httpx.AsyncClient() as client:
                response = await client.post(
                    url,
                    json=payload,
                    headers=self.headers,
                    timeout=30.0
                )

                response.raise_for_status()
                result = response.json()

                logger.info(f"List sent to {phone_number}")
                return result

        except httpx.HTTPError as e:
            logger.error(f"HTTP error sending list: {e}")
            raise
        except Exception as e:
            logger.error(f"Error sending list: {e}")
            raise

    async def send_buttons(
        self,
        phone_number: str,
        title: str,
        description: str,
        buttons: list,
        footer: str = ""
    ) -> Dict:
        """
        Send message with buttons

        Args:
            phone_number: Recipient phone number
            title: Button message title
            description: Button message description
            buttons: List of button objects
                Each button: {
                    "type": "reply",  # or "url", "call", "copy"
                    "displayText": "Button Text",
                    "id": "button_id"  # for reply type
                    # or "url": "https://..." for url type
                    # or "phoneNumber": "..." for call type
                }
            footer: Optional footer text

        Returns:
            API response dictionary
        """
        # التحقق من الحظر
        if self._is_blocked(phone_number):
            logger.warning(f"🚫 Buttons blocked for {phone_number}")
            return {"status": "blocked", "message": "Number is blocked"}

        try:
            url = f"{self.base_url}/message/sendButtons/{self.instance_name}"

            payload = {
                "number": phone_number,
                "title": title,
                "description": description,
                "footer": footer,
                "buttons": buttons
            }

            async with httpx.AsyncClient() as client:
                response = await client.post(
                    url,
                    json=payload,
                    headers=self.headers,
                    timeout=30.0
                )

                response.raise_for_status()
                result = response.json()

                logger.info(f"Buttons sent to {phone_number}")
                return result

        except httpx.HTTPError as e:
            logger.error(f"HTTP error sending buttons: {e}")
            raise
        except Exception as e:
            logger.error(f"Error sending buttons: {e}")
            raise

    async def send_presence(
        self,
        phone_number: str,
        state: str = "composing"
    ) -> Dict:
        """
        Send presence/typing indicator

        Args:
            phone_number: Recipient phone number
            state: Presence state - "composing" (typing) or "available" (stop typing)

        Returns:
            API response dictionary
        """
        try:
            url = f"{self.base_url}/chat/sendPresence/{self.instance_name}"

            payload = {
                "number": phone_number,
                "state": state,
                "delay": 3000  # Show typing for 3 seconds
            }

            async with httpx.AsyncClient() as client:
                response = await client.post(
                    url,
                    json=payload,
                    headers=self.headers,
                    timeout=10.0
                )

                response.raise_for_status()
                result = response.json()

                logger.debug(f"Presence sent to {phone_number}: {state}")
                return result

        except Exception as e:
            logger.error(f"Error sending presence: {e}")
            # Don't raise - this is not critical
            return {}

    def extract_phone_number(self, remote_jid: str) -> str:
        """
        Extract clean phone number from remoteJid

        Args:
            remote_jid: Remote JID from webhook (e.g., "96812345678@s.whatsapp.net")

        Returns:
            Clean phone number
        """
        return remote_jid.split('@')[0]

    @staticmethod
    def create_reply_button(text: str, button_id: str) -> Dict:
        """
        Create a reply button

        Args:
            text: Button display text
            button_id: Button ID (will be sent back when clicked)

        Returns:
            Button dictionary
        """
        return {
            "type": "reply",
            "displayText": text,
            "id": button_id
        }

    @staticmethod
    def create_url_button(text: str, url: str) -> Dict:
        """
        Create a URL button

        Args:
            text: Button display text
            url: URL to open

        Returns:
            Button dictionary
        """
        return {
            "type": "url",
            "displayText": text,
            "url": url
        }

    @staticmethod
    def create_call_button(text: str, phone_number: str) -> Dict:
        """
        Create a call button

        Args:
            text: Button display text
            phone_number: Phone number to call

        Returns:
            Button dictionary
        """
        return {
            "type": "call",
            "displayText": text,
            "phoneNumber": phone_number
        }

    async def send_location(
        self,
        phone_number: str,
        latitude: float,
        longitude: float,
        name: str = "",
        address: str = "",
        caption: str = None
    ) -> Dict:
        """
        Send location message

        Args:
            phone_number: Recipient phone number (with or without @s.whatsapp.net)
            latitude: Location latitude
            longitude: Location longitude
            name: Location name (optional)
            address: Location address (optional)
            caption: Caption text to show with location (optional)

        Returns:
            API response dictionary
        """
        # التحقق من الحظر
        if self._is_blocked(phone_number):
            logger.warning(f"🚫 Location message blocked for {phone_number}")
            return {"status": "blocked", "message": "Number is blocked"}

        try:
            url = f"{self.base_url}/message/sendLocation/{self.instance_name}"

            # Clean phone number (remove @s.whatsapp.net if present)
            clean_number = phone_number.replace('@s.whatsapp.net', '')

            payload = {
                "number": clean_number,
                "latitude": latitude,
                "longitude": longitude,
                "name": name or "Delivery location",  # Evolution API requires name
                "address": address or "موقع التوصيل"  # Evolution API requires address
            }

            # Add caption if provided
            if caption:
                payload["caption"] = caption

            logger.info(f"📍 Sending location to {clean_number}: lat={latitude}, lon={longitude}, name={name}")
            if caption:
                logger.info(f"📝 With caption: {caption[:100]}...")
            logger.info(f"📍 Payload: {payload}")

            async with httpx.AsyncClient() as client:
                response = await client.post(
                    url,
                    json=payload,
                    headers=self.headers,
                    timeout=30.0
                )

                # Log response details
                logger.info(f"📍 Response status: {response.status_code}")
                logger.info(f"📍 Response body: {response.text}")

                response.raise_for_status()
                result = response.json()

                logger.info(f"📍 Location sent successfully to {clean_number}")
                return result

        except httpx.HTTPStatusError as e:
            logger.error(f"Error sending location: {e}")
            logger.error(f"Response body: {e.response.text}")
            raise
        except Exception as e:
            logger.error(f"Error sending location: {e}")
            raise

    def extract_location_from_message(self, message_data: Dict) -> Optional[Dict]:
        """
        Extract location data from WhatsApp message

        Args:
            message_data: Message data from webhook

        Returns:
            Dictionary with latitude, longitude, name, address or None
        """
        location_message = message_data.get('locationMessage')

        if not location_message:
            return None

        latitude = location_message.get('degreesLatitude')
        longitude = location_message.get('degreesLongitude')

        if latitude is None or longitude is None:
            logger.warning("Location message missing latitude or longitude")
            return None

        return {
            'latitude': float(latitude),
            'longitude': float(longitude),
            'name': location_message.get('name', ''),
            'address': location_message.get('address', '')
        }

    async def transcribe_audio(self, audio_message: dict, message_id: str) -> Optional[str]:
        """
        Transcribe audio message using OpenAI Whisper API

        Args:
            audio_message: Audio message data from Evolution API
            message_id: Message ID to download media

        Returns:
            Transcribed text or None if failed
        """
        try:
            from io import BytesIO
            from openai import AsyncOpenAI

            # Use Evolution API to download decrypted media
            download_url = f"{self.base_url}/chat/getBase64FromMediaMessage/{self.instance_name}"

            logger.info(f"Downloading decrypted audio via Evolution API...")

            # Download audio file using Evolution API
            try:
                async with httpx.AsyncClient(timeout=30.0) as client:
                    response = await client.post(
                        download_url,
                        json={
                            "message": {
                                "key": {
                                    "id": message_id
                                }
                            },
                            "convertToMp4": False
                        },
                        headers=self.headers
                    )
                    response.raise_for_status()
                    result = response.json()

                    # Log the full response for debugging
                    logger.info(f"Evolution API response keys: {result.keys() if isinstance(result, dict) else 'Not a dict'}")
                    logger.info(f"Evolution API full response: {result}")

                # Get base64 data - try different possible keys
                import base64
                audio_base64 = result.get('base64') or result.get('data') or result.get('base64Data')

                # If result itself is the base64 string
                if not audio_base64 and isinstance(result, str):
                    audio_base64 = result

                if not audio_base64:
                    logger.error(f"No base64 data in Evolution API response. Response structure: {result}")
                    return None

                # Decode base64 to bytes
                audio_bytes = base64.b64decode(audio_base64)
                logger.info(f"Downloaded and decoded audio: {len(audio_bytes)} bytes")

            except httpx.HTTPStatusError as e:
                logger.error(f"HTTP error from Evolution API: {e.response.status_code} - {e.response.text}")
                return None
            except Exception as e:
                logger.error(f"Failed to download audio via Evolution API: {e}")
                import traceback
                logger.error(f"Traceback: {traceback.format_exc()}")
                return None

            # Create BytesIO buffer with audio data
            audio_buffer = BytesIO(audio_bytes)
            audio_buffer.seek(0)  # Reset to beginning

            # Transcribe using OpenAI Whisper
            # IMPORTANT: Must pass as tuple (filename, file_object, mime_type)
            logger.info("Sending audio to Whisper API...")
            client = AsyncOpenAI(api_key=settings.OPENAI_API_KEY)

            # Prompt to help Whisper understand Gulf/Egyptian dialects
            dialect_prompt = """
            أريد أطلب، بغيت، عايز، أبي، بدي، شوية، كام، طلب، توصيل،
            مشاكيك، سمبوسة، مظبي، كبسة، شاورما، فلافل، حمص، فول
            """

            transcript = await client.audio.transcriptions.create(
                model="whisper-1",
                file=("audio.ogg", audio_buffer, "audio/ogg"),
                language="ar",  # Arabic
                prompt=dialect_prompt  # Help Whisper understand dialects
            )

            transcribed_text = transcript.text
            logger.info(f"✅ Audio transcribed successfully: {transcribed_text[:100]}...")
            return transcribed_text

        except Exception as e:
            logger.error(f"Error transcribing audio: {e}")
            return None


# Create a singleton instance
evolution_api = EvolutionAPIService()
