"""
Credits Manager Service
Handles credits/points tracking and management
"""

import logging
from typing import Dict, Optional
from datetime import datetime
from supabase import Client

logger = logging.getLogger(__name__)


class CreditsManager:
    """Manages system credits and message costs"""
    
    def __init__(self, supabase_client: Client):
        self.supabase = supabase_client
    
    async def get_credits_balance(self) -> Dict:
        """
        Get current credits balance
        
        Returns:
            Dict with credits information
        """
        try:
            response = self.supabase.table('system_credits') \
                .select('*') \
                .limit(1) \
                .execute()
            
            if response.data:
                credits = response.data[0]
                return {
                    "success": True,
                    "total_credits": credits.get('total_credits', 0),
                    "used_credits": credits.get('used_credits', 0),
                    "remaining_credits": credits.get('remaining_credits', 0),
                    "credits_per_message": float(credits.get('credits_per_message', 0.001)),
                    "last_refill_date": credits.get('last_refill_date'),
                    "updated_at": credits.get('updated_at')
                }
            else:
                logger.warning("⚠️ No credits record found")
                return {
                    "success": False,
                    "error": "No credits record found"
                }
                
        except Exception as e:
            logger.error(f"❌ Error getting credits balance: {str(e)}")
            return {
                "success": False,
                "error": str(e)
            }
    
    async def log_message_cost(
        self,
        chat_id: Optional[str] = None,
        customer_id: Optional[str] = None,
        message_id: Optional[str] = None,
        message_type: str = 'ai_request',
        tokens_used: int = 0,
        model_used: str = 'gpt-4o-mini'
    ) -> bool:
        """
        Log message cost to database
        
        Args:
            chat_id: Chat UUID (optional)
            customer_id: Customer UUID (optional)
            message_id: Message ID from WhatsApp
            message_type: Type of message (incoming, outgoing, ai_request)
            tokens_used: Number of tokens used
            model_used: AI model used
        
        Returns:
            True if successful, False otherwise
        """
        try:
            # Get current credits settings
            credits_response = self.supabase.table('system_credits') \
                .select('credits_per_message') \
                .limit(1) \
                .execute()
            
            if not credits_response.data:
                logger.warning("⚠️ No credits settings found")
                return False
            
            # Calculate cost
            cost_per_token = float(credits_response.data[0].get('credits_per_message', 0.001))
            credits_cost = tokens_used * cost_per_token
            
            # Insert message cost record
            self.supabase.table('message_costs').insert({
                'chat_id': chat_id,
                'customer_id': customer_id,
                'message_id': message_id,
                'message_type': message_type,
                'tokens_used': tokens_used,
                'credits_cost': credits_cost,
                'model_used': model_used
            }).execute()
            
            logger.info(f"💰 Logged message cost: {tokens_used} tokens = {credits_cost:.4f} credits")
            
            # Note: The trigger in database will automatically update system_credits
            
            return True
            
        except Exception as e:
            logger.error(f"❌ Error logging message cost: {str(e)}")
            return False
    
    async def refill_credits(self, amount: int, admin_id: Optional[str] = None, description: Optional[str] = None) -> Dict:
        """
        Add credits to the system
        
        Args:
            amount: Number of credits to add
            admin_id: Admin user ID who performed the refill
            description: Optional description
        
        Returns:
            Dict with result
        """
        try:
            # Get current balance
            balance_response = await self.get_credits_balance()
            
            if not balance_response.get('success'):
                return balance_response
            
            balance_before = balance_response.get('remaining_credits', 0)
            balance_after = balance_before + amount
            
            # Update system_credits
            self.supabase.table('system_credits') \
                .update({
                    'total_credits': self.supabase.table('system_credits').select('total_credits').execute().data[0]['total_credits'] + amount,
                    'remaining_credits': balance_after,
                    'last_refill_date': datetime.utcnow().isoformat(),
                    'updated_at': datetime.utcnow().isoformat()
                }) \
                .eq('id', self.supabase.table('system_credits').select('id').limit(1).execute().data[0]['id']) \
                .execute()
            
            # Log transaction
            self.supabase.table('credit_transactions').insert({
                'transaction_type': 'refill',
                'amount': amount,
                'balance_before': balance_before,
                'balance_after': balance_after,
                'description': description or f'Credits refill: +{amount}',
                'admin_id': admin_id
            }).execute()
            
            logger.info(f"✅ Credits refilled: +{amount} (new balance: {balance_after})")
            
            return {
                "success": True,
                "message": f"Successfully added {amount} credits",
                "balance_before": balance_before,
                "balance_after": balance_after
            }
            
        except Exception as e:
            logger.error(f"❌ Error refilling credits: {str(e)}")
            return {
                "success": False,
                "error": str(e)
            }
    
    async def get_credit_transactions(self, limit: int = 50, offset: int = 0) -> Dict:
        """
        Get credit transaction history
        
        Args:
            limit: Number of transactions to return
            offset: Offset for pagination
        
        Returns:
            Dict with transactions
        """
        try:
            response = self.supabase.table('credit_transactions') \
                .select('*') \
                .order('created_at', desc=True) \
                .range(offset, offset + limit - 1) \
                .execute()
            
            return {
                "success": True,
                "count": len(response.data),
                "transactions": response.data
            }
            
        except Exception as e:
            logger.error(f"❌ Error getting credit transactions: {str(e)}")
            return {
                "success": False,
                "error": str(e)
            }
    
    async def get_credits_stats(self, days: int = 30) -> Dict:
        """
        Get credits usage statistics
        
        Args:
            days: Number of days to analyze
        
        Returns:
            Dict with statistics
        """
        try:
            from datetime import timedelta
            
            # Get current balance
            balance = await self.get_credits_balance()
            
            # Get usage for the period
            start_date = (datetime.utcnow() - timedelta(days=days)).isoformat()
            
            usage_response = self.supabase.table('message_costs') \
                .select('credits_cost, tokens_used, message_type, created_at') \
                .gte('created_at', start_date) \
                .execute()
            
            total_cost = sum(float(msg.get('credits_cost', 0)) for msg in usage_response.data)
            total_tokens = sum(int(msg.get('tokens_used', 0)) for msg in usage_response.data)
            total_messages = len(usage_response.data)
            
            # Calculate daily average
            daily_average = total_cost / days if days > 0 else 0
            
            # Estimate days remaining
            remaining = balance.get('remaining_credits', 0)
            days_remaining = remaining / daily_average if daily_average > 0 else 999
            
            return {
                "success": True,
                "period_days": days,
                "total_credits": balance.get('total_credits', 0),
                "used_credits": balance.get('used_credits', 0),
                "remaining_credits": remaining,
                "period_usage": {
                    "total_cost": round(total_cost, 4),
                    "total_tokens": total_tokens,
                    "total_messages": total_messages,
                    "daily_average": round(daily_average, 4)
                },
                "estimates": {
                    "days_remaining": round(days_remaining, 1),
                    "estimated_depletion_date": (datetime.utcnow() + timedelta(days=days_remaining)).isoformat() if days_remaining < 999 else None
                }
            }
            
        except Exception as e:
            logger.error(f"❌ Error getting credits stats: {str(e)}")
            return {
                "success": False,
                "error": str(e)
            }

