"""
Conversation History Service
Manages conversation history for context-aware responses
"""
from typing import List, Dict, Optional
from app.utils.logger import app_logger as logger


class ConversationHistoryService:
    """Service for managing conversation history"""
    
    def __init__(self, supabase_client):
        self.supabase = supabase_client
    
    async def get_chat_uuid(self, chat_id: str) -> Optional[str]:
        """
        Get chat UUID from chat_id (WhatsApp JID)
        
        Args:
            chat_id: WhatsApp chat ID (e.g., "201556307784@s.whatsapp.net")
            
        Returns:
            Chat UUID or None if not found
        """
        try:
            # Find chat by chat_id directly
            response = self.supabase.table('chats') \
                .select('id') \
                .eq('chat_id', chat_id) \
                .limit(1) \
                .execute()

            if response.data:
                return response.data[0]['id']

            return None
            
        except Exception as e:
            logger.error(f"Error getting chat UUID: {e}")
            return None
    
    async def get_recent_messages(
        self,
        chat_id: str,
        limit: int = 20
    ) -> List[Dict]:
        """
        Get recent messages for a chat

        Args:
            chat_id: WhatsApp chat ID (e.g., "201556307784@s.whatsapp.net")
            limit: Maximum number of messages to retrieve (default: 20)

        Returns:
            List of message dictionaries with role and content
        """
        try:
            # Fetch recent messages from conversation_history table
            response = self.supabase.table('conversation_history') \
                .select('role, content, created_at') \
                .eq('chat_id', chat_id) \
                .order('created_at', desc=True) \
                .limit(limit) \
                .execute()

            if not response.data:
                logger.info(f"No conversation history found for {chat_id}")
                return []

            # Convert to conversation format (reverse to get chronological order)
            messages = []
            for msg in reversed(response.data):
                role = msg['role']
                content = msg.get('content', '').strip()

                if content:  # Only include non-empty messages
                    messages.append({
                        "role": role,
                        "content": content,
                        "timestamp": msg['created_at']
                    })

            logger.info(f"Retrieved {len(messages)} messages for chat {chat_id}")
            return messages

        except Exception as e:
            logger.error(f"Error getting recent messages: {e}")
            return []
    
    async def save_message(
        self,
        chat_id: str,
        text_content: str,
        from_me: bool,
        message_type: str = "text"
    ) -> bool:
        """
        Save a message to the conversation history

        Args:
            chat_id: WhatsApp chat ID
            text_content: Message text
            from_me: True if message is from bot, False if from customer
            message_type: Type of message (default: "text")

        Returns:
            True if successful, False otherwise
        """
        try:
            # Determine role
            role = "assistant" if from_me else "user"

            # Save message to conversation_history table
            self.supabase.table('conversation_history') \
                .insert({
                    'chat_id': chat_id,
                    'role': role,
                    'content': text_content
                }) \
                .execute()

            logger.debug(f"Saved message to history: role={role}, length={len(text_content)}")
            return True

        except Exception as e:
            logger.error(f"Error saving message: {e}")
            return False
    
    def format_conversation_history(self, messages: List[Dict]) -> str:
        """
        Format conversation history for inclusion in prompt

        Args:
            messages: List of message dictionaries

        Returns:
            Formatted conversation history string
        """
        if not messages:
            return ""

        history_text = "## 📜 Conversation History (Last messages):\n\n"

        for msg in messages:
            role_label = "🤖 Bot" if msg['role'] == "assistant" else "👤 Customer"
            history_text += f"{role_label}: {msg['content']}\n"

        history_text += "\n---\n\n"

        return history_text

    async def get_last_user_messages(
        self,
        chat_id: str,
        limit: int = 2,
        within_seconds: int = 10
    ) -> List[Dict]:
        """
        الحصول على آخر رسائل المستخدم خلال فترة زمنية محددة

        يستخدم لدمج الرسائل المتقطعة

        Args:
            chat_id: WhatsApp chat ID
            limit: عدد الرسائل المطلوبة
            within_seconds: الفترة الزمنية بالثواني

        Returns:
            قائمة الرسائل
        """
        try:
            from datetime import datetime, timedelta

            # حساب الوقت المحدد
            time_threshold = datetime.utcnow() - timedelta(seconds=within_seconds)

            # جلب آخر رسائل المستخدم فقط
            response = self.supabase.table('conversation_history') \
                .select('role, content, created_at') \
                .eq('chat_id', chat_id) \
                .eq('role', 'user') \
                .gte('created_at', time_threshold.isoformat()) \
                .order('created_at', desc=True) \
                .limit(limit) \
                .execute()

            if not response.data:
                return []

            # عكس الترتيب للحصول على الترتيب الزمني
            messages = []
            for msg in reversed(response.data):
                messages.append({
                    "role": msg['role'],
                    "content": msg.get('content', '').strip(),
                    "timestamp": msg['created_at']
                })

            return messages

        except Exception as e:
            logger.error(f"Error getting last user messages: {e}")
            return []

    def should_merge_messages(
        self,
        last_message: str,
        new_message: str
    ) -> bool:
        """
        التحقق من إمكانية دمج رسالتين

        Args:
            last_message: الرسالة السابقة
            new_message: الرسالة الجديدة

        Returns:
            True إذا يجب الدمج
        """
        from app.utils.text_helpers import combine_if_related

        # استخدام دالة الدمج من text_helpers
        merged = combine_if_related(last_message, new_message)

        return merged is not None

    async def get_merged_message(
        self,
        chat_id: str,
        current_message: str
    ) -> str:
        """
        الحصول على الرسالة المدمجة إذا كانت مرتبطة بالرسالة السابقة

        Args:
            chat_id: WhatsApp chat ID
            current_message: الرسالة الحالية

        Returns:
            الرسالة المدمجة أو الرسالة الحالية
        """
        try:
            # جلب آخر رسالتين من المستخدم خلال 10 ثوانٍ
            recent_messages = await self.get_last_user_messages(
                chat_id,
                limit=2,
                within_seconds=10
            )

            if not recent_messages:
                return current_message

            # الحصول على آخر رسالة
            last_message = recent_messages[-1]['content']

            # محاولة الدمج
            from app.utils.text_helpers import combine_if_related
            merged = combine_if_related(last_message, current_message)

            if merged:
                logger.info(f"🔗 Messages merged: '{last_message}' + '{current_message}' = '{merged}'")
                return merged

            return current_message

        except Exception as e:
            logger.error(f"Error merging messages: {e}")
            return current_message

