"""
Context Manager - Enhanced context and memory management
Provides smart context window management, semantic memory, and state tracking
"""
from typing import List, Dict, Optional, Tuple
from datetime import datetime, timedelta
from app.utils.logger import app_logger as logger


class ContextManager:
    """
    Enhanced context manager for better conversation understanding
    
    Features:
    1. Smart context window management (adaptive message selection)
    2. Semantic memory (remembers key facts from conversation)
    3. Session state tracking (tracks what user is doing)
    4. Context compression (reduces tokens while keeping meaning)
    """
    
    def __init__(self, supabase_client):
        self.supabase = supabase_client
        self.max_context_messages = 5  # Maximum messages to keep (optimized for speed)
        self.context_window_minutes = 30  # Consider messages from last 30 minutes

        # Item context memory (for price queries after item search)
        self._last_items_cache = {}  # {chat_id: {'item_name': str, 'timestamp': datetime}}
    
    async def build_enhanced_context(
        self,
        chat_id: str,
        current_message: str,
        session_context: Dict,
        conversation_history: List[Dict]
    ) -> Dict:
        """
        Build enhanced context with smart selection and compression
        
        Args:
            chat_id: Chat identifier
            current_message: Current user message
            session_context: Current session state
            conversation_history: Recent conversation history
            
        Returns:
            Enhanced context dictionary
        """
        try:
            # 1. Extract semantic memory (key facts from conversation)
            semantic_memory = self._extract_semantic_memory(
                conversation_history,
                session_context
            )
            
            # 2. Select relevant messages (smart filtering)
            relevant_messages = self._select_relevant_messages(
                conversation_history,
                current_message,
                session_context
            )
            
            # 3. Compress context (reduce tokens)
            compressed_context = self._compress_context(
                relevant_messages,
                semantic_memory
            )
            
            # 4. Track conversation state
            conversation_state = self._analyze_conversation_state(
                conversation_history,
                session_context,
                current_message
            )
            
            # 5. Build final context
            enhanced_context = {
                'semantic_memory': semantic_memory,
                'relevant_messages': compressed_context,
                'conversation_state': conversation_state,
                'session_context': session_context,
                'message_count': len(conversation_history),
                'relevant_count': len(relevant_messages)
            }
            
            logger.info(
                f"📚 Enhanced context: {len(relevant_messages)}/{len(conversation_history)} messages, "
                f"{len(semantic_memory)} facts"
            )
            
            return enhanced_context
            
        except Exception as e:
            logger.error(f"Error building enhanced context: {str(e)}")
            # Fallback to basic context
            return {
                'semantic_memory': {},
                'relevant_messages': conversation_history[-5:],  # Last 5 messages
                'conversation_state': {},
                'session_context': session_context
            }
    
    def _extract_semantic_memory(
        self,
        conversation_history: List[Dict],
        session_context: Dict
    ) -> Dict:
        """
        Extract key facts from conversation (semantic memory)
        
        Examples:
        - User mentioned they want delivery to specific address
        - User asked about specific menu items
        - User's preferences (e.g., "no onions", "extra spicy")
        """
        memory = {}
        
        # Extract from session context
        if session_context.get('items'):
            memory['current_order'] = {
                'items': session_context['items'],
                'subtotal': session_context.get('subtotal', 0),
                'item_count': len(session_context['items'])
            }
        
        if session_context.get('address'):
            memory['delivery_address'] = session_context['address']
        
        if session_context.get('payment_method'):
            memory['payment_method'] = session_context['payment_method']
        
        # Extract from recent conversation
        last_bot_message = None
        last_user_message = None
        
        for msg in reversed(conversation_history):
            if msg['role'] == 'assistant' and not last_bot_message:
                last_bot_message = msg['content']
            elif msg['role'] == 'user' and not last_user_message:
                last_user_message = msg['content']
            
            if last_bot_message and last_user_message:
                break
        
        if last_bot_message:
            memory['last_bot_message'] = last_bot_message[:200]  # First 200 chars
        
        if last_user_message:
            memory['last_user_message'] = last_user_message[:100]
        
        # Check if bot asked a question (expecting specific answer)
        if last_bot_message:
            if '؟' in last_bot_message or '?' in last_bot_message:
                memory['bot_asked_question'] = True
                
                # Check if bot presented options
                if any(marker in last_bot_message for marker in ['1.', '2.', '1-', '2-', '١.', '٢.']):
                    memory['bot_presented_options'] = True
        
        return memory
    
    def _select_relevant_messages(
        self,
        conversation_history: List[Dict],
        current_message: str,
        session_context: Dict
    ) -> List[Dict]:
        """
        Select most relevant messages for current context
        
        Strategy:
        1. Always include last 3 messages (immediate context)
        2. Include messages related to current stage
        3. Include messages with questions/answers
        4. Limit by time window (last 30 minutes)
        """
        if not conversation_history:
            return []
        
        relevant = []
        now = datetime.now()
        
        # Get current stage
        stage = session_context.get('stage', 'items')
        
        for msg in conversation_history:
            # Check time window
            try:
                msg_time = datetime.fromisoformat(msg.get('timestamp', '').replace('Z', '+00:00'))
                if (now - msg_time) > timedelta(minutes=self.context_window_minutes):
                    continue  # Too old
            except:
                pass  # If timestamp parsing fails, include the message
            
            # Always include recent messages
            if len(conversation_history) - conversation_history.index(msg) <= 3:
                relevant.append(msg)
                continue
            
            content = msg['content'].lower()
            
            # Include messages related to current stage
            stage_keywords = {
                'items': ['صنف', 'طلب', 'أريد', 'عندكم', 'menu', 'item', 'order'],
                'address': ['عنوان', 'موقع', 'توصيل', 'address', 'location', 'delivery'],
                'payment': ['دفع', 'نقد', 'بطاقة', 'payment', 'cash', 'card'],
                'confirmation': ['تأكيد', 'نعم', 'confirm', 'yes']
            }
            
            if any(keyword in content for keyword in stage_keywords.get(stage, [])):
                relevant.append(msg)
                continue
            
            # Include questions and answers
            if '؟' in content or '?' in content:
                relevant.append(msg)
                continue
        
        # Limit to max messages
        if len(relevant) > self.max_context_messages:
            # Keep first 2 and last (max-2) messages
            relevant = relevant[:2] + relevant[-(self.max_context_messages-2):]
        
        return relevant
    
    def _compress_context(
        self,
        messages: List[Dict],
        semantic_memory: Dict
    ) -> List[Dict]:
        """
        Compress context to reduce tokens while keeping meaning
        
        Strategy:
        1. Truncate very long messages
        2. Remove redundant information
        3. Keep last 5 messages uncompressed
        """
        if len(messages) <= 5:
            return messages  # Don't compress if few messages
        
        compressed = []
        
        for i, msg in enumerate(messages):
            # Keep last 5 messages uncompressed
            if i >= len(messages) - 5:
                compressed.append(msg)
                continue
            
            # Compress older messages
            content = msg['content']
            
            # Truncate if too long
            if len(content) > 150:
                content = content[:150] + "..."
            
            compressed.append({
                'role': msg['role'],
                'content': content,
                'timestamp': msg.get('timestamp')
            })
        
        return compressed
    
    def _analyze_conversation_state(
        self,
        conversation_history: List[Dict],
        session_context: Dict,
        current_message: str
    ) -> Dict:
        """
        Analyze current conversation state
        
        Returns:
            Dictionary with conversation state information
        """
        state = {
            'stage': session_context.get('stage', 'items'),
            'has_active_order': len(session_context.get('items', [])) > 0,
            'message_count': len(conversation_history),
            'last_interaction': None
        }
        
        # Detect if user is responding to a question
        if conversation_history:
            last_bot_msg = None
            for msg in reversed(conversation_history):
                if msg['role'] == 'assistant':
                    last_bot_msg = msg['content']
                    break
            
            if last_bot_msg:
                # Check if bot asked a question
                if '؟' in last_bot_msg or '?' in last_bot_msg:
                    state['responding_to_question'] = True
                    
                    # Check if bot presented options
                    if any(marker in last_bot_msg for marker in ['1.', '2.', '1-', '2-']):
                        state['selecting_from_options'] = True
        
        # Detect user intent from current message
        current_lower = current_message.lower()
        
        # Greetings
        if any(word in current_lower for word in ['سلام', 'مرحبا', 'هلا', 'hello', 'hi']):
            state['user_intent'] = 'greeting'
        
        # Numbers (selecting option)
        elif current_message.strip() in ['1', '2', '3', '4', '5', 'الاول', 'الثاني', 'الثالث']:
            state['user_intent'] = 'selecting_option'
        
        # Confirmation
        elif any(word in current_lower for word in ['نعم', 'أكيد', 'تمام', 'yes', 'ok']):
            state['user_intent'] = 'confirmation'
        
        # Cancellation
        elif any(word in current_lower for word in ['لا', 'إلغاء', 'cancel', 'no']):
            state['user_intent'] = 'cancellation'
        
        # Ordering
        elif any(word in current_lower for word in ['أريد', 'عندكم', 'طلب', 'want', 'order']):
            state['user_intent'] = 'ordering'
        
        return state
    
    def format_context_for_prompt(self, enhanced_context: Dict) -> str:
        """
        Format enhanced context for inclusion in prompt (COMPACT VERSION)

        Args:
            enhanced_context: Enhanced context dictionary

        Returns:
            Formatted string for prompt (compact, only critical info)
        """
        parts = []

        # Add only critical semantic memory
        memory = enhanced_context.get('semantic_memory', {})
        state = enhanced_context.get('conversation_state', {})

        # Only add context if user is responding to bot's question/options
        if memory.get('bot_asked_question') or memory.get('bot_presented_options'):
            parts.append("⚠️ CONTEXT:")

            if memory.get('bot_presented_options'):
                parts.append("- User selecting from options you just presented")
                # Add last bot message (truncated to 100 chars)
                if 'last_bot_message' in memory:
                    msg = memory['last_bot_message'][:100]
                    parts.append(f"- Your last message: {msg}...")
            elif memory.get('bot_asked_question'):
                parts.append("- User responding to your question")

            if state.get('user_intent') == 'selecting_option':
                parts.append("- Intent: Selecting option by number/text")

        return "\n".join(parts) if parts else ""

    def set_last_item(self, chat_id: str, item_name: str):
        """
        Store the last menu item mentioned in the conversation

        Args:
            chat_id: Chat identifier
            item_name: Name of the menu item
        """
        self._last_items_cache[chat_id] = {
            'item_name': item_name,
            'timestamp': datetime.now()
        }
        logger.info(f"💾 Stored last item for {chat_id}: '{item_name}'")

    def get_last_item(self, chat_id: str, max_age_minutes: int = 10) -> Optional[str]:
        """
        Retrieve the last menu item mentioned in the conversation

        Args:
            chat_id: Chat identifier
            max_age_minutes: Maximum age of the cached item in minutes (default: 10)

        Returns:
            Item name if found and not expired, None otherwise
        """
        if chat_id not in self._last_items_cache:
            return None

        cached = self._last_items_cache[chat_id]
        age = datetime.now() - cached['timestamp']

        if age.total_seconds() / 60 > max_age_minutes:
            # Cache expired
            del self._last_items_cache[chat_id]
            logger.info(f"⏰ Last item cache expired for {chat_id}")
            return None

        logger.info(f"📖 Retrieved last item for {chat_id}: '{cached['item_name']}'")
        return cached['item_name']

    def clear_last_item(self, chat_id: str):
        """
        Clear the last item cache for a chat

        Args:
            chat_id: Chat identifier
        """
        if chat_id in self._last_items_cache:
            del self._last_items_cache[chat_id]
            logger.info(f"🗑️ Cleared last item cache for {chat_id}")

    def set_order_state(
        self,
        context: Dict,
        state: str,
        pending_item: Optional[str] = None
    ) -> Dict:
        """
        تعيين حالة الطلب في السياق

        الحالات المدعومة:
        - awaiting_quantity: في انتظار إدخال الكمية
        - awaiting_confirmation: في انتظار تأكيد الطلب
        - awaiting_payment: في انتظار اختيار طريقة الدفع
        - awaiting_address: في انتظار إدخال العنوان

        Args:
            context: السياق الحالي
            state: الحالة الجديدة
            pending_item: اسم الصنف المعلق (اختياري)

        Returns:
            السياق المحدث
        """
        context['state'] = state

        if pending_item:
            context['pending_item'] = pending_item

        logger.info(f"📊 Order state set: {state}" + (f" (item: {pending_item})" if pending_item else ""))

        return context

    def get_order_state(self, context: Dict) -> Optional[str]:
        """
        الحصول على حالة الطلب الحالية

        Args:
            context: السياق الحالي

        Returns:
            الحالة الحالية أو None
        """
        return context.get('state')

    def clear_order_state(self, context: Dict) -> Dict:
        """
        مسح حالة الطلب من السياق

        Args:
            context: السياق الحالي

        Returns:
            السياق المحدث
        """
        if 'state' in context:
            del context['state']

        if 'pending_item' in context:
            del context['pending_item']

        logger.info(f"🗑️ Order state cleared")

        return context

    def is_in_order_flow(self, context: Dict) -> bool:
        """
        التحقق من أن المستخدم في فلو الطلب

        Args:
            context: السياق الحالي

        Returns:
            True إذا كان في فلو الطلب
        """
        state = self.get_order_state(context)

        order_states = [
            'awaiting_quantity',
            'awaiting_confirmation',
            'awaiting_payment',
            'awaiting_address'
        ]

        return state in order_states

