"""
Cart Service
Manages draft cart items stored in draft_cart_items table
"""
from typing import List, Dict, Optional
from uuid import UUID
from supabase import create_client
from app.config import settings
from app.utils.logger import app_logger as logger

# Initialize Supabase client
supabase = create_client(settings.SUPABASE_URL, settings.SUPABASE_KEY)


class CartService:
    """Service for managing cart items in draft_cart_items table"""
    
    @staticmethod
    def add_item(
        session_id: str,
        item_code: str,
        item_name: str,
        item_name_ar: Optional[str] = None,
        item_name_en: Optional[str] = None,
        quantity: int = 1,
        unit_price: float = 0.0
    ) -> Dict:
        """
        Add item to cart or update quantity if already exists
        
        Args:
            session_id: Draft session UUID
            item_code: Menu item code (e.g., 'HAREES_CHICKEN')
            item_name: Item name (Arabic or English)
            item_name_ar: Arabic name
            item_name_en: English name
            quantity: Quantity to add
            unit_price: Price per unit in OMR
            
        Returns:
            Dict with cart item data
        """
        try:
            # Check if item already exists in cart
            existing = supabase.table('draft_cart_items').select('*').eq(
                'session_id', session_id
            ).eq('item_code', item_code).execute()
            
            subtotal = quantity * unit_price
            
            if existing.data and len(existing.data) > 0:
                # Update existing item (add to quantity)
                existing_item = existing.data[0]
                new_qty = existing_item['qty'] + quantity
                new_subtotal = new_qty * unit_price
                
                result = supabase.table('draft_cart_items').update({
                    'qty': new_qty,
                    'subtotal': new_subtotal,
                    'updated_at': 'NOW()'
                }).eq('id', existing_item['id']).execute()
                
                logger.info(f"✅ Updated cart item: {item_code} (qty: {existing_item['qty']} → {new_qty})")
                return result.data[0] if result.data else existing_item
            else:
                # Insert new item
                result = supabase.table('draft_cart_items').insert({
                    'session_id': session_id,
                    'item_code': item_code,
                    'item_name': item_name,
                    'item_name_ar': item_name_ar or item_name,
                    'item_name_en': item_name_en or item_name,
                    'qty': quantity,
                    'unit_price': unit_price,
                    'subtotal': subtotal
                }).execute()
                
                logger.info(f"✅ Added new cart item: {item_code} (qty: {quantity})")
                return result.data[0] if result.data else {}
                
        except Exception as e:
            logger.error(f"❌ Error adding item to cart: {e}")
            raise
    
    @staticmethod
    def get_cart_items(session_id: str) -> List[Dict]:
        """
        Get all cart items for a session
        
        Args:
            session_id: Draft session UUID
            
        Returns:
            List of cart items
        """
        try:
            result = supabase.table('draft_cart_items').select('*').eq(
                'session_id', session_id
            ).order('created_at').execute()
            
            items = result.data if result.data else []
            logger.info(f"📦 Retrieved {len(items)} cart items for session {session_id}")
            return items
            
        except Exception as e:
            logger.error(f"❌ Error getting cart items: {e}")
            return []
    
    @staticmethod
    def update_item_quantity(
        session_id: str,
        item_code: str,
        new_quantity: int
    ) -> Optional[Dict]:
        """
        Update quantity of an existing cart item
        
        Args:
            session_id: Draft session UUID
            item_code: Menu item code
            new_quantity: New quantity (if 0, item will be removed)
            
        Returns:
            Updated cart item or None
        """
        try:
            if new_quantity <= 0:
                return CartService.remove_item(session_id, item_code)
            
            # Get existing item
            existing = supabase.table('draft_cart_items').select('*').eq(
                'session_id', session_id
            ).eq('item_code', item_code).execute()
            
            if not existing.data or len(existing.data) == 0:
                logger.warning(f"⚠️ Item {item_code} not found in cart")
                return None
            
            item = existing.data[0]
            new_subtotal = new_quantity * item['unit_price']
            
            result = supabase.table('draft_cart_items').update({
                'qty': new_quantity,
                'subtotal': new_subtotal,
                'updated_at': 'NOW()'
            }).eq('id', item['id']).execute()
            
            logger.info(f"✅ Updated item quantity: {item_code} (qty: {item['qty']} → {new_quantity})")
            return result.data[0] if result.data else None
            
        except Exception as e:
            logger.error(f"❌ Error updating item quantity: {e}")
            return None
    
    @staticmethod
    def remove_item(session_id: str, item_code: str) -> bool:
        """
        Remove item from cart
        
        Args:
            session_id: Draft session UUID
            item_code: Menu item code
            
        Returns:
            True if removed successfully
        """
        try:
            result = supabase.table('draft_cart_items').delete().eq(
                'session_id', session_id
            ).eq('item_code', item_code).execute()
            
            logger.info(f"🗑️ Removed item from cart: {item_code}")
            return True
            
        except Exception as e:
            logger.error(f"❌ Error removing item from cart: {e}")
            return False
    
    @staticmethod
    def clear_cart(session_id: str) -> bool:
        """
        Clear all items from cart
        
        Args:
            session_id: Draft session UUID
            
        Returns:
            True if cleared successfully
        """
        try:
            result = supabase.table('draft_cart_items').delete().eq(
                'session_id', session_id
            ).execute()
            
            logger.info(f"🗑️ Cleared cart for session {session_id}")
            return True
            
        except Exception as e:
            logger.error(f"❌ Error clearing cart: {e}")
            return False
    
    @staticmethod
    def calculate_totals(session_id: str) -> Dict:
        """
        Calculate cart totals

        Args:
            session_id: Draft session UUID

        Returns:
            Dict with subtotal, delivery_fee, discount, total
        """
        try:
            items = CartService.get_cart_items(session_id)

            subtotal = sum(float(item['subtotal']) for item in items)

            # Get delivery fee and discount from session
            session = supabase.table('draft_sessions').select(
                'service_type'
            ).eq('id', session_id).single().execute()

            # Calculate delivery fee
            # 🆕 إضافة رسوم التوصيل دائماً إذا كانت السلة غير فارغة
            delivery_fee = 0.0
            if subtotal > 0:
                # إذا كان service_type محدد وهو 'delivery'، أو إذا لم يكن محدداً (افتراضياً delivery)
                service_type = session.data.get('service_type') if session.data else None
                logger.info(f"🚚 service_type from session: {service_type}")
                if service_type == 'delivery' or service_type is None:
                    delivery_fee = 0.5  # Default delivery fee (500 بيسة)
                    logger.info(f"✅ Delivery fee set to: {delivery_fee}")
                else:
                    logger.info(f"⚠️ No delivery fee - service_type is: {service_type}")

            discount = 0.0  # Can be extended later

            total = subtotal + delivery_fee - discount

            return {
                'subtotal': round(subtotal, 3),
                'delivery_fee': round(delivery_fee, 3),
                'discount': round(discount, 3),
                'total': round(total, 3),
                'items_count': len(items)
            }

        except Exception as e:
            logger.error(f"❌ Error calculating totals: {e}")
            return {
                'subtotal': 0.0,
                'delivery_fee': 0.0,
                'discount': 0.0,
                'total': 0.0,
                'items_count': 0
            }
    
    @staticmethod
    def get_cart_summary(session_id: str, lang: str = 'ar') -> Dict:
        """
        Get cart summary with items and totals
        
        Args:
            session_id: Draft session UUID
            lang: Language ('ar' or 'en')
            
        Returns:
            Dict with items and totals
        """
        try:
            items = CartService.get_cart_items(session_id)
            totals = CartService.calculate_totals(session_id)
            
            # Format items for display
            formatted_items = []
            for item in items:
                name = item.get('item_name_ar' if lang == 'ar' else 'item_name_en', item['item_name'])
                formatted_items.append({
                    'code': item['item_code'],
                    'name': name,
                    'name_ar': item.get('item_name_ar', item['item_name']),
                    'name_en': item.get('item_name_en', item['item_name']),
                    'qty': int(item['qty']),
                    'quantity': int(item['qty']),  # For compatibility
                    'price': float(item['unit_price']),
                    'subtotal': float(item['subtotal'])
                })
            
            return {
                'items': formatted_items,
                **totals
            }
            
        except Exception as e:
            logger.error(f"❌ Error getting cart summary: {e}")
            return {
                'items': [],
                'subtotal': 0.0,
                'delivery_fee': 0.0,
                'discount': 0.0,
                'total': 0.0,
                'items_count': 0
            }


# Singleton instance
cart_service = CartService()

