"""
Analytics Service - خدمة التحليلات المتقدمة
تحليل شامل لأداء النظام وسلوك المستخدمين

🟥 LAYER 4: OUTPUT LAYER
"""

from typing import Dict, List, Optional
from datetime import datetime, timedelta
from dataclasses import dataclass, asdict
from collections import defaultdict
from app.utils.logger import app_logger as logger


@dataclass
class IntentAnalytics:
    """تحليلات النوايا"""
    intent_type: str
    total_count: int
    success_rate: float
    avg_response_time_ms: float
    classification_method_breakdown: Dict[str, int]
    peak_hours: List[int]


@dataclass
class PerformanceMetrics:
    """مقاييس الأداء"""
    total_messages: int
    avg_response_time_ms: float
    rule_based_percentage: float
    ai_fallback_percentage: float
    total_cost_usd: float
    avg_cost_per_message: float


@dataclass
class UserBehaviorMetrics:
    """مقاييس سلوك المستخدم"""
    total_users: int
    active_users_today: int
    avg_messages_per_user: float
    most_common_intents: List[Dict]
    conversion_rate: float  # نسبة إتمام الطلبات


class AnalyticsService:
    """
    خدمة التحليلات المتقدمة
    
    المسؤوليات:
    1. تحليل توزيع النوايا
    2. تحليل أداء النظام
    3. تحليل سلوك المستخدمين
    4. تحليل معدلات التحويل
    5. تحليل التكاليف
    
    الاستخدام:
        analytics = AnalyticsService()
        report = await analytics.generate_daily_report()
    """
    
    def __init__(self):
        # تخزين مؤقت للبيانات
        self.intent_counts = defaultdict(int)
        self.intent_response_times = defaultdict(list)
        self.classification_methods = defaultdict(lambda: defaultdict(int))
        self.hourly_distribution = defaultdict(lambda: defaultdict(int))
        
        # مقاييس الأداء
        self.total_messages = 0
        self.total_response_time = 0.0
        self.rule_based_count = 0
        self.ai_fallback_count = 0
        self.total_cost = 0.0
        
        # مقاييس المستخدمين
        self.user_message_counts = defaultdict(int)
        self.active_users_today = set()
        self.completed_orders = 0
        self.started_orders = 0
    
    async def track_intent(
        self,
        intent_type: str,
        response_time_ms: float,
        classification_method: str,
        success: bool = True,
        cost_usd: float = 0.0
    ):
        """
        تتبع نية
        
        Args:
            intent_type: نوع النية
            response_time_ms: زمن الاستجابة
            classification_method: طريقة التصنيف
            success: نجاح المعالجة
            cost_usd: التكلفة
        """
        try:
            # تحديث العدادات
            self.intent_counts[intent_type] += 1
            self.intent_response_times[intent_type].append(response_time_ms)
            self.classification_methods[intent_type][classification_method] += 1
            
            # تحديث التوزيع الساعي
            current_hour = datetime.now().hour
            self.hourly_distribution[intent_type][current_hour] += 1
            
            # تحديث مقاييس الأداء
            self.total_messages += 1
            self.total_response_time += response_time_ms
            
            if classification_method in ['rule_based', 'intent_guard']:
                self.rule_based_count += 1
            elif classification_method == 'ai_fallback':
                self.ai_fallback_count += 1
            
            self.total_cost += cost_usd
            
            logger.debug(f"📊 Tracked intent: {intent_type} ({classification_method}) - {response_time_ms:.1f}ms")
            
        except Exception as e:
            logger.error(f"Error tracking intent: {str(e)}")
    
    async def track_user_activity(
        self,
        user_id: str,
        intent_type: str
    ):
        """
        تتبع نشاط المستخدم
        
        Args:
            user_id: معرف المستخدم
            intent_type: نوع النية
        """
        try:
            self.user_message_counts[user_id] += 1
            self.active_users_today.add(user_id)
            
            # تتبع بداية وإتمام الطلبات
            if intent_type == 'ORDER_FOOD':
                self.started_orders += 1
            elif intent_type == 'CONFIRM_ORDER':
                self.completed_orders += 1
            
        except Exception as e:
            logger.error(f"Error tracking user activity: {str(e)}")
    
    async def get_intent_analytics(self, intent_type: str) -> Optional[IntentAnalytics]:
        """
        الحصول على تحليلات نية معينة
        
        Args:
            intent_type: نوع النية
            
        Returns:
            IntentAnalytics أو None
        """
        try:
            if intent_type not in self.intent_counts:
                return None
            
            total_count = self.intent_counts[intent_type]
            response_times = self.intent_response_times[intent_type]
            avg_response_time = sum(response_times) / len(response_times) if response_times else 0.0
            
            # حساب معدل النجاح (افتراضياً 100% - يمكن تحسينه)
            success_rate = 1.0
            
            # توزيع طرق التصنيف
            method_breakdown = dict(self.classification_methods[intent_type])
            
            # الساعات الأكثر نشاطاً
            hourly_dist = self.hourly_distribution[intent_type]
            peak_hours = sorted(hourly_dist.keys(), key=lambda h: hourly_dist[h], reverse=True)[:3]
            
            return IntentAnalytics(
                intent_type=intent_type,
                total_count=total_count,
                success_rate=success_rate,
                avg_response_time_ms=avg_response_time,
                classification_method_breakdown=method_breakdown,
                peak_hours=peak_hours
            )
            
        except Exception as e:
            logger.error(f"Error getting intent analytics: {str(e)}")
            return None
    
    async def get_performance_metrics(self) -> PerformanceMetrics:
        """
        الحصول على مقاييس الأداء
        
        Returns:
            PerformanceMetrics
        """
        try:
            avg_response_time = (
                self.total_response_time / self.total_messages
                if self.total_messages > 0 else 0.0
            )
            
            rule_based_pct = (
                (self.rule_based_count / self.total_messages) * 100
                if self.total_messages > 0 else 0.0
            )
            
            ai_fallback_pct = (
                (self.ai_fallback_count / self.total_messages) * 100
                if self.total_messages > 0 else 0.0
            )
            
            avg_cost = (
                self.total_cost / self.total_messages
                if self.total_messages > 0 else 0.0
            )
            
            return PerformanceMetrics(
                total_messages=self.total_messages,
                avg_response_time_ms=avg_response_time,
                rule_based_percentage=rule_based_pct,
                ai_fallback_percentage=ai_fallback_pct,
                total_cost_usd=self.total_cost,
                avg_cost_per_message=avg_cost
            )
            
        except Exception as e:
            logger.error(f"Error getting performance metrics: {str(e)}")
            return PerformanceMetrics(0, 0.0, 0.0, 0.0, 0.0, 0.0)
    
    async def get_user_behavior_metrics(self) -> UserBehaviorMetrics:
        """
        الحصول على مقاييس سلوك المستخدم
        
        Returns:
            UserBehaviorMetrics
        """
        try:
            total_users = len(self.user_message_counts)
            active_today = len(self.active_users_today)
            
            avg_messages = (
                sum(self.user_message_counts.values()) / total_users
                if total_users > 0 else 0.0
            )
            
            # أكثر النوايا شيوعاً
            most_common = [
                {'intent': intent, 'count': count}
                for intent, count in sorted(
                    self.intent_counts.items(),
                    key=lambda x: x[1],
                    reverse=True
                )[:5]
            ]
            
            # معدل التحويل (إتمام الطلبات)
            conversion_rate = (
                (self.completed_orders / self.started_orders) * 100
                if self.started_orders > 0 else 0.0
            )
            
            return UserBehaviorMetrics(
                total_users=total_users,
                active_users_today=active_today,
                avg_messages_per_user=avg_messages,
                most_common_intents=most_common,
                conversion_rate=conversion_rate
            )
            
        except Exception as e:
            logger.error(f"Error getting user behavior metrics: {str(e)}")
            return UserBehaviorMetrics(0, 0, 0.0, [], 0.0)
    
    async def generate_daily_report(self) -> Dict:
        """
        إنشاء تقرير يومي شامل
        
        Returns:
            Dict يحتوي على جميع التحليلات
        """
        try:
            logger.info("📊 Generating daily analytics report...")
            
            # جمع جميع التحليلات
            performance = await self.get_performance_metrics()
            user_behavior = await self.get_user_behavior_metrics()
            
            # تحليلات النوايا
            intent_analytics = {}
            for intent_type in self.intent_counts.keys():
                analytics = await self.get_intent_analytics(intent_type)
                if analytics:
                    intent_analytics[intent_type] = asdict(analytics)
            
            report = {
                'date': datetime.now().isoformat(),
                'performance': asdict(performance),
                'user_behavior': asdict(user_behavior),
                'intent_analytics': intent_analytics,
                'summary': {
                    'total_messages': self.total_messages,
                    'total_users': len(self.user_message_counts),
                    'avg_response_time_ms': performance.avg_response_time_ms,
                    'rule_based_percentage': performance.rule_based_percentage,
                    'conversion_rate': user_behavior.conversion_rate,
                    'total_cost_usd': performance.total_cost_usd
                }
            }
            
            logger.info("✅ Daily report generated successfully")
            
            return report
            
        except Exception as e:
            logger.error(f"Error generating daily report: {str(e)}")
            return {}
    
    def reset_daily_stats(self):
        """إعادة تعيين الإحصائيات اليومية"""
        self.active_users_today.clear()
        logger.info("🔄 Daily stats reset")


# Singleton instance
analytics_service = AnalyticsService()

