"""
Advanced Menu Search
بحث متقدم في المنيو مع دعم المرادفات والبحث الضبابي
"""

from typing import List, Dict, Optional, Tuple
from fuzzywuzzy import fuzz
import re
import logging

logger = logging.getLogger(__name__)


class AdvancedMenuSearch:
    """
    بحث متقدم في المنيو
    
    الميزات:
    - دعم المرادفات (حمص = hummus = همس)
    - Fuzzy matching (تصحيح الأخطاء الإملائية)
    - البحث في الأسماء والأوصاف
    - ترتيب النتائج حسب الأهمية
    """
    
    def __init__(self):
        self.menu_items: List[Dict] = []
        self.synonyms: Dict[str, List[str]] = {}
        self.search_index: Dict[str, str] = {}  # keyword -> item_code
        
        # مرادفات افتراضية للأصناف الشائعة
        self._build_default_synonyms()
    
    def _build_default_synonyms(self):
        """بناء مرادفات افتراضية"""
        self.synonyms = {
            # مقبلات
            'APP_HUMMUS': ['حمص', 'همس', 'hummus', 'homos', 'حموص'],
            'APP_MOUTABAL': ['متبل', 'بابا غنوج', 'moutabal', 'mutabal', 'baba ghanoush'],
            'APP_TABBOULEH': ['تبولة', 'تبوله', 'tabbouleh', 'tabouleh', 'tabouli'],
            'APP_FATTOUSH': ['فتوش', 'فتوشة', 'fattoush', 'fatoush'],
            'APP_VINE_LEAVES': ['ورق عنب', 'ورق العنب', 'vine leaves', 'dolma', 'دولمة'],
            'APP_KIBBEH': ['كبة', 'كبه', 'kibbeh', 'kebbeh'],
            'APP_SAMBOUSEK': ['سمبوسة', 'سمبوسك', 'sambousek', 'samosa'],
            
            # شوربات
            'SOUP_LENTIL': ['شوربة عدس', 'عدس', 'lentil soup', 'عدسية'],
            'SOUP_MUSHROOM': ['شوربة مشروم', 'مشروم', 'mushroom soup', 'فطر'],
            'SOUP_CHICKEN': ['شوربة دجاج', 'chicken soup'],
            'SOUP_SEAFOOD': ['شوربة روبيان', 'روبيان', 'seafood soup', 'shrimp soup'],
            'SOUP_CORN': ['شوربة ذرة', 'ذرة', 'corn soup'],
            
            # سلطات
            'SALAD_GREEK': ['سلطة يونانية', 'يونانية', 'greek salad'],
            'SALAD_CAESAR': ['سلطة سيزر', 'سيزر', 'caesar salad'],
            'SALAD_ROCCA': ['سلطة جرجير', 'جرجير', 'rocca salad', 'arugula'],
            'SALAD_ARABIC': ['سلطة عربية', 'عربية', 'arabic salad'],
            
            # أطباق عمانية
            'OMANI_ARSI': ['عرسي', 'عرسي جعلان', 'arsi', 'عرصي'],
            'OMANI_MASHUAI': ['مشوي', 'مشواي', 'mashuai', 'مصوي'],
            'OMANI_KABSA': ['كبسة', 'كبسه', 'kabsa', 'كبسة عوان'],
            'OMANI_MALEH': ['مالح', 'ملح', 'maleh'],
            'OMANI_SAHNA': ['صحنة', 'صحنه', 'sahna'],
            'OMANI_QASHAA': ['قاشع', 'قاشع برية', 'qashaa'],
            'OMANI_MASLOONA_TUNA': ['مصلونة تونة', 'مصلونة', 'masloona tuna'],
            'OMANI_FRIED_FISH': ['سمك مقلي', 'سمك', 'fried fish'],
            'OMANI_CAMEL_MEAT': ['لحم جمل', 'جمل', 'camel meat'],
            'OMANI_HAREES_MEAT': ['هريس لحم', 'هريس', 'harees meat'],
            'OMANI_HAREES_CHICKEN': ['هريس دجاج', 'harees chicken'],
            'OMANI_MASLOONA_CHICKEN': ['مصلونة دجاج', 'masloona chicken'],
            'OMANI_MASLOONA_MEAT': ['مصلونة لحم', 'masloona meat'],
            
            # مشروبات
            'DRINK_JUICE': ['عصير', 'juice', 'عصائر'],
            'DRINK_LABAN': ['لبن', 'laban', 'لبان'],
            'DRINK_WATER': ['ماء', 'water', 'مويه'],
            'DRINK_SODA': ['مشروب غازي', 'soda', 'كولا', 'cola', 'بيبسي', 'pepsi'],
            
            # أخرى
            'BURGER': ['برجر', 'برغر', 'burger', 'همبرغر'],
            'FRIES': ['بطاطس', 'بطاطا', 'fries', 'french fries'],
        }
    
    async def load_menu(self, menu_items: List[Dict]):
        """
        تحميل المنيو وبناء فهرس البحث
        
        Args:
            menu_items: قائمة الأصناف من قاعدة البيانات
        """
        self.menu_items = menu_items
        self.search_index.clear()
        
        # بناء فهرس البحث
        for item in menu_items:
            code = item['code']
            
            # إضافة الاسم العربي
            name_ar = item.get('name_ar', '').lower()
            self._add_to_index(name_ar, code)
            
            # إضافة كلمات الاسم العربي
            for word in name_ar.split():
                if len(word) > 2:
                    self._add_to_index(word, code)
            
            # إضافة الاسم الإنجليزي
            name_en = item.get('name_en', '').lower()
            self._add_to_index(name_en, code)
            
            # إضافة كلمات الاسم الإنجليزي
            for word in name_en.split():
                if len(word) > 2:
                    self._add_to_index(word, code)
            
            # إضافة المرادفات
            if code in self.synonyms:
                for synonym in self.synonyms[code]:
                    self._add_to_index(synonym.lower(), code)
        
        logger.info(f"✅ Menu loaded: {len(menu_items)} items, {len(self.search_index)} keywords")
    
    def _add_to_index(self, keyword: str, item_code: str):
        """إضافة كلمة مفتاحية للفهرس"""
        if keyword and len(keyword) > 1:
            self.search_index[keyword] = item_code
    
    async def search(
        self,
        query: str,
        top_k: int = 5,
        min_score: int = 60
    ) -> List[Tuple[Dict, int]]:
        """
        البحث في المنيو
        
        Args:
            query: نص البحث
            top_k: عدد النتائج
            min_score: الحد الأدنى للدرجة (0-100)
        
        Returns:
            قائمة من (item, score) مرتبة حسب الأهمية
        """
        query_lower = query.lower().strip()
        
        if not query_lower:
            return []
        
        results = []
        
        # 1. البحث المباشر (Exact match)
        exact_matches = self._exact_search(query_lower)
        results.extend([(item, 100) for item in exact_matches])
        
        # 2. البحث الضبابي (Fuzzy match)
        if len(results) < top_k:
            fuzzy_matches = self._fuzzy_search(query_lower, min_score)
            results.extend(fuzzy_matches)
        
        # 3. إزالة التكرار
        seen_codes = set()
        unique_results = []
        for item, score in results:
            if item['code'] not in seen_codes:
                seen_codes.add(item['code'])
                unique_results.append((item, score))
        
        # 4. ترتيب حسب الدرجة
        unique_results.sort(key=lambda x: x[1], reverse=True)
        
        return unique_results[:top_k]
    
    def _exact_search(self, query: str) -> List[Dict]:
        """البحث المباشر"""
        matches = []
        
        # البحث في الفهرس
        if query in self.search_index:
            item_code = self.search_index[query]
            item = self._get_item_by_code(item_code)
            if item:
                matches.append(item)
        
        # البحث في الأسماء الكاملة
        for item in self.menu_items:
            name_ar = item.get('name_ar', '').lower()
            name_en = item.get('name_en', '').lower()
            
            if query == name_ar or query == name_en:
                if item not in matches:
                    matches.append(item)
        
        return matches
    
    def _fuzzy_search(self, query: str, min_score: int) -> List[Tuple[Dict, int]]:
        """البحث الضبابي (مع تصحيح الأخطاء)"""
        matches = []
        
        for item in self.menu_items:
            # حساب التشابه مع الاسم العربي
            name_ar = item.get('name_ar', '').lower()
            score_ar = fuzz.partial_ratio(query, name_ar)
            
            # حساب التشابه مع الاسم الإنجليزي
            name_en = item.get('name_en', '').lower()
            score_en = fuzz.partial_ratio(query, name_en)
            
            # حساب التشابه مع المرادفات
            score_synonyms = 0
            if item['code'] in self.synonyms:
                for synonym in self.synonyms[item['code']]:
                    score = fuzz.ratio(query, synonym.lower())
                    score_synonyms = max(score_synonyms, score)
            
            # أخذ أعلى درجة
            max_score = max(score_ar, score_en, score_synonyms)
            
            if max_score >= min_score:
                matches.append((item, max_score))
        
        return matches
    
    def _get_item_by_code(self, code: str) -> Optional[Dict]:
        """الحصول على صنف بالكود"""
        for item in self.menu_items:
            if item['code'] == code:
                return item
        return None
    
    async def search_by_category(self, category: str) -> List[Dict]:
        """البحث حسب الفئة"""
        return [
            item for item in self.menu_items
            if item.get('category', '').lower() == category.lower()
        ]
    
    async def search_by_price_range(
        self,
        min_price: float,
        max_price: float
    ) -> List[Dict]:
        """البحث حسب نطاق السعر"""
        return [
            item for item in self.menu_items
            if min_price <= item.get('price_omr', 0) <= max_price
        ]
    
    async def get_all_categories(self) -> List[str]:
        """الحصول على جميع الفئات"""
        categories = set()
        for item in self.menu_items:
            category = item.get('category')
            if category:
                categories.add(category)
        return sorted(list(categories))
    
    async def add_synonym(self, item_code: str, synonym: str):
        """إضافة مرادف لصنف"""
        if item_code not in self.synonyms:
            self.synonyms[item_code] = []
        
        synonym_lower = synonym.lower()
        if synonym_lower not in self.synonyms[item_code]:
            self.synonyms[item_code].append(synonym_lower)
            self._add_to_index(synonym_lower, item_code)
            logger.info(f"✅ Added synonym '{synonym}' for {item_code}")
    
    def get_stats(self) -> Dict:
        """إحصائيات البحث"""
        return {
            'total_items': len(self.menu_items),
            'total_keywords': len(self.search_index),
            'total_synonyms': sum(len(syns) for syns in self.synonyms.values()),
            'categories': len(set(item.get('category') for item in self.menu_items))
        }


# Singleton instance
advanced_menu_search = AdvancedMenuSearch()

