"""
Webhook Routes
Handles incoming WhatsApp messages from Evolution API
"""
from fastapi import APIRouter, Request, HTTPException, BackgroundTasks
from typing import Dict, Optional, Tuple, List
import asyncio
from datetime import datetime
from app.utils.logger import app_logger as logger
from app.utils.text_helpers import arabic_to_english_numbers, extract_order_number, detect_language
from app.utils.enhanced_language_detector import enhanced_language_detector
from app.services.session_manager import SessionManager
from app.services.prompt_builder_v2 import PromptBuilderV2
from app.services.openai_service_structured import OpenAIServiceStructured
from app.services.intent_service import intent_service
from app.services.evolution_api import EvolutionAPIService
from app.services.order_manager import OrderManager
from app.services.conversation_history import ConversationHistoryService
from app.services.credits_manager import CreditsManager
from app.services.context_manager import ContextManager
from app.services.advanced_menu_search import advanced_menu_search
from app.services.faiss_menu_search import faiss_menu_search
from app.services.cart_service import CartService
from app.intents.rule_based_classifier import rule_based_classifier
from app.intents.intent_types import IntentType
from app.responses.bilingual_templates import response_templates, ResponseType
from supabase import create_client
from app.config import settings
from app.utils.cart_formatter import cart_formatter

router = APIRouter()

# Valid actions that can be returned by AI
VALID_ACTIONS = ['add_item', 'update_session', 'create_order', 'track_order', 'cancel_order', 'clear_cart', 'send_location', 'send_locations', 'repeat_order', 'reorder_last_order', 'view_last_order']

# Initialize services
supabase = create_client(settings.SUPABASE_URL, settings.SUPABASE_KEY)
session_manager = SessionManager(supabase)
prompt_builder = PromptBuilderV2(supabase)  # V2: Uses server-side prompts
credits_manager = CreditsManager(supabase)
openai_service = OpenAIServiceStructured(credits_manager=credits_manager)
evolution_api = EvolutionAPIService()
order_manager = OrderManager(supabase)
conversation_history_service = ConversationHistoryService(supabase)
context_manager = ContextManager(supabase)  # Enhanced context management

# 🆕 NEW: Initialize advanced systems
_menu_loaded = False

def get_session_items(session_id: str) -> List[Dict]:
    """
    Get cart items for a session from draft_cart_items table

    Args:
        session_id: Draft session UUID

    Returns:
        List of cart items in the format expected by the rest of the code
    """
    try:
        cart_summary = CartService.get_cart_summary(session_id, 'ar')
        return cart_summary.get('items', [])
    except Exception as e:
        logger.error(f"❌ Error getting session items: {e}")
        return []

async def initialize_advanced_systems():
    """Initialize advanced systems (FAISS search, rule-based classifier)"""
    global _menu_loaded

    if _menu_loaded:
        return

    try:
        # Check if FAISS is already loaded (from main.py startup)
        if faiss_menu_search.is_loaded():
            logger.info("✅ FAISS already loaded from startup")
            _menu_loaded = True
            return

        # Load menu items from cache
        import json
        import os

        logger.info("🔄 Loading menu items from cache for advanced systems...")
        cache_path = os.path.join(
            os.path.dirname(os.path.dirname(os.path.dirname(__file__))),
            'data', 'cache', 'menu_cache.json'
        )

        with open(cache_path, 'r', encoding='utf-8') as f:
            cache_data = json.load(f)
            menu_items = cache_data.get('items', [])

        # Initialize FAISS menu search (primary)
        await faiss_menu_search.load_menu(menu_items)

        # Initialize rule-based classifier
        await rule_based_classifier.update_menu_keywords(menu_items)

        logger.info(f"✅ FAISS menu search initialized with {len(menu_items)} items from cache")
        logger.info(f"✅ Rule-based classifier initialized")

        _menu_loaded = True

    except Exception as e:
        import traceback
        logger.error(f"❌ Error initializing advanced systems: {e}")
        logger.error(f"Traceback: {traceback.format_exc()}")
        logger.warning("⚠️ Falling back to old search system")


async def handle_intent_with_templates(
    intent_type,
    message_text: str,
    context: Dict,
    session: Dict
) -> Optional[Dict]:
    """
    Handle intent using response templates (no AI needed)

    Returns:
        AI response format dict or None if can't handle
    """
    lang = session.get('lang', 'ar')

    try:
        # GREETING - 🆕 استخدام النظام الذكي الجديد
        if intent_type == IntentType.GREETING:
            from app.responses.greeting_replies import get_smart_greeting_reply, detect_greeting_type

            # استخدام النظام الذكي للردود
            greeting_type = detect_greeting_type(message_text)
            reply = get_smart_greeting_reply(message_text)
            logger.info(f"👋 Smart greeting: type={greeting_type}, message='{message_text}'")

            return {
                'intent': 'greeting',
                'reply': reply,
                'action': 'update_session',
                'data': {}
            }

        # GOODBYE
        elif intent_type == IntentType.GOODBYE:
            reply = response_templates.get(ResponseType.GOODBYE, lang=lang)
            return {
                'intent': 'goodbye',
                'reply': reply,
                'action': 'update_session',
                'data': {}
            }

        # THANKS
        elif intent_type == IntentType.THANKS:
            reply = response_templates.get(ResponseType.THANKS_REPLY, lang=lang)
            return {
                'intent': 'thanks',
                'reply': reply,
                'action': None,
                'data': {}
            }

        # SHOW_MENU
        elif intent_type == IntentType.SHOW_MENU:
            # Get menu items by category (use FAISS with fallback)
            try:
                if faiss_menu_search.is_loaded():
                    categories = await faiss_menu_search.get_all_categories()
                else:
                    categories = await advanced_menu_search.get_all_categories()
            except Exception as e:
                logger.error(f"❌ Error getting categories: {e}")
                categories = await advanced_menu_search.get_all_categories()

            # Build menu message
            if lang == 'ar':
                reply = "📋 **قائمة الطعام:**\n\n"
            else:
                reply = "📋 **Menu:**\n\n"

            for category in categories:
                try:
                    if faiss_menu_search.is_loaded():
                        items = await faiss_menu_search.search_by_category(category)
                    else:
                        items = await advanced_menu_search.search_by_category(category)
                except Exception as e:
                    logger.error(f"❌ Error searching category: {e}")
                    items = await advanced_menu_search.search_by_category(category)
                if items:
                    category_name = items[0].get('category_ar' if lang == 'ar' else 'category_en', category)
                    reply += f"\n**{category_name}:**\n"
                    for item in items[:5]:  # Show first 5 items per category
                        name = item.get('name_ar' if lang == 'ar' else 'name_en', '')
                        price = item.get('price_omr', 0)
                        reply += f"• {name} - {price:.3f} ر.ع\n"

            reply += "\n💡 " + ("اكتب اسم الصنف للطلب" if lang == 'ar' else "Type item name to order")

            return {
                'intent': 'show_menu',
                'reply': reply,
                'action': 'update_session',
                'data': {'last_bot_showed_options': True}
            }

        # SHOW_CART
        elif intent_type == IntentType.SHOW_CART:
            # 🆕 Get items from CartService (reads from draft_cart_items table)
            session_id = session.get('id')
            try:
                cart_summary = CartService.get_cart_summary(session_id, lang)
                items = cart_summary.get('items', [])

                if not items:
                    reply = response_templates.get(ResponseType.CART_EMPTY, lang=lang)
                else:
                    # Use totals from CartService
                    totals = CartService.calculate_totals(session_id)
                    total = totals['total']

                    # Format items for display
                    items_formatted = []
                    for item in items:
                        items_formatted.append({
                            'name': item.get('item_name_ar') if lang == 'ar' else item.get('item_name_en', item.get('item_name')),
                            'quantity': item.get('qty', 1),
                            'price_omr': item.get('unit_price', 0)
                        })

                    reply = response_templates.get_cart_summary(items_formatted, total, lang=lang)
            except Exception as e:
                logger.error(f"❌ Error getting cart: {e}")
                reply = response_templates.get(ResponseType.CART_EMPTY, lang=lang)

            return {
                'intent': 'show_cart',
                'reply': reply,
                'action': None,
                'data': {}
            }

        # HELP
        elif intent_type == IntentType.HELP:
            reply = response_templates.get(ResponseType.HELP, lang=lang)
            return {
                'intent': 'help',
                'reply': reply,
                'action': None,
                'data': {}
            }

        # ORDER_FOOD - Search in menu
        elif intent_type == IntentType.ORDER_FOOD:
            # Search for item in menu using FAISS (with fallback)
            try:
                if faiss_menu_search.is_loaded():
                    # Use FAISS (fast, semantic search)
                    search_results = await faiss_menu_search.search(message_text, top_k=3, min_score=0.3)
                else:
                    # Fallback to old search
                    logger.warning("⚠️ FAISS not loaded, using fallback search")
                    search_results = await advanced_menu_search.search(message_text, top_k=3, min_score=60)
            except Exception as e:
                logger.error(f"❌ Search error: {e}, using fallback")
                search_results = await advanced_menu_search.search(message_text, top_k=3, min_score=60)

            if not search_results:
                reply = response_templates.get(
                    ResponseType.ITEM_NOT_FOUND,
                    lang=lang,
                    item_name=message_text
                )
                return {
                    'intent': 'order_food',
                    'reply': reply,
                    'action': None,
                    'data': {}
                }

            # If exact match (score >= 90), add directly
            best_match, score = search_results[0]

            if score >= 90:
                # Add item to cart
                item_name = best_match.get('name_ar' if lang == 'ar' else 'name_en', '')
                price = best_match.get('price_omr', 0)

                # 🆕 This will be handled by the 'add_item' action below
                # No need to manually add to items here - cart_service will handle it

                reply = response_templates.get_item_added(
                    item_name=item_name,
                    quantity=1,
                    price=price,
                    total=0,  # Will be calculated by cart_service
                    lang=lang
                )

                return {
                    'intent': 'order_food',
                    'reply': reply,
                    'action': 'add_item',
                    'data': {
                        'code': best_match['code'],
                        'name': item_name,
                        'quantity': 1,
                        'price': float(price),
                        'items': items
                    }
                }
            else:
                # Show options (multiple matches)
                if lang == 'ar':
                    reply = f"🔍 وجدت عدة خيارات لـ '{message_text}':\n\n"
                else:
                    reply = f"🔍 Found several options for '{message_text}':\n\n"

                for i, (item, score) in enumerate(search_results, 1):
                    name = item.get('name_ar' if lang == 'ar' else 'name_en', '')
                    price = item.get('price_omr', 0)
                    reply += f"{i}. {name} - {price:.3f} ر.ع\n"

                reply += "\n" + ("اختر رقم الصنف" if lang == 'ar' else "Choose item number")

                return {
                    'intent': 'order_food',
                    'reply': reply,
                    'action': 'update_session',
                    'data': {
                        'pending_selection': [item[0]['code'] for item in search_results],
                        'last_bot_showed_options': True
                    }
                }

        # COMPLETE_ORDER
        elif intent_type == IntentType.COMPLETE_ORDER:
            # 🆕 Get items from CartService (reads from draft_cart_items table)
            session_id = session.get('id')
            try:
                cart_summary = CartService.get_cart_summary(session_id, lang)
                items = cart_summary.get('items', [])
            except Exception as e:
                logger.error(f"❌ Error getting cart: {e}")
                items = []

            if not items:
                reply = response_templates.get(ResponseType.CART_EMPTY, lang=lang)
                return {
                    'intent': 'complete_order',
                    'reply': reply,
                    'action': None,
                    'data': {}
                }

            # 🔧 FIX: Set last_question, order_phase, and stage when asking for address
            from app.utils.question_context_manager import question_context_manager, QuestionType
            from app.state_machine.order_states import OrderPhase

            # Ask for delivery address
            reply = response_templates.get(ResponseType.ASK_DELIVERY_ADDRESS, lang=lang)

            # Prepare session updates
            session_updates = {
                'conversation_state': 'awaiting_address',
                'order_phase': OrderPhase.ADDRESS_INFO.value,
                'stage': 'address'
            }

            # Add last_question to session updates
            question_data = question_context_manager.create_question_data(QuestionType.PROVIDE_ADDRESS)
            session_updates['last_question'] = question_data

            logger.info(f"✅ COMPLETE_ORDER: Setting last_question=PROVIDE_ADDRESS, order_phase=ADDRESS_INFO, stage=address")

            return {
                'intent': 'complete_order',
                'reply': reply,
                'action': 'update_session',
                'data': session_updates
            }

        # TRACK_ORDER
        elif intent_type == IntentType.TRACK_ORDER:
            # Extract order number from message if present
            import re
            from app.utils.arabic_normalizer import normalize_arabic_numbers

            # Normalize Arabic numbers to English
            normalized_message = normalize_arabic_numbers(message_text)

            # Extract order number (4 or 5 digits)
            order_number_match = re.search(r'(\d{5}|\d{4})', normalized_message)
            order_number = order_number_match.group(1) if order_number_match else None

            if order_number:
                # Specific order number provided
                reply = "جاري البحث عن الطلب... ⏳" if lang == 'ar' else "Searching for order... ⏳"
                return {
                    'intent': 'track_order',
                    'reply': reply,
                    'action': 'track_order',
                    'data': {'order_number': order_number}
                }
            else:
                # No order number - will search for last order
                reply = "جاري البحث عن آخر طلب لك... ⏳" if lang == 'ar' else "Searching for your last order... ⏳"
                return {
                    'intent': 'track_order',
                    'reply': reply,
                    'action': 'track_order',
                    'data': {}
                }

        # CONFIRM (generic yes)
        elif intent_type == IntentType.CONFIRM:
            # Context-dependent - need AI to understand what user is confirming
            # Don't handle generic "yes" without context
            return None

        # REJECT (generic no)
        elif intent_type == IntentType.REJECT:
            # Context-dependent - need AI to understand what user is rejecting
            # Don't handle generic "no" without context
            return None

        # Can't handle this intent with templates
        else:
            return None

    except Exception as e:
        logger.error(f"❌ Error in handle_intent_with_templates: {e}")
        return None


@router.post("/webhook/whatsapp")
@router.post("/webhook/whatsapp/messages-upsert")
@router.post("/webhook/whatsapp/{event_type}")
async def whatsapp_webhook(request: Request, background_tasks: BackgroundTasks, event_type: str = None):
    """
    Receive WhatsApp messages from Evolution API
    
    Evolution API sends webhooks in this format:
    {
        "event": "messages.upsert",
        "instance": "restaurant-bot",
        "data": {
            "key": {
                "remoteJid": "96812345678@s.whatsapp.net",
                "fromMe": false,
                "id": "message_id"
            },
            "message": {
                "conversation": "أريد 2 هريس لحم"
            },
            "messageType": "conversation",
            "messageTimestamp": 1234567890
        }
    }
    """
    try:
        # Parse webhook payload
        payload = await request.json()
        logger.info(f"Received webhook: {payload.get('event')}")
        
        # Extract event type
        event = payload.get('event')
        
        # Only process incoming messages
        if event != 'messages.upsert':
            return {"status": "ignored", "reason": "not a message event"}
        
        data = payload.get('data', {})
        key = data.get('key', {})

        # 🔍 Debug: Log key details to diagnose fromMe issue
        logger.info(f"🔍 Key details: fromMe={key.get('fromMe')}, remoteJid={key.get('remoteJid')}")  # 🔧 Changed to INFO

        # Ignore messages from bot itself
        if key.get('fromMe'):
            logger.info(f"⏭️ Ignoring message from bot (fromMe=True)")
            return {"status": "ignored", "reason": "message from bot"}
        
        # Extract message details
        remote_jid = key.get('remoteJid')
        message_id = key.get('id')
        message_data = data.get('message', {})
        message_type = data.get('messageType', 'conversation')

        # 🚫 Block bot's own number and blocked numbers (additional safety check)
        # These numbers should never process messages from them
        BLOCKED_NUMBERS = ["201002051593"]  # 🔧 تم إزالة 201091842968 من القائمة
        if remote_jid and any(blocked in remote_jid for blocked in BLOCKED_NUMBERS):
            logger.info(f"⏭️ Ignoring message from blocked number: {remote_jid}")
            return {"status": "ignored", "reason": "message from blocked number"}

        # Get message text
        message_text = (
            message_data.get('conversation') or
            message_data.get('extendedTextMessage', {}).get('text') or
            ""
        )

        # Check if it's an audio message
        audio_message = message_data.get('audioMessage')

        # Check if it's a location message
        location_message = message_data.get('locationMessage')

        # Debug: Log audio message structure
        if audio_message:
            logger.info(f"Audio message structure: {audio_message.keys()}")
            logger.info(f"Audio message full data: {audio_message}")

        # Debug: Log location message
        if location_message:
            logger.info(f"📍 Location message received: {location_message}")

        # Handle location message
        if location_message:
            logger.info(f"📍 Processing location message from {remote_jid}")

            # Extract location data
            location_data = evolution_api.extract_location_from_message(message_data)

            if location_data:
                # Process location in background
                background_tasks.add_task(
                    process_location_message,
                    remote_jid,
                    location_data,
                    message_id
                )
                return {"status": "processing"}
            else:
                logger.warning("Failed to extract location data from message")
                return {"status": "ignored", "reason": "invalid location data"}

        if not message_text and not audio_message:
            return {"status": "ignored", "reason": "no text, audio, or location in message"}

        # 🆕 Additional check: Ignore messages that look like bot responses
        # This is a safety check in case fromMe doesn't work properly
        bot_response_indicators = [
            "مرحباً! 👋 أهلاً بك في مطعم",
            "يا هلا! 😊",
            "عذراً، حدث خطأ في معالجة رسالتك",
            "🛒 سلتك الحالية:",
            "📋 المنيو:",
            "🤔 **هل تقصد:**",
            "مراحب 🤗",  # 🔧 إضافة
            "تمام! ✅",  # 🔧 إضافة
            "📍 أرسل موقعك الآن",  # 🔧 إضافة
            "شكراً! 📍",  # 🔧 إضافة
            "حياك الله! عندنا أطيب الأطباق",  # 🔧 إضافة
            "يا أهلا بالطيب 🌟",  # 🔧 إضافة
            "تفضل قائمة الطعام 📸",  # 🔧 إضافة
            "لكن لست في مرحلة طلب العنوان",  # 🔧 إضافة
        ]

        if message_text and any(indicator in message_text for indicator in bot_response_indicators):
            logger.info(f"⏭️ Ignoring message that looks like bot response: {message_text[:50]}...")
            return {"status": "ignored", "reason": "message looks like bot response"}

        logger.info(f"Processing {message_type} from {remote_jid}: {message_text[:50] if message_text else 'audio message'}...")

        # Process message in background
        background_tasks.add_task(
            process_message,
            remote_jid,
            message_text,
            message_id,
            audio_message
        )

        return {"status": "processing"}
        
    except Exception as e:
        logger.error(f"Error in webhook: {e}")
        raise HTTPException(status_code=500, detail=str(e))


async def process_location_message(chat_id: str, location_data: Dict, message_id: str):
    """
    Process location message from customer

    Args:
        chat_id: WhatsApp chat ID (remoteJid)
        location_data: Location data (latitude, longitude, name, address)
        message_id: Message ID
    """
    try:
        # Mark as read
        await evolution_api.mark_message_as_read(chat_id, message_id)

        # Send typing indicator (disabled for faster response)
        # await evolution_api.send_presence(chat_id, "composing")

        # Get or create session
        session = await session_manager.get_or_create_session(chat_id)

        # Check if session is in address stage
        current_stage = session.get('stage')

        # If in 'items' stage and has items, auto-transition to 'address'
        session_items = get_session_items(session.get('id'))
        if current_stage == 'items' and len(session_items) > 0:
            logger.info(f"📍 Auto-transitioning from 'items' to 'address' stage for {chat_id}")
            await session_manager.update_session(
                session.get('id'),
                {'stage': 'address'}
            )
            session['stage'] = 'address'
            current_stage = 'address'

        # 🧠 معالجة ذكية للموقع - نحفظه دائماً بغض النظر عن المرحلة
        # Save location to session first (graceful handling)
        latitude = location_data.get('latitude')
        longitude = location_data.get('longitude')
        location_name = location_data.get('name', '')
        location_address = location_data.get('address', '')

        # Create address text from location
        if location_address:
            address_text = location_address
        elif location_name:
            address_text = location_name
        else:
            address_text = "موقع التوصيل"

        logger.info(f"📍 Saving location for {chat_id}: {address_text}")

        # حفظ الموقع في السيشن دائماً
        await session_manager.update_session(
            session.get('id'),
            {
                'address': address_text,
                'latitude': latitude,
                'longitude': longitude,
                'location_name': location_name,
                'address_type': 'location'
            }
        )

        # Detect language from session
        session_lang = session.get('lang', 'ar')
        is_arabic = (session_lang == 'ar')
        phone_number = evolution_api.extract_phone_number(chat_id)

        # التحقق من المرحلة وإرسال الرد المناسب
        if current_stage not in ['address', 'payment']:
            # ليس في مرحلة الموقع، لكن نحفظه ونسأل المستخدم
            if is_arabic:
                reply = f"تم حفظ موقعك في {address_text} 📍\n\n"

                # التحقق من وجود أصناف في السلة
                if len(session_items) > 0:
                    reply += "هل ترغب في استخدام هذا الموقع لتوصيل طلبك الحالي؟ 🚗\n\n"
                    reply += "اكتب 'نعم' للمتابعة أو 'لا' لتغيير الموقع."
                    # تحديث المرحلة إلى address لأن لديه أصناف
                    await session_manager.update_session(
                        session.get('id'),
                        {'stage': 'address', 'awaiting_location_confirmation': True}
                    )
                else:
                    reply += "سأستخدم هذا الموقع عند تأكيد طلبك 😊\n\n"
                    reply += "ابدأ الآن باختيار الأصناف من المنيو!"
            else:
                reply = f"Your location in {address_text} has been saved 📍\n\n"

                if len(session_items) > 0:
                    reply += "Would you like to use this location for your current order? 🚗\n\n"
                    reply += "Type 'yes' to continue or 'no' to change location."
                    await session_manager.update_session(
                        session.get('id'),
                        {'stage': 'address', 'awaiting_location_confirmation': True}
                    )
                else:
                    reply += "I'll use this location when you confirm your order 😊\n\n"
                    reply += "Start now by choosing items from the menu!"

            await evolution_api.send_text_message(phone_number, reply)
            await evolution_api.send_presence(chat_id, "available")
            return

        # المستخدم في مرحلة address أو payment - نكمل العملية بشكل طبيعي
        # Check if location already exists (updating)
        is_updating = bool(session.get('address'))

        # Update session with location data and move to payment
        await session_manager.update_session(
            session.get('id'),
            {
                'address': address_text,
                'latitude': latitude,
                'longitude': longitude,
                'location_name': location_name,
                'address_type': 'location',
                'stage': 'payment',
                'last_question': 'choose_payment'
            }
        )

        # تحديث الحالة في الذاكرة أيضاً
        session['stage'] = 'payment'
        session['last_question'] = 'choose_payment'

        logger.info(f"📍 Location received from {chat_id}, state moved to payment stage")
        logger.info(f"✅ Session updated: stage=payment, last_question=choose_payment")

        # 1. Re-send the location back to user
        # Evolution API requires name and address fields
        if is_arabic:
            location_display_name = "موقع التوصيل"
            location_display_address = "Delivery location"
        else:
            location_display_name = "Delivery location"
            location_display_address = "موقع التوصيل"

        await evolution_api.send_location(
            phone_number=phone_number,
            latitude=latitude,
            longitude=longitude,
            name=location_display_name,
            address=location_display_address
        )

        # 2. Send order summary after location
        items = get_session_items(session.get('id'))

        # ⚠️ Fix old sessions: Add name_en if missing
        for item in items:
            if 'name_en' not in item:
                item_code = item.get('item_code') or item.get('code')
                if item_code:
                    result = supabase.table('menu_items').select('name_ar, name_en').eq('code', item_code).execute()
                    if result.data:
                        menu_item = result.data[0]
                        item['name'] = menu_item.get('name_ar', item.get('name', ''))
                        item['name_en'] = menu_item.get('name_en', item.get('name', ''))

        # 🆕 Calculate totals from cart (don't rely on session.subtotal)
        cart_totals = CartService.calculate_totals(session.get('id'))
        subtotal = cart_totals.get('subtotal', 0)
        delivery_fee = 0.500
        total = subtotal + delivery_fee

        # Build items list
        if is_arabic:
            items_text = "\n".join([
                f"• {int(item.get('qty') or item.get('quantity', 1))} × {item.get('name', '')} ({item.get('price', 0):.3f} ر.ع) = {int(item.get('qty') or item.get('quantity', 1)) * item.get('price', 0):.3f} ر.ع"
                for item in items
            ])

            reply = f"""📋 ملخص طلبك:
{items_text}

💰 المجموع الفرعي: {subtotal:.3f} ر.ع
🚚 رسوم التوصيل: {delivery_fee:.3f} ر.ع
💵 الإجمالي: {total:.3f} ر.ع
📍 الموقع: موقع التوصيل

ما هي طريقة الدفع التي تفضلها؟

1️⃣ نقدي 💵
2️⃣ بطاقة 💳"""
        else:
            items_text = "\n".join([
                f"• {int(item.get('qty') or item.get('quantity', 1))} × {item.get('name_en', item.get('name', ''))} ({item.get('price', 0):.3f} OMR) = {int(item.get('qty') or item.get('quantity', 1)) * item.get('price', 0):.3f} OMR"
                for item in items
            ])

            reply = f"""📋 Order summary:
{items_text}

💰 Subtotal: {subtotal:.3f} OMR
🚚 Delivery fee: {delivery_fee:.3f} OMR
💵 Total: {total:.3f} OMR
📍 Location: Delivery location

What payment method do you prefer?

1️⃣ Cash 💵
2️⃣ Card 💳"""

        await evolution_api.send_text_message(phone_number, reply)

        # Stop typing
        await evolution_api.send_presence(chat_id, "available")

        logger.info(f"✅ Location saved for {chat_id}: lat={latitude}, lon={longitude}")

    except Exception as e:
        logger.error(f"❌ Error processing location: {e}")
        import traceback
        logger.error(f"Traceback: {traceback.format_exc()}")

        try:
            phone_number = evolution_api.extract_phone_number(chat_id)

            # Detect language from session
            is_arabic = True
            if session:
                session_lang = session.get('lang', 'ar')
                is_arabic = (session_lang == 'ar')

            if is_arabic:
                error_msg = "عذراً، حدث خطأ في حفظ الموقع. يرجى المحاولة مرة أخرى أو إرسال العنوان نصاً."
            else:
                error_msg = "Sorry, there was an error saving the location. Please try again or send your address as text."

            await evolution_api.send_text_message(phone_number, error_msg)
            await evolution_api.send_presence(chat_id, "available")
        except:
            pass


async def process_message(chat_id: str, message_text: str, message_id: str, audio_message: dict = None):
    """
    Process incoming message

    Args:
        chat_id: WhatsApp chat ID (remoteJid)
        message_text: Message text
        message_id: Message ID
        audio_message: Audio message data (if audio)
    """
    try:
        # Mark message as read
        await evolution_api.mark_message_as_read(chat_id, message_id)

        # Send typing indicator (disabled for faster response)
        # await evolution_api.send_presence(chat_id, "composing")

        # If audio message, transcribe it first
        if audio_message and not message_text:
            logger.info(f"Transcribing audio message from {chat_id}...")
            phone_number = evolution_api.extract_phone_number(chat_id)

            # Transcribe audio (without sending notification message)
            message_text = await evolution_api.transcribe_audio(audio_message, message_id)

            if not message_text:
                await evolution_api.send_presence(chat_id, "available")
                await evolution_api.send_text_message(
                    phone_number,
                    "عذراً، لم أتمكن من تحويل الرسالة الصوتية. يرجى إرسال رسالة نصية."
                )
                return

            logger.info(f"Transcribed audio: {message_text[:100]}...")

        # ✅ PREPROCESSING: Convert Arabic numerals to English
        original_message = message_text
        message_text = arabic_to_english_numbers(message_text)

        # Try to extract order number from message
        extracted_order_number = extract_order_number(message_text)
        if extracted_order_number:
            logger.info(f"📦 Extracted order number: {extracted_order_number} from: {original_message}")

        # 🆕 Initialize advanced systems if needed
        await initialize_advanced_systems()

        # Get or create session
        session = await session_manager.get_or_create_session(chat_id)
        session_id = session['id']

        # Initialize variables
        intent = None
        action = None
        reply = None
        data = {}

        # 🆕 Check if user is confirming reorder
        skip_normal_processing = False
        waiting_for_reorder = session.get('waiting_for_reorder_confirmation', False)
        if waiting_for_reorder:
            # Check if message is confirmation (yes/no)
            message_lower = message_text.lower().strip()
            is_yes = message_lower in ['نعم', 'yes', 'اي', 'ايوه', 'اه', 'اكيد', 'تمام', 'موافق', 'ok']
            is_no = message_lower in ['لا', 'no', 'لأ', 'مش عايز', 'لاء']

            if is_yes:
                logger.info("✅ User confirmed reorder - triggering reorder_last_order action")
                # Clear the flag
                await session_manager.update_session(session_id, {
                    'waiting_for_reorder_confirmation': False
                })
                # Force action to reorder
                action = 'reorder_last_order'
                intent = 'Reorder Last Order'
                reply = None  # Will be set by reorder handler
                # Skip normal AI processing
                skip_normal_processing = True
            elif is_no:
                logger.info("❌ User declined reorder")
                # Clear the flag
                await session_manager.update_session(session_id, {
                    'waiting_for_reorder_confirmation': False
                })
                # Send appropriate response
                session_lang = session.get('lang', 'ar')
                if session_lang == 'ar':
                    reply = "حسناً! 👍 اختر من المنيو لإضافة أصناف!"
                else:
                    reply = "Okay! 👍 Choose from the menu to add items!"
                # Skip normal AI processing
                skip_normal_processing = True
                action = None
                intent = 'Decline Reorder'

        # 🆕 Enhanced language detection
        detected_language, confidence = enhanced_language_detector.detect(
            message_text,
            user_id=chat_id,
            session_lang=session.get('lang')
        )
        logger.info(f"🌐 Detected language: {detected_language} (confidence: {confidence:.2f})")

        # Update session language if detected with high confidence
        if detected_language and confidence >= 0.7 and session.get('lang') != detected_language:
            await session_manager.update_session(
                session_id,
                {'lang': detected_language}
            )
            session['lang'] = detected_language
            logger.info(f"✅ Updated session language to: {detected_language}")

        # Get session context
        context = await session_manager.get_session_context(session)

        # Get conversation history (optimized to 8 for balance between context and speed)
        conversation_history = await conversation_history_service.get_recent_messages(
            chat_id,
            limit=8  # Balanced: enough context, not too slow
        )

        # Build enhanced context with smart filtering and semantic memory
        enhanced_context = await context_manager.build_enhanced_context(
            chat_id=chat_id,
            current_message=message_text,
            session_context=context,
            conversation_history=conversation_history
        )

        # Use filtered conversation history (more relevant, fewer tokens)
        filtered_history = enhanced_context['relevant_messages']

        # Add conversation history to context for prompt building
        context['recent_messages'] = filtered_history
        logger.info(
            f"📚 Context: {len(filtered_history)}/{len(conversation_history)} messages "
            f"({len(enhanced_context['semantic_memory'])} facts)"
        )

        # Build dynamic prompt (pass extracted order number)
        system_prompt, user_message = await prompt_builder.build_dynamic_prompt(
            message_text,
            context,
            extracted_order_number
        )

        # 🆕 Skip normal processing if already handled (e.g., reorder confirmation)
        ai_response = None  # Initialize

        if not skip_normal_processing:
            # 🆕 PRIORITY 0: Check for greeting FIRST (before checking last_question)
            # Greetings should reset context and start fresh conversation
            # BUT: If greeting + food order in same message, treat as order!

            # 🤖 Use AI Intent Analyzer ONLY for complex cases
            from app.utils.ai_intent_analyzer import AIIntentAnalyzer

            intent = "other"
            confidence = 0.0
            is_compound = False
            is_greeting = False
            is_greeting_with_order = False

            # تحديد ذكي: هل نحتاج AI؟
            if AIIntentAnalyzer.should_use_ai_analyzer(message_text):
                logger.info(f"🤖 Complex message detected - using AI Intent Analyzer")

                # Build conversation history for context
                conversation_history = []
                if filtered_history:
                    for msg in filtered_history[-3:]:  # Last 3 messages
                        conversation_history.append({
                            "role": msg.get("sender", "user"),
                            "content": msg.get("message", "")
                        })

                # Analyze intent with AI
                ai_result = await AIIntentAnalyzer.analyze_intent(
                    current_message=message_text,
                    conversation_history=conversation_history,
                    use_menu=True  # تمرير المنيو للحالات المعقدة
                )

                intent = ai_result.get("intent", "other")
                confidence = ai_result.get("confidence", 0.0)
                is_compound = ai_result.get("is_compound", False)
                item_not_in_menu = ai_result.get("item_not_in_menu", False)

                logger.info(f"🎯 AI Intent: {intent} (confidence: {confidence:.2f}, compound: {is_compound})")
                logger.info(f"📝 Reasoning: {ai_result.get('reasoning', 'N/A')}")

                # 🚨 معالجة خاصة: صنف غير موجود في المنيو
                if item_not_in_menu and intent in ["order_food", "greeting_with_order"]:
                    logger.warning(f"⚠️ Item not in menu detected!")

                    extracted_items = ai_result.get("extracted_items", [])
                    suggested_alternatives = ai_result.get("suggested_alternatives", [])

                    # بناء رد ذكي وودي
                    lang = context.get('lang', 'ar')
                    if lang == 'ar':
                        item_name = extracted_items[0] if extracted_items else "هذا الصنف"
                        reply = f"😔 عذراً، \"{item_name}\" غير متوفر في منيونا حالياً.\n\n"

                        if suggested_alternatives:
                            reply += "💡 لكن لدينا بدائل رائعة قد تعجبك:\n\n"
                            for i, alt in enumerate(suggested_alternatives[:3], 1):
                                reply += f"{i}. {alt}\n"
                            reply += "\n🍽️ هل تريد تجربة أحد هذه الأصناف؟\n"
                            reply += "أو أرسل \"المنيو\" لرؤية القائمة الكاملة. 📋"
                        else:
                            reply += "🍽️ أرسل \"المنيو\" لرؤية جميع الأصناف المتوفرة لدينا. 📋"
                    else:
                        item_name = extracted_items[0] if extracted_items else "this item"
                        reply = f"😔 Sorry, \"{item_name}\" is not available in our menu.\n\n"

                        if suggested_alternatives:
                            reply += "💡 But we have great alternatives you might like:\n\n"
                            for i, alt in enumerate(suggested_alternatives[:3], 1):
                                reply += f"{i}. {alt}\n"
                            reply += "\n🍽️ Would you like to try one of these?\n"
                            reply += "Or send \"menu\" to see the full menu. 📋"
                        else:
                            reply += "🍽️ Send \"menu\" to see all available items. 📋"

                    # إرسال الرد مباشرة
                    ai_response = {
                        'intent': 'Ask Information',
                        'reply': reply,
                        'action': None,
                        'data': None
                    }
                    skip_normal_processing = True
                    logger.info(f"✅ Item not in menu handled: {item_name}")

                # Determine if it's a pure greeting or greeting with order
                is_greeting = intent == "greeting_only"
                is_greeting_with_order = intent == "greeting_with_order"

                # If greeting with order, treat as normal order (skip greeting handler)
                if is_greeting_with_order:
                    logger.info("🍔 Greeting + Order detected - treating as food order")
                    # Continue to normal order processing
                    pass
            else:
                # استخدام الطريقة التقليدية (بدون AI) للحالات البسيطة
                logger.info(f"📋 Simple message - using rule-based detection")
                message_lower = message_text.lower().strip()
                greeting_keywords = ["السلام عليكم", "السلام", "مرحبا", "صباح الخير", "مساء الخير", "هلا", "hello", "hi", "hey"]
                is_greeting = any(keyword in message_lower for keyword in greeting_keywords) and len(message_text.split()) <= 3

            if is_greeting:
                logger.info("👋 Pure greeting detected - clearing old context and starting fresh")

                # Clear old questions and waiting states IMMEDIATELY
                session_updates = {
                    'last_question': {
                        'type': 'none',
                        'asked_at': None
                    },
                    'waiting_for_more_items': False,
                    'waiting_for_item_choice': False,
                    'waiting_for_quantity': False,
                    'pending_item_choices': [],
                    'selected_item_code': None
                }

                # Update session in database IMMEDIATELY
                await handle_update_session(session_id, session_updates, session)

                # Update local session object
                session.update(session_updates)
                context['session'] = session

                logger.info("✅ Context cleared - proceeding with greeting handler")

                # 🆕 استدعاء معالج التحية مباشرة
                from app.handlers.small_talk_handler import small_talk_handler
                from app.responses.greeting_replies import get_smart_greeting_reply, detect_greeting_type

                try:
                    # استخدام النظام الذكي للردود
                    greeting_type = detect_greeting_type(message_text)
                    reply = get_smart_greeting_reply(message_text)
                    logger.info(f"👋 Smart greeting reply: type={greeting_type}")

                    # إنشاء الرد
                    ai_response = {
                        'intent': 'greeting',
                        'reply': reply,
                        'action': None,
                        'data': None
                    }

                    skip_normal_processing = True
                    logger.info("✅ Greeting handled with smart reply system")

                except Exception as e:
                    logger.error(f"❌ Error in smart greeting: {e}")
                    # Fallback to normal processing

            # 🗑️ Handle remove from cart intent
            elif intent == "remove_from_cart":
                logger.info("🗑️ Remove from cart intent detected")
                # Let the normal AI system handle it
                pass

            # 🆕 PRIORITY 1: Check for active question (yes/no responses)
            last_question = session.get('last_question')

            # 🆕 Handle both old format (dict) and new format (string)
            if isinstance(last_question, dict):
                question_type = last_question.get('type')
            elif isinstance(last_question, str):
                question_type = last_question
            else:
                question_type = None

            if question_type and question_type != 'none':
                logger.info(f"❓ Active question detected: {question_type}")

                # Try yes_no_handler first
                from app.handlers.yes_no_handler import yes_no_handler

                try:
                    yes_no_response = await yes_no_handler.handle(
                        user_message=message_text,
                        context=context,
                        conversation_history=filtered_history
                    )

                    if yes_no_response:
                        logger.info(f"✅ Handled by yes_no_handler for question: {question_type}")
                        ai_response = yes_no_response
                        # Skip to response handling
                        skip_normal_processing = True
                except Exception as e:
                    logger.warning(f"⚠️ yes_no_handler failed: {e}")

            # 🆕 PRIORITY 2: Try Rule-Based Classification (90% of cases)
            rule_result = None  # Initialize
            if not skip_normal_processing:
                logger.info("🎯 Trying rule-based classification...")

                # Prepare context for rule-based classifier
                rule_context = {
                    'conversation_state': context.get('conversation_state', ''),
                    'last_bot_message': context.get('last_bot_message', ''),
                    'last_bot_question': context.get('last_bot_question', ''),
                    'last_bot_showed_options': context.get('last_bot_showed_options', False),
                    'items': context.get('items', []),
                    'lang': session.get('lang', 'ar'),
                    'waiting_for_more_items': session.get('waiting_for_more_items', False)  # 🆕 Add flag
                }

                rule_result = rule_based_classifier.classify(message_text, rule_context)

                if rule_result:
                    intent_type, rule_confidence = rule_result
                    logger.info(f"✅ Rule-based match: {intent_type.value} (confidence: {rule_confidence:.2f})")

                    # 🎯 Handle intent with templates (no AI needed!)
                    ai_response = await handle_intent_with_templates(
                        intent_type=intent_type,
                        message_text=message_text,
                        context=context,
                        session=session
                    )

                    # If handled successfully, skip AI
                    if ai_response:
                        logger.info(f"✅ Handled by rule-based system (no AI used!)")
                    else:
                        # Fallback to AI
                        logger.info(f"⚠️ Rule-based handler returned None, falling back to AI")
                        rule_result = None

            # If rule-based didn't work, use AI
            if not skip_normal_processing and (not rule_result or not ai_response):
                logger.info("🤖 Using AI-based system (rule-based didn't match)")

                try:
                    # Use the new intent service
                    ai_response = await intent_service.process_message(
                        user_message=message_text,
                        context=context,
                        conversation_history=filtered_history
                    )
                except TimeoutError:
                    logger.error("❌ Intent service timeout! Sending fallback response...")
                    # Send fallback response to user
                    fallback_reply = "عذراً، حدث خطأ مؤقت. من فضلك أعد المحاولة." if session.get('lang') == 'ar' else "Sorry, a temporary error occurred. Please try again."
                    await evolution_api.send_message(
                        instance_name=settings.EVOLUTION_INSTANCE_NAME,
                        phone_number=phone_number,
                        message=fallback_reply
                    )
                    return {"status": "timeout", "message": "Intent service timeout"}
                except Exception as e:
                    logger.error(f"❌ Error in intent service: {str(e)}")
                    # Send error response to user
                    error_reply = "عذراً، حدث خطأ. من فضلك حاول مرة أخرى." if session.get('lang') == 'ar' else "Sorry, an error occurred. Please try again."
                    await evolution_api.send_message(
                        instance_name=settings.EVOLUTION_INSTANCE_NAME,
                        phone_number=phone_number,
                        message=error_reply
                    )
                    return {"status": "error", "message": str(e)}

        # Extract response fields (handle both AI response and reorder confirmation)
        if 'ai_response' in locals() and ai_response:
            if intent is None:
                intent = ai_response.get('intent')
            if reply is None:
                reply = ai_response.get('reply')
            if action is None:
                action = ai_response.get('action')
            if not data:
                data = ai_response.get('data', {})

        logger.info(f"AI Response - Intent: {intent or 'None'}, Action: {action or 'None'}")

        # ⚠️ LANGUAGE ENFORCEMENT: Force reply to match session language
        session_lang = session.get('lang', 'ar')
        if reply and session_lang:
            # Check if reply language matches session language
            reply_has_arabic = any(ord(c) > 127 for c in reply[:100])

            if session_lang == 'en' and reply_has_arabic:
                logger.warning(f"⚠️ Language mismatch! Session lang={session_lang} but reply has Arabic. Translating...")
                # Simple translation for common responses
                if 'هريس لحم' in reply:
                    reply = reply.replace('هريس لحم', 'Harees (meat)')
                    reply = reply.replace('بسعر', 'costs')
                    reply = reply.replace('ر.ع', 'OMR')
                    reply = reply.replace('هل تريد طلبه؟', 'Would you like to order it?')
                    logger.info(f"✅ Translated reply to English")
            elif session_lang == 'ar' and not reply_has_arabic:
                logger.warning(f"⚠️ Language mismatch! Session lang={session_lang} but reply is in English.")
                # Keep English for now, but log the issue

        # Validate action
        if action and action not in VALID_ACTIONS:
            logger.warning(f"⚠️ Unknown action received: {action}. Setting to None.")
            action = None

        # Handle action
        if action == 'add_item':
            # 🆕 NEW: Add item to cart using cart_service (stores in draft_cart_items table)
            if data:
                item_code = data.get('code')
                item_name = data.get('name')
                item_name_ar = data.get('name_ar', item_name)
                item_name_en = data.get('name_en', item_name)
                item_quantity = data.get('quantity', 1)
                item_price = data.get('price')

                # ✅ إذا لم يتم تمرير السعر أو الأسماء، جلبها من قاعدة البيانات
                if item_price is None or item_price == 0 or not item_name_ar or not item_name_en:
                    logger.warning(f"⚠️ Missing data for {item_code}, fetching from database...")
                    try:
                        item_result = supabase.table('menu_items').select('price_omr, name_ar, name_en').eq('code', item_code).single().execute()
                        if item_result.data:
                            if item_price is None or item_price == 0:
                                item_price = float(item_result.data['price_omr'])
                                logger.info(f"✅ Fetched price from database: {item_price}")
                            if not item_name_ar:
                                item_name_ar = item_result.data.get('name_ar', item_name)
                                logger.info(f"✅ Fetched name_ar from database: {item_name_ar}")
                            if not item_name_en:
                                item_name_en = item_result.data.get('name_en', item_name)
                                logger.info(f"✅ Fetched name_en from database: {item_name_en}")
                        else:
                            logger.error(f"❌ Item {item_code} not found in database")
                            item_price = 0.0
                    except Exception as e:
                        logger.error(f"❌ Error fetching data from database: {e}")
                        item_price = 0.0
                else:
                    item_price = float(item_price)

                logger.info(f"🛒 Adding item to cart: {item_name} (code={item_code}, qty={item_quantity}, price={item_price})")

                try:
                    # Add item to cart using CartService
                    CartService.add_item(
                        session_id=session_id,
                        item_code=item_code,
                        item_name=item_name,
                        item_name_ar=item_name_ar,
                        item_name_en=item_name_en,
                        quantity=item_quantity,
                        unit_price=item_price
                    )

                    # Get cart summary with totals
                    lang = session.get('lang', 'ar')
                    cart_summary_data = CartService.get_cart_summary(session_id, lang)
                    current_items = cart_summary_data['items']
                    totals = {
                        'subtotal': cart_summary_data['subtotal'],
                        'delivery_fee': cart_summary_data['delivery_fee'],
                        'discount': cart_summary_data['discount'],
                        'total': cart_summary_data['total']
                    }

                    logger.info(f"✅ Cart updated: {len(current_items)} items, total={totals['total']:.2f} OMR")

                    # Set question context: ADD_MORE_ITEMS
                    from app.utils.question_context_manager import question_context_manager, QuestionType
                    question_context_manager.set_question(session, QuestionType.ADD_MORE_ITEMS)
                    logger.info(f"❓ Set question context: ADD_MORE_ITEMS")

                    # Update local session object
                    session['waiting_for_more_items'] = True
                    # 🆕 حفظ آخر صنف تم إضافته (للتعديل لاحقاً)
                    session['last_ordered_item'] = item_name_ar or item_name
                    logger.info(f"💾 Saved last_ordered_item: {session['last_ordered_item']}")

                    # 🆕 Update session with waiting flag AND last_question together
                    try:
                        supabase.table('draft_sessions').update({
                            'waiting_for_more_items': True,
                            'last_question': session.get('last_question'),
                            'updated_at': datetime.utcnow().isoformat()
                        }).eq('id', session_id).execute()
                        logger.info(f"✅ Saved waiting_for_more_items=True and last_question to database")
                    except Exception as e:
                        logger.error(f"❌ Error saving session: {e}")

                    # 🆕 تحديد نوع الرسالة بناءً على عدد العناصر في السلة
                    RTL = '\u200F'
                    if len(current_items) == 1:
                        # أول صنف: رسالة بسيطة فقط (بدون عرض السلة)
                        if lang == 'ar':
                            reply = f"{RTL}✅ تمت إضافة {item_name_ar} × {item_quantity} إلى سلتك!\n\n"
                            reply += f"{RTL}🤔 هل تريد إضافة المزيد؟"
                        else:
                            reply = f"✅ Added {item_name_en} × {item_quantity} to your cart!\n\n"
                            reply += "🤔 Would you like to add more items?"

                        logger.info(f"📝 First item: showing simple message (no cart details)")
                    else:
                        # عنصر ثاني أو أكثر: عرض السلة الكاملة
                        cart_summary = cart_formatter.format_cart(current_items, lang, totals)

                        # إضافة سؤال "هل تريد إضافة المزيد؟"
                        if lang == 'ar':
                            reply = f"{cart_summary}\n\n{RTL}🤔 هل تريد إضافة المزيد؟"
                        else:
                            reply = f"{cart_summary}\n\n🤔 Would you like to add more items?"

                        logger.info(f"📝 Multiple items ({len(current_items)}): showing full cart")

                except Exception as e:
                    logger.error(f"❌ Error adding item to cart: {e}")
                    reply = "عذراً، حدث خطأ أثناء إضافة الصنف إلى السلة. يرجى المحاولة مرة أخرى." if session.get('lang') == 'ar' else "Sorry, an error occurred while adding the item to cart. Please try again."
        elif action == 'send_location':
            # Update session with data (e.g., payment_method) before sending location
            if data:
                await handle_update_session(session_id, data, session)
                logger.info(f"✅ Session updated before sending location: {list(data.keys())}")

            # Send location message before reply
            latitude = session.get('latitude')
            longitude = session.get('longitude')
            location_name = session.get('location_name', 'موقع العميل')
            address = session.get('address', '')

            if latitude and longitude:
                phone_number = evolution_api.extract_phone_number(chat_id)
                await evolution_api.send_location(
                    phone_number=phone_number,
                    latitude=latitude,
                    longitude=longitude,
                    name=location_name,
                    address=address
                )
                logger.info(f"📍 Location sent: {latitude}, {longitude}")
        elif action == 'send_locations':
            # Send multiple branch locations
            phone_number = evolution_api.extract_phone_number(chat_id)
            branches = data.get('branches', []) if data else []

            # Detect language from session (more reliable than reply)
            session_lang = session.get('lang', 'ar')
            is_arabic = (session_lang == 'ar')

            logger.info(f"📍 Sending {len(branches)} branch locations")
            logger.info(f"🌐 Language: session_lang={session_lang}, is_arabic={is_arabic}")

            # ✅ FIRST: Send the intro message (e.g., "سأرسل لك مواقع فروعنا:")
            if reply:
                await evolution_api.send_text_message(phone_number, reply)
                logger.info(f"✅ Sent intro message before locations")
                await asyncio.sleep(1)  # Small delay before sending locations

            # ✅ SECOND: Send the locations with caption
            for branch in branches:
                # Build caption with branch details
                if is_arabic:
                    caption = f"""📍 {branch['name']}

📫 {branch['address']}
📞 {branch['phone']}
🕐 {branch['hours']}"""
                else:
                    caption = f"""📍 {branch['name_en']}

📫 {branch['address_en']}
📞 {branch['phone']}
🕐 {branch['hours_en']}"""

                # Send location with caption
                await evolution_api.send_location(
                    phone_number=phone_number,
                    latitude=branch['latitude'],
                    longitude=branch['longitude'],
                    name=branch['name'] if is_arabic else branch['name_en'],
                    address=branch['address'] if is_arabic else branch['address_en'],
                    caption=caption
                )

                # Small delay between branches
                await asyncio.sleep(0.5)

            logger.info(f"✅ Sent {len(branches)} branch locations")

            # ⚠️ IMPORTANT: Clear reply so it's not sent again at the end
            reply = None
        elif action == 'update_session':
            await handle_update_session(session_id, data, session)
        elif action == 'confirm_order':
            # User confirmed the order
            # The yes_no_handler already set the next question to PROVIDE_ADDRESS in memory
            # Now we need to save it to the database
            logger.info("✅ User confirmed order, saving next question PROVIDE_ADDRESS to database")

            # 🔧 FIX: Save last_question to database
            if data and 'last_question' in data:
                await handle_update_session(session_id, data, session)
            else:
                logger.warning("⚠️ No last_question in data, but should have been set by yes_no_handler")
        elif action == 'create_order':
            # Update session with payment_method if provided in data
            if data and data.get('payment_method'):
                await handle_update_session(session_id, data, session)
                session['payment_method'] = data.get('payment_method')
                logger.info(f"✅ Updated payment_method: {data.get('payment_method')}")

            # Create order and get order ID
            order = await order_manager.create_order_from_session(session)
            order_id_text = order.get('order_id', '')  # The order ID (e.g., "10025")
            order_uuid = order.get('id', '')  # The UUID primary key
            logger.info(f"✅ Order created: {order_id_text} (UUID: {order_uuid})")

            # Clear reminder count for this session (order completed)
            try:
                from app.services.reminder_service import reminder_service
                chat_id = session.get('chat_id')
                if chat_id:
                    reminder_service.clear_reminder_count(chat_id)
            except Exception as e:
                logger.warning(f"⚠️ Failed to clear reminder count: {e}")

            # Build order details message
            session_lang = session.get('lang', 'ar')
            is_arabic = (session_lang == 'ar')

            # Get order items
            items = get_session_items(session.get('id'))
            items_text = '\n'.join([
                f"• {item.get('name')} x{int(item.get('qty') or item.get('quantity', 1))}"
                for item in items
            ])

            # 🆕 Calculate totals from cart (don't rely on session.subtotal)
            cart_totals = CartService.calculate_totals(session.get('id'))
            subtotal = cart_totals.get('subtotal', 0)
            delivery_fee = 0.500
            total = subtotal + delivery_fee

            logger.info(f"💰 Order confirmation - Subtotal: {subtotal}, Total: {total}")

            # Get delivery info
            delivery_address = session.get('address', '')
            payment_method = session.get('payment_method', '')

            # Format payment method
            if payment_method == 'cash':
                payment_text = 'الدفع عند الاستلام (نقداً)' if is_arabic else 'Cash on delivery'
            elif payment_method == 'credit_card':
                payment_text = 'بطاقة ائتمان' if is_arabic else 'Credit card'
            else:
                payment_text = payment_method

            # Build complete order message
            if is_arabic:
                reply = f"""✅ **تم تأكيد طلبك بنجاح!**

📋 **رقم الطلب:** {order_id_text}

🛒 **الأصناف:**
{items_text}

💰 **المجموع الفرعي:** {subtotal:.3f} ر.ع
🚚 **رسوم التوصيل:** {delivery_fee:.3f} ر.ع
💵 **الإجمالي:** {total:.3f} ر.ع

📍 **عنوان التوصيل:** {delivery_address}
💳 **طريقة الدفع:** {payment_text}

📦 **حالة الطلب:** جاري التحضير

⏱️ **وقت التوصيل المتوقع:** 30-45 دقيقة

شكراً لطلبك! 🙏"""
            else:
                reply = f"""✅ **Your order has been confirmed!**

📋 **Order Number:** {order_id_text}

🛒 **Items:**
{items_text}

💰 **Subtotal:** {subtotal:.3f} OMR
🚚 **Delivery Fee:** {delivery_fee:.3f} OMR
💵 **Total:** {total:.3f} OMR

📍 **Delivery Address:** {delivery_address}
💳 **Payment Method:** {payment_text}

📦 **Order Status:** Preparing

⏱️ **Estimated Delivery:** 30-45 minutes

Thank you for your order! 🙏"""

            logger.info(f"📦 Order details message prepared for order {order_id_text}")
        elif action == 'track_order':
            # Track completed orders - show both incomplete and completed orders
            order_number = data.get('order_number') if data else None
            order_info = None
            # Use session language instead of message text
            session_lang = session.get('lang', 'ar')
            is_arabic = (session_lang == 'ar')

            reply_parts = []

            if order_number:
                # Specific order number provided - just show that order
                order_info = await handle_track_order(order_number, chat_id)

                if order_info:
                    order_details = await build_order_details(order_info, is_arabic)
                    reply = order_details
                else:
                    if is_arabic:
                        reply = f"عذراً، لم أتمكن من العثور على الطلب رقم {order_number}."
                    else:
                        reply = f"Sorry, I couldn't find order {order_number}."
            else:
                # No order number - check draft first, then show latest completed order
                logger.info("🔍 No order number provided - searching for latest orders...")

                # 1. Check for incomplete order in draft_cart_items FIRST
                try:
                    cart_summary = CartService.get_cart_summary(session.get('id'), session_lang)
                    items = cart_summary.get('items', [])
                    logger.info(f"📦 Found {len(items)} items in draft_cart_items")
                except Exception as e:
                    logger.error(f"❌ Error getting cart: {e}")
                    items = []

                if items:
                    # Has draft session - show ONLY the draft
                    items_text = '\n'.join([
                        f"• {item.get('name_ar' if is_arabic else 'name_en', item.get('name', 'صنف' if is_arabic else 'Item'))} ×{int(item.get('qty', 1))}"
                        for item in items
                    ])

                    # Calculate totals from CartService
                    totals = CartService.calculate_totals(session.get('id'))
                    subtotal = totals['subtotal']
                    delivery_fee = totals['delivery_fee']
                    total = totals['total']

                    if is_arabic:
                        reply = f"""✅ **لديك طلب حالي قيد الإعداد:**

{items_text}

💰 المجموع الفرعي: {subtotal:.3f} ر.ع
🚚 رسوم التوصيل: {delivery_fee:.3f} ر.ع
💵 الإجمالي: {total:.3f} ر.ع

هل تريد إكمال هذا الطلب؟"""
                    else:
                        reply = f"""✅ **You have a current order in progress:**

{items_text}

💰 Subtotal: {subtotal:.3f} OMR
🚚 Delivery Fee: {delivery_fee:.3f} OMR
💵 Total: {total:.3f} OMR

Would you like to complete this order?"""

                    logger.info("✅ Showing draft session order")
                else:
                    # No draft - show latest COMPLETED order with date
                    logger.info("🔍 No draft session - searching for completed orders...")
                    try:
                        # Get chat UUID first
                        chat_response = supabase.table('chats') \
                            .select('id') \
                            .eq('chat_id', chat_id) \
                            .single() \
                            .execute()

                        if chat_response.data:
                            chat_uuid = chat_response.data.get('id')
                            logger.info(f"✅ Found chat UUID: {chat_uuid}")

                            response = supabase.table('orders') \
                                .select('*') \
                                .eq('chat_id', chat_uuid) \
                                .order('created_at', desc=True) \
                                .limit(1) \
                                .execute()

                            if response.data:
                                order_info = response.data[0]
                                logger.info(f"✅ Found completed order: {order_info.get('order_id')}")
                                order_details = await build_order_details(order_info, is_arabic)

                                # Format the date
                                created_at = order_info.get('created_at', '')
                                try:
                                    # Parse ISO format: 2025-10-07T10:30:00+00:00
                                    dt = datetime.fromisoformat(created_at.replace('Z', '+00:00'))
                                    if is_arabic:
                                        # Format: 7 أكتوبر 2025، الساعة 1:30 م
                                        months_ar = ['يناير', 'فبراير', 'مارس', 'أبريل', 'مايو', 'يونيو',
                                                    'يوليو', 'أغسطس', 'سبتمبر', 'أكتوبر', 'نوفمبر', 'ديسمبر']
                                        month_name = months_ar[dt.month - 1]
                                        hour = dt.hour
                                        minute = dt.minute
                                        period = 'م' if hour >= 12 else 'ص'
                                        if hour > 12:
                                            hour -= 12
                                        elif hour == 0:
                                            hour = 12
                                        date_str = f"{dt.day} {month_name} {dt.year}، الساعة {hour}:{minute:02d} {period}"
                                    else:
                                        # Format: October 7, 2025 at 1:30 PM
                                        date_str = dt.strftime("%B %d, %Y at %I:%M %p")
                                except Exception as e:
                                    logger.error(f"Error parsing date: {e}")
                                    date_str = created_at

                                if is_arabic:
                                    reply = f"""✅ **آخر طلب لك:**

{order_details}

📅 تاريخ الطلب: {date_str}"""
                                else:
                                    reply = f"""✅ **Your last order:**

{order_details}

📅 Order Date: {date_str}"""

                                logger.info("✅ Showing completed order with date")
                            else:
                                logger.info("ℹ️ No completed orders found")
                                if is_arabic:
                                    reply = "لم أجد أي طلبات لك. هل تريد البدء بطلب جديد؟"
                                else:
                                    reply = "I couldn't find any orders for you. Would you like to start a new order?"
                        else:
                            logger.warning(f"❌ Chat not found for {chat_id}")
                            if is_arabic:
                                reply = "لم أجد أي طلبات لك. هل تريد البدء بطلب جديد؟"
                            else:
                                reply = "I couldn't find any orders for you. Would you like to start a new order?"
                    except Exception as e:
                        logger.error(f"❌ Error fetching latest order: {e}")
                        if is_arabic:
                            reply = "عذراً، حدث خطأ أثناء البحث عن طلباتك. يرجى المحاولة مرة أخرى."
                        else:
                            reply = "Sorry, an error occurred while searching for your orders. Please try again."
        elif action == 'escalate_support':
            # Handle contact request / escalate to support
            phone_number = data.get('phone_number') if data else None
            reason = data.get('reason', 'customer_requested_contact') if data else 'customer_requested_contact'

            # Log the support request
            logger.info(f"📞 Support escalation requested by {chat_id}")
            logger.info(f"   Phone: {phone_number or 'Not provided yet'}")
            logger.info(f"   Reason: {reason}")

            # TODO: Send notification to support team (e.g., via email, Slack, etc.)
            # For now, just log it

            # If phone number is provided, save it to the session
            if phone_number:
                try:
                    supabase.table('draft_sessions').update({
                        'contact_phone': phone_number
                    }).eq('id', session_id).execute()
                    logger.info(f"✅ Contact phone saved to session: {phone_number}")
                except Exception as e:
                    logger.error(f"❌ Error saving contact phone: {e}")
        elif action == 'cancel_order':
            # Check if cancelling a confirmed order or draft session
            order_number = data.get('order_number') if data else None
            success = await handle_cancel_order(session_id, order_number, chat_id)

            if not success:
                # Update reply with error message
                if any(ord(c) > 127 for c in message_text):
                    reply = "عذراً، حدث خطأ أثناء إلغاء الطلب. يرجى المحاولة مرة أخرى."
                else:
                    reply = "Sorry, an error occurred while cancelling the order. Please try again."
        elif action == 'clear_cart':
            # 🆕 حذف/مسح السلة
            logger.info(f"🗑️ Executing clear_cart action for session: {session_id}")

            try:
                success = CartService.clear_cart(session_id)

                if success:
                    logger.info(f"✅ Cart cleared successfully for session: {session_id}")
                    # الرد موجود بالفعل من handler (من yes_no_handler)
                else:
                    logger.error(f"❌ Failed to clear cart for session: {session_id}")
                    # Update reply with error message
                    if any(ord(c) > 127 for c in message_text):
                        reply = "عذراً، حدث خطأ أثناء حذف السلة. يرجى المحاولة مرة أخرى."
                    else:
                        reply = "Sorry, an error occurred while clearing the cart. Please try again."
            except Exception as e:
                logger.error(f"❌ Error clearing cart: {e}")
                if any(ord(c) > 127 for c in message_text):
                    reply = "عذراً، حدث خطأ أثناء حذف السلة. يرجى المحاولة مرة أخرى."
                else:
                    reply = "Sorry, an error occurred while clearing the cart. Please try again."
        elif action == 'repeat_order':
            # Repeat last completed order
            session_lang = session.get('lang', 'ar')
            is_arabic = (session_lang == 'ar')

            try:
                # Get chat UUID
                chat_response = supabase.table('chats') \
                    .select('id') \
                    .eq('chat_id', chat_id) \
                    .single() \
                    .execute()

                if chat_response.data:
                    chat_uuid = chat_response.data.get('id')

                    # Get latest completed order
                    order_response = supabase.table('orders') \
                        .select('*') \
                        .eq('chat_id', chat_uuid) \
                        .order('created_at', desc=True) \
                        .limit(1) \
                        .execute()

                    if order_response.data:
                        last_order = order_response.data[0]

                        # Get order items
                        items_response = supabase.table('order_items') \
                            .select('*') \
                            .eq('order_id', last_order['id']) \
                            .execute()

                        if items_response.data:
                            # Copy items to draft session
                            items = []
                            subtotal = 0

                            for item in items_response.data:
                                item_data = {
                                    'code': item['item_code'],
                                    'name': item['item_name'],
                                    'qty': float(item['qty']),
                                    'price': float(item['unit_price_omr'])
                                }
                                items.append(item_data)
                                subtotal += float(item['subtotal_omr'])

                            # Update session with repeated order
                            await session_manager.update_session(session_id, {
                                'stage': 'address',
                                'items': items,
                                'subtotal': subtotal
                            })

                            logger.info(f"✅ Repeated order with {len(items)} items, subtotal: {subtotal}")

                            # AI will provide the reply with order details
                        else:
                            logger.warning(f"❌ No items found for order {last_order['id']}")
                            if is_arabic:
                                reply = "عذراً، لم أتمكن من تحميل تفاصيل الطلب السابق."
                            else:
                                reply = "Sorry, I couldn't load the previous order details."
                    else:
                        logger.warning(f"❌ No completed orders found for chat {chat_id}")
                        if is_arabic:
                            reply = "ليس لديك طلبات سابقة لتكرارها. ماذا تريد أن تطلب؟"
                        else:
                            reply = "You don't have any previous orders to repeat. What would you like to order?"
                else:
                    logger.warning(f"❌ Chat not found for {chat_id}")
                    if is_arabic:
                        reply = "عذراً، حدث خطأ. يرجى المحاولة مرة أخرى."
                    else:
                        reply = "Sorry, an error occurred. Please try again."
            except Exception as e:
                logger.error(f"❌ Error repeating order: {e}")
                if is_arabic:
                    reply = "عذراً، حدث خطأ أثناء تكرار الطلب. يرجى المحاولة مرة أخرى."
                else:
                    reply = "Sorry, an error occurred while repeating the order. Please try again."

        elif action == 'view_last_order':
            # View last order details (without reordering)
            session_lang = session.get('lang', 'ar')

            try:
                # Get chat UUID
                chat_response = supabase.table('chats') \
                    .select('id') \
                    .eq('chat_id', chat_id) \
                    .single() \
                    .execute()

                if not chat_response.data:
                    logger.warning(f"❌ Chat not found for {chat_id}")
                    if session_lang == 'ar':
                        reply = "لا يوجد طلب سابق لعرضه."
                    else:
                        reply = "No previous order to display."
                else:
                    chat_uuid = chat_response.data.get('id')

                    # Get latest order
                    order_response = supabase.table('orders') \
                        .select('id, order_id, total_omr, created_at, status') \
                        .eq('chat_id', chat_uuid) \
                        .order('created_at', desc=True) \
                        .limit(1) \
                        .execute()

                    if order_response.data and len(order_response.data) > 0:
                        last_order = order_response.data[0]
                        order_id = last_order['id']
                        order_number = last_order.get('order_id', order_id)

                        logger.info(f"📜 Found last order: {order_number}")

                        # Get order items
                        items_response = supabase.table('order_items') \
                            .select('*') \
                            .eq('order_id', order_id) \
                            .execute()

                        if items_response.data and len(items_response.data) > 0:
                            # Format order details
                            items_list = []
                            for item in items_response.data:
                                items_list.append({
                                    'name': item['item_name'],
                                    'quantity': int(item['qty']),
                                    'price': float(item['unit_price_omr']),
                                    'subtotal': float(item['subtotal_omr'])
                                })

                            # Build reply
                            if session_lang == 'ar':
                                reply = f"📜 **آخر طلب لك (رقم {order_number})**\n\n"
                                for item in items_list:
                                    reply += f"• {item['name']} × {item['quantity']} = {item['subtotal']:.3f} ر.ع.\n"
                                reply += f"\n**الإجمالي:** {last_order['total_omr']:.3f} ر.ع.\n"
                                reply += f"**الحالة:** {last_order['status']}\n\n"
                                reply += "هل تريد إعادة هذا الطلب؟"
                            else:
                                reply = f"📜 **Your Last Order (#{order_number})**\n\n"
                                for item in items_list:
                                    reply += f"• {item['name']} × {item['quantity']} = {item['subtotal']:.3f} OMR\n"
                                reply += f"\n**Total:** {last_order['total_omr']:.3f} OMR\n"
                                reply += f"**Status:** {last_order['status']}\n\n"
                                reply += "Would you like to reorder this?"

                            # Set flag to indicate waiting for reorder confirmation
                            await session_manager.update_session(session['id'], {
                                'waiting_for_reorder_confirmation': True,
                                'last_viewed_order_id': order_id
                            })
                        else:
                            logger.warning(f"❌ No items found for order {order_id}")
                            if session_lang == 'ar':
                                reply = "لا يوجد طلب سابق لعرضه."
                            else:
                                reply = "No previous order to display."
                    else:
                        logger.info(f"ℹ️ No previous orders found for chat {chat_id}")
                        if session_lang == 'ar':
                            reply = "لا يوجد طلب سابق لعرضه."
                        else:
                            reply = "No previous order to display."

            except Exception as e:
                logger.error(f"❌ Error in view_last_order: {str(e)}")
                import traceback
                logger.error(traceback.format_exc())
                if session_lang == 'ar':
                    reply = "عذراً، حدث خطأ أثناء عرض الطلب. يرجى المحاولة مرة أخرى."
                else:
                    reply = "Sorry, an error occurred while viewing the order. Please try again."

        elif action == 'reorder_last_order':
            # Reorder last completed order - creates new cart
            from app.handlers.reorder_handler import reorder_handler

            session_lang = session.get('lang', 'ar')

            # Get chat UUID for querying orders
            try:
                chat_response = supabase.table('chats') \
                    .select('id') \
                    .eq('chat_id', chat_id) \
                    .single() \
                    .execute()

                if not chat_response.data:
                    logger.warning(f"❌ Chat not found for {chat_id}")
                    reply = reorder_handler.format_no_previous_order(session_lang)
                    # Continue to send reply
                else:
                    chat_uuid = chat_response.data.get('id')

                    # Get latest completed order
                    order_response = supabase.table('orders') \
                        .select('id, order_id, total_omr, created_at, status') \
                        .eq('chat_id', chat_uuid) \
                        .in_('status', ['confirmed', 'delivered']) \
                        .order('created_at', desc=True) \
                        .limit(1) \
                        .execute()

                    if order_response.data and len(order_response.data) > 0:
                        last_order = order_response.data[0]
                        order_id = last_order['id']
                        order_number = last_order.get('order_id', order_id)

                        logger.info(f"🔄 Found last order: {order_number}")

                        # Get order items
                        items_response = supabase.table('order_items') \
                            .select('*') \
                            .eq('order_id', order_id) \
                            .execute()

                        if items_response.data and len(items_response.data) > 0:
                            # Prepare items for new cart
                            new_cart_items = []
                            subtotal = 0.0

                            for item in items_response.data:
                                # Get menu item details for name_ar and name_en
                                menu_item_response = supabase.table('menu_items') \
                                    .select('name_ar, name_en, price_omr, is_available') \
                                    .eq('code', item['item_code']) \
                                    .single() \
                                    .execute()

                                if menu_item_response.data:
                                    menu_item = menu_item_response.data

                                    # Check if item is still available
                                    if menu_item.get('is_available', True):
                                        cart_item = {
                                            'code': item['item_code'],
                                            'name': item['item_name'],
                                            'name_ar': menu_item.get('name_ar', item['item_name']),
                                            'name_en': menu_item.get('name_en', item['item_name']),
                                            'quantity': int(item['qty']),
                                            'price': float(menu_item.get('price_omr', item['unit_price_omr']))
                                        }
                                        new_cart_items.append(cart_item)
                                        subtotal += cart_item['quantity'] * cart_item['price']

                            if len(new_cart_items) > 0:
                                # Calculate totals
                                discount = 0.0
                                delivery_fee = cart_formatter.DEFAULT_DELIVERY_FEE
                                totals = cart_formatter.calculate_totals(new_cart_items, discount, delivery_fee)

                                # Update session with new cart
                                await handle_update_session(session_id, {
                                    'items': new_cart_items,
                                    'subtotal': totals['subtotal'],
                                    'total': totals['total'],
                                    'delivery_fee': totals['delivery_fee'],
                                    'discount': discount
                                }, session)

                                logger.info(f"✅ Reordered {len(new_cart_items)} items, total: {totals['total']} OMR")

                                # Format success message
                                reply = reorder_handler.format_reorder_success(
                                    new_cart_items,
                                    totals,
                                    session_lang
                                )
                            else:
                                # All items unavailable
                                logger.warning(f"⚠️ All items from last order are unavailable")
                                reply = reorder_handler.format_no_previous_order(session_lang)
                        else:
                            logger.warning(f"❌ No items found for order {order_id}")
                            reply = reorder_handler.format_no_previous_order(session_lang)
                    else:
                        # No previous orders
                        logger.info(f"ℹ️ No previous orders found for chat {chat_id}")
                        reply = reorder_handler.format_no_previous_order(session_lang)

            except Exception as e:
                logger.error(f"❌ Error in reorder_last_order: {str(e)}")
                import traceback
                logger.error(traceback.format_exc())
                if session_lang == 'ar':
                    reply = "عذراً، حدث خطأ أثناء إعادة الطلب. يرجى المحاولة مرة أخرى."
                else:
                    reply = "Sorry, an error occurred while reordering. Please try again."

        # Check if intent or message is related to menu (send images or text)
        menu_keywords_ar = ['منيو', 'قائمة', 'القائمة', 'المنيو', 'قايمة']
        menu_keywords_en = ['menu', 'list', 'items']

        message_lower = message_text.lower()
        is_menu_request = (
            (intent and 'menu' in intent.lower()) or
            any(keyword in message_lower for keyword in menu_keywords_ar) or
            any(keyword in message_lower for keyword in menu_keywords_en)
        )

        if is_menu_request:
            # Use session language instead of message text
            session_lang = session.get('lang', 'ar')
            is_arabic = (session_lang == 'ar')

            # Check if user explicitly wants text menu
            text_keywords_ar = ['نص', 'نصي', 'كتابة', 'مكتوب']
            text_keywords_en = ['text', 'written', 'txt']
            wants_text = any(keyword in message_lower for keyword in text_keywords_ar + text_keywords_en)

            if wants_text:
                # User explicitly wants text menu - AI will handle this in reply
                logger.info(f"📝 User explicitly requested TEXT menu: '{message_text}' - AI will provide text")
                # Don't send images, let AI provide text menu in reply
            else:
                # طبقة حماية: التحقق من أن الرسالة لا تحتوي على كلمات "سعر"
                from app.intents.intent_guard import intent_guard

                if not intent_guard.validate_intent_for_action(
                    message_text,
                    IntentType.SHOW_MENU,
                    "send_menu_images"
                ):
                    logger.warning(
                        f"⚠️ Blocked sending menu images: message contains price keywords"
                    )
                    # لا ترسل صور المنيو، دع الـ AI يتعامل مع السؤال عن السعر
                else:
                    # Default: Send menu images ONLY (no text from AI)
                    logger.info(f"📸 Detected menu request in message: '{message_text}' - Sending menu images ONLY")
                    success = await send_menu_images(chat_id, language='ar' if is_arabic else 'en')

                    if success:
                        # Send ONLY confirmation message, no menu text
                        if is_arabic:
                            reply = "تفضل قائمة الطعام 📸\n\nهل تريد طلب شيء؟"
                        else:
                            reply = "Here's our menu 📸\n\nWould you like to order something?"

                        logger.info("✅ Sent menu images only, no text menu")
                    else:
                        logger.warning("⚠️ Failed to send menu images, falling back to AI text response")

        # Save customer message to conversation history
        try:
            await conversation_history_service.save_message(
                chat_id=chat_id,
                text_content=message_text,
                from_me=False
            )
        except Exception as e:
            logger.error(f"❌ Error saving customer message to history: {e}")

        # Save bot reply to conversation history (only if reply exists)
        if reply:
            try:
                await conversation_history_service.save_message(
                    chat_id=chat_id,
                    text_content=reply,
                    from_me=True
                )
            except Exception as e:
                logger.error(f"❌ Error saving bot reply to history: {e}")

        # Send reply to customer (only if reply exists and wasn't sent already)
        if reply:
            phone_number = evolution_api.extract_phone_number(chat_id)

            # 🔍 Debug: Log sending details
            logger.info(f"📤 Preparing to send reply:")
            logger.info(f"   - chat_id: {chat_id}")
            logger.info(f"   - phone_number: {phone_number}")
            logger.info(f"   - reply preview: {reply[:100]}...")

            # Stop typing indicator before sending message (disabled for faster response)
            # await evolution_api.send_presence(chat_id, "available")

            await evolution_api.send_text_message(phone_number, reply)

            logger.info(f"✅ Reply sent to {chat_id} (phone: {phone_number})")
        else:
            logger.info(f"No reply to send (already sent or empty)")

    except Exception as e:
        logger.error(f"Error processing message: {e}")
        # Send error message to customer
        try:
            phone_number = evolution_api.extract_phone_number(chat_id)

            # Stop typing indicator
            await evolution_api.send_presence(chat_id, "available")

            # Detect language from session or message
            error_message = "عذراً، حدث خطأ في معالجة رسالتك. يرجى المحاولة مرة أخرى."
            try:
                session = await session_manager.get_or_create_session(chat_id)
                if session and session.get('lang') == 'en':
                    error_message = "Sorry, an error occurred while processing your message. Please try again."
            except:
                # If can't get session, use default Arabic message
                pass

            await evolution_api.send_text_message(
                phone_number,
                error_message
            )
        except:
            pass


async def handle_update_session(session_id: str, data: Dict, session: Dict = None):
    """Handle session update action"""
    try:
        if not data:
            return

        updates = {}

        # Get session object if not provided
        if session is None:
            session = await session_manager.get_session(session_id)

        # Update stage
        if 'stage' in data:
            updates['stage'] = data['stage']

        # ⚠️ DEPRECATED: 'items' column has been removed from draft_sessions
        # Items are now stored in draft_cart_items table via cart_service
        # This code is kept for backward compatibility but does nothing
        if 'items' in data:
            logger.warning("⚠️ Attempted to update 'items' column which no longer exists. Use cart_service instead.")
            # Skip this update - items are managed by cart_service now
            pass
            updates['subtotal'] = subtotal

        # Update address
        if 'address' in data:
            updates['address'] = data['address']

        # Update payment method
        if 'payment_method' in data:
            updates['payment_method'] = data['payment_method']

        # Update notes
        if 'notes' in data or 'delivery_notes' in data:
            updates['notes'] = data.get('notes') or data.get('delivery_notes')

        # ✅ Update waiting states for item selection flow
        if 'waiting_for_item_choice' in data:
            updates['waiting_for_item_choice'] = data['waiting_for_item_choice']

        if 'pending_item_choices' in data:
            updates['pending_item_choices'] = data['pending_item_choices']

        if 'waiting_for_quantity' in data:
            updates['waiting_for_quantity'] = data['waiting_for_quantity']

        if 'selected_item_code' in data:
            updates['selected_item_code'] = data['selected_item_code']

        # Note: pending_quantity is stored in memory (session) only, not in DB
        # Store it in the session object directly
        if 'pending_quantity' in data:
            session['pending_quantity'] = data['pending_quantity']
            logger.info(f"💾 Stored pending_quantity in memory: {data['pending_quantity']}")

        # 🆕 Clear waiting_for_more_items flag when user makes a choice
        if 'waiting_for_more_items' in data:
            updates['waiting_for_more_items'] = data['waiting_for_more_items']

        # 🆕 Update last_question if provided
        if 'last_question' in data:
            from app.utils.question_context_manager import question_context_manager, QuestionType
            from datetime import datetime

            # Set the question with timestamp
            last_q = data['last_question']
            if isinstance(last_q, dict) and last_q.get('type'):
                # فقط أضف timestamp إذا لم يكن السؤال "none"
                if last_q.get('type') != QuestionType.NONE.value and not last_q.get('asked_at'):
                    last_q['asked_at'] = datetime.utcnow().isoformat()
                updates['last_question'] = last_q
                logger.info(f"❓ Setting last_question: {last_q.get('type')}")

        # 💾 Store last_item_name for price queries
        if 'last_item_name' in data:
            updates['last_item_name'] = data['last_item_name']

            # Also store in context_manager for cross-session access
            try:
                session_result = supabase.table('draft_sessions').select('chat_id').eq('id', session_id).execute()
                if session_result.data:
                    chat_id = session_result.data[0].get('chat_id')
                    if chat_id:
                        context_manager.set_last_item(chat_id, data['last_item_name'])
            except Exception as e:
                logger.warning(f"⚠️ Could not store last item in context manager: {e}")

        if updates:
            await session_manager.update_session(session_id, updates)
            logger.info(f"Session {session_id} updated: {list(updates.keys())}")

    except Exception as e:
        logger.error(f"Error handling update_session: {e}")
        raise


async def build_order_details(order_info: Dict, is_arabic: bool = True) -> str:
    """
    Build order details string for display

    Args:
        order_info: Order information dictionary
        is_arabic: Whether to use Arabic language (default: True)

    Returns:
        Formatted order details string
    """
    try:

        # Get order items
        order_id = order_info.get('id')
        order_items_response = supabase.table('order_items') \
            .select('*') \
            .eq('order_id', order_id) \
            .execute()

        items_list = []
        if order_items_response.data:
            for item in order_items_response.data:
                # Use item_name directly from order_items (correct column name)
                name = item.get('item_name', 'صنف' if is_arabic else 'Item')
                quantity = float(item.get('qty', 1))  # ✅ Correct: matches order_items.qty
                price = float(item.get('unit_price_omr', 0))  # ✅ Correct: matches order_items.unit_price_omr
                total = quantity * price

                if is_arabic:
                    items_list.append(f"• {int(quantity)} × {name} ({price:.3f} ر.ع للوحدة) = {total:.3f} ر.ع")
                else:
                    items_list.append(f"• {int(quantity)} × {name} ({price:.3f} OMR each) = {total:.3f} OMR")

        items_text = '\n'.join(items_list) if items_list else (
            "لا توجد أصناف" if is_arabic else "No items"
        )

        # Build status text
        status = order_info.get('status', 'unknown')
        status_map_ar = {
            'confirmed': 'مؤكد',
            'preparing': 'قيد التحضير',
            'ready': 'جاهز',
            'out_for_delivery': 'في الطريق',
            'delivered': 'تم التوصيل',
            'cancelled': 'ملغي'
        }
        status_map_en = {
            'confirmed': 'Confirmed',
            'preparing': 'Preparing',
            'ready': 'Ready',
            'out_for_delivery': 'Out for Delivery',
            'delivered': 'Delivered',
            'cancelled': 'Cancelled'
        }

        status_text = status_map_ar.get(status, status) if is_arabic else status_map_en.get(status, status)

        # Build details
        if is_arabic:
            details = f"""📋 تفاصيل الطلب رقم {order_info.get('order_id', '')}:

{items_text}

💰 المجموع الفرعي: {order_info.get('subtotal', 0):.3f} ر.ع
🚚 رسوم التوصيل: {order_info.get('delivery_fee', 0):.3f} ر.ع
💵 الإجمالي: {order_info.get('total_amount', 0):.3f} ر.ع

📍 العنوان: {order_info.get('delivery_address', 'غير محدد')}
💳 طريقة الدفع: {order_info.get('payment_method', 'غير محدد')}
📊 الحالة: {status_text}"""
        else:
            details = f"""📋 Order #{order_info.get('order_id', '')} Details:

{items_text}

💰 Subtotal: {order_info.get('subtotal', 0):.3f} OMR
🚚 Delivery Fee: {order_info.get('delivery_fee', 0):.3f} OMR
💵 Total: {order_info.get('total_amount', 0):.3f} OMR

📍 Address: {order_info.get('delivery_address', 'Not specified')}
💳 Payment: {order_info.get('payment_method', 'Not specified')}
📊 Status: {status_text}"""

        return details

    except Exception as e:
        logger.error(f"Error building order details: {e}")
        return "Error loading order details" if 'en' in intent.lower() else "خطأ في تحميل تفاصيل الطلب"


async def get_latest_order_for_customer(chat_id: str) -> Dict:
    """
    Get the latest order for a customer (completed or incomplete)

    Args:
        chat_id: Chat ID

    Returns:
        Dictionary with order info and type ('completed' or 'incomplete') or None
    """
    try:
        # First, check for incomplete order in draft_sessions
        draft_response = supabase.table('draft_sessions') \
            .select('*') \
            .eq('chat_id', chat_id) \
            .eq('status', 'active') \
            .order('updated_at', desc=True) \
            .limit(1) \
            .execute()

        if draft_response.data and len(draft_response.data) > 0:
            session = draft_response.data[0]
            # Check if session has items (not just created)
            session_items = get_session_items(session.get('id'))
            if len(session_items) > 0:
                logger.info(f"Found incomplete order for {chat_id}")
                return {
                    'type': 'incomplete',
                    'data': session
                }

        # Second, check for completed orders in orders table
        # Get chat UUID from chat_id (string)
        chat_response = supabase.table('chats') \
            .select('id') \
            .eq('chat_id', chat_id) \
            .single() \
            .execute()

        if not chat_response.data:
            logger.warning(f"❌ Chat not found for {chat_id}")
            return None

        chat_uuid = chat_response.data.get('id')

        # Search by chat UUID
        order_response = supabase.table('orders') \
            .select('*') \
            .eq('chat_id', chat_uuid) \
            .order('created_at', desc=True) \
            .limit(1) \
            .execute()

        if order_response.data and len(order_response.data) > 0:
            order = order_response.data[0]
            logger.info(f"✅ Found completed order {order.get('order_id')} for {chat_id}")
            return {
                'type': 'completed',
                'data': order
            }

        logger.info(f"ℹ️ No orders found for {chat_id}")
        return None

    except Exception as e:
        logger.error(f"Error getting latest order: {e}")
        return None


async def handle_track_order(order_number: str, chat_id: str) -> Dict:
    """
    Handle order tracking action

    Args:
        order_number: Order number to track (e.g., "2025-019", "019-2025", or "10001")
        chat_id: Chat ID (string) to verify ownership

    Returns:
        Order information dictionary or None if not found
    """
    try:
        # Normalize order number
        normalized_order_number = order_number.strip()

        # Get chat UUID from chat_id (string)
        chat_response = supabase.table('chats') \
            .select('id') \
            .eq('chat_id', chat_id) \
            .single() \
            .execute()

        if not chat_response.data:
            logger.warning(f"❌ Chat not found for {chat_id}")
            return None

        chat_uuid = chat_response.data.get('id')
        logger.debug(f"🔍 Found chat UUID: {chat_uuid} for chat_id: {chat_id}")

        # Get phone number from chat to match with orders
        phone_number = None
        try:
            # Get phone from chats table
            chat_data = supabase.table('chats') \
                .select('phone_number') \
                .eq('chat_id', chat_id) \
                .single() \
                .execute()

            if chat_data.data:
                phone_number = chat_data.data.get('phone_number')
                logger.debug(f"🔍 Found phone: {phone_number} for chat_id: {chat_id}")
        except Exception as e:
            logger.warning(f"⚠️ Could not get phone number for chat {chat_id}: {e}")

        # Try to find order with different formats
        order_response = None

        # Case 1: Direct match by order_id (e.g., "10025")
        try:
            query = supabase.table('orders') \
                .select('*') \
                .eq('order_id', normalized_order_number)

            # If we have phone number, filter by it for security
            if phone_number:
                query = query.eq('phone', phone_number)

            order_response = query.single().execute()

            if order_response.data:
                logger.info(f"✅ Found order {order_response.data.get('order_id')} for chat {chat_id}")
                return order_response.data
        except Exception as e:
            logger.debug(f"Direct match failed for {normalized_order_number}: {e}")

        # Case 2: If order number contains '-', try reversed format
        if '-' in normalized_order_number:
            parts = normalized_order_number.split('-')
            if len(parts) == 2:
                # Try reversed format (e.g., "019-2025" -> "2025-019")
                reversed_format = f"{parts[1]}-{parts[0]}"

                try:
                    query = supabase.table('orders') \
                        .select('*') \
                        .eq('order_id', reversed_format)

                    # If we have phone number, filter by it for security
                    if phone_number:
                        query = query.eq('phone', phone_number)

                    order_response = query.single().execute()

                    if order_response.data:
                        logger.info(f"✅ Found order {order_response.data.get('order_id')} for chat {chat_id} (reversed format)")
                        return order_response.data
                except Exception as e:
                    logger.debug(f"Reversed format match failed for {reversed_format}: {e}")

        logger.warning(f"❌ Order {order_number} not found for chat {chat_id}")
        return None

    except Exception as e:
        logger.error(f"Error tracking order {order_number}: {e}")
        return None


async def handle_cancel_order(session_id: str, order_number: str = None, chat_id: str = None):
    """
    Handle order cancellation action

    Args:
        session_id: Session ID to cancel
        order_number: Optional order number for confirmed orders
        chat_id: Chat ID for verification
    """
    try:
        # If order_number is provided, cancel the confirmed order
        if order_number and chat_id:
            # Get chat UUID from chat_id
            chat_response = supabase.table('chats') \
                .select('id') \
                .eq('chat_id', chat_id) \
                .single() \
                .execute()

            if not chat_response.data:
                logger.warning(f"❌ Chat not found for {chat_id}")
                return False

            chat_uuid = chat_response.data.get('id')

            # Find and cancel the order by chat_uuid
            order_response = supabase.table('orders') \
                .select('*') \
                .eq('chat_id', chat_uuid) \
                .eq('order_id', order_number) \
                .single() \
                .execute()

            if order_response.data:
                # Update order status to cancelled
                supabase.table('orders') \
                    .update({'status': 'cancelled', 'updated_at': datetime.utcnow().isoformat()}) \
                    .eq('id', order_response.data.get('id')) \
                    .execute()

                logger.info(f"✅ Order {order_number} cancelled for chat {chat_id}")
                return True

        # Cancel the draft session
        await session_manager.cancel_session(session_id)
        logger.info(f"Session {session_id} cancelled")
        return True

    except Exception as e:
        logger.error(f"Error handling cancel_order: {e}")
        return False


async def send_menu_images(chat_id: str, language: str = 'ar') -> bool:
    """
    Send menu images to customer

    Args:
        chat_id: WhatsApp chat ID
        language: Language for captions ('ar' or 'en')

    Returns:
        True if successful, False otherwise
    """
    try:
        logger.info(f"📸 Sending menu images to {chat_id} (language={language})")

        # Fetch active menu images from database
        response = supabase.table('menu_images') \
            .select('*') \
            .eq('is_active', True) \
            .eq('image_type', 'menu') \
            .order('display_order') \
            .execute()

        images = response.data

        if not images:
            logger.warning("⚠️ No menu images found in database")
            return False

        logger.info(f"📸 Found {len(images)} menu images to send")

        # Send each image
        for image in images:
            caption = image.get('caption_ar') if language == 'ar' else image.get('caption_en')
            image_url = image.get('image_url')

            if not image_url:
                logger.warning(f"⚠️ Image {image.get('id')} has no URL, skipping")
                continue

            logger.info(f"📤 Sending image: {image_url}")

            # Send image via Evolution API
            await evolution_api.send_media_message(
                phone_number=chat_id,
                media_url=image_url,
                caption=caption or "",
                media_type='image'
            )

            logger.info(f"✅ Sent menu image {image.get('image_number')}")

        logger.info(f"✅ Successfully sent all {len(images)} menu images")
        return True

    except Exception as e:
        logger.error(f"❌ Error sending menu images: {str(e)}")
        return False


@router.get("/webhook/health")
async def webhook_health():
    """Health check endpoint"""
    return {
        "status": "healthy",
        "service": "restaurant-chatbot-webhook"
    }

