"""
Menu Management API Routes
Handles menu images and menu items management
"""

from fastapi import APIRouter, HTTPException, UploadFile, File
from typing import List, Dict, Optional
import logging
from supabase import create_client
from app.config import settings

router = APIRouter(prefix="/menu", tags=["menu"])
logger = logging.getLogger(__name__)

# Initialize Supabase client
supabase = create_client(settings.SUPABASE_URL, settings.SUPABASE_KEY)


@router.get("/images")
async def get_menu_images(
    active_only: bool = True,
    image_type: Optional[str] = None
) -> Dict:
    """
    Get all menu images
    
    Args:
        active_only: Return only active images (default: True)
        image_type: Filter by image type (menu, offer, banner)
    
    Returns:
        List of menu images
    """
    try:
        logger.info(f"📸 Fetching menu images (active_only={active_only}, type={image_type})")
        
        query = supabase.table('menu_images').select('*')
        
        if active_only:
            query = query.eq('is_active', True)
        
        if image_type:
            query = query.eq('image_type', image_type)
        
        response = query.order('display_order').execute()
        
        logger.info(f"✅ Found {len(response.data)} menu images")
        
        return {
            "success": True,
            "count": len(response.data),
            "images": response.data
        }
        
    except Exception as e:
        logger.error(f"❌ Error fetching menu images: {str(e)}")
        raise HTTPException(status_code=500, detail=str(e))


@router.get("/images/{image_id}")
async def get_menu_image(image_id: str) -> Dict:
    """
    Get a specific menu image by ID
    
    Args:
        image_id: UUID of the image
    
    Returns:
        Menu image details
    """
    try:
        logger.info(f"📸 Fetching menu image: {image_id}")
        
        response = supabase.table('menu_images') \
            .select('*') \
            .eq('id', image_id) \
            .single() \
            .execute()
        
        if not response.data:
            raise HTTPException(status_code=404, detail="Image not found")
        
        logger.info(f"✅ Found menu image: {image_id}")
        
        return {
            "success": True,
            "image": response.data
        }
        
    except HTTPException:
        raise
    except Exception as e:
        logger.error(f"❌ Error fetching menu image: {str(e)}")
        raise HTTPException(status_code=500, detail=str(e))


@router.post("/images")
async def create_menu_image(data: Dict) -> Dict:
    """
    Create a new menu image entry
    
    Args:
        data: Image data (image_url, caption_ar, caption_en, etc.)
    
    Returns:
        Created image details
    """
    try:
        logger.info(f"📸 Creating new menu image")
        
        # Validate required fields
        if 'image_url' not in data:
            raise HTTPException(status_code=400, detail="image_url is required")
        
        # Set defaults
        if 'image_type' not in data:
            data['image_type'] = 'menu'
        if 'is_active' not in data:
            data['is_active'] = True
        if 'display_order' not in data:
            # Get max display_order and add 1
            max_order_response = supabase.table('menu_images') \
                .select('display_order') \
                .order('display_order', desc=True) \
                .limit(1) \
                .execute()
            
            max_order = max_order_response.data[0]['display_order'] if max_order_response.data else 0
            data['display_order'] = max_order + 1
        
        response = supabase.table('menu_images').insert(data).execute()
        
        logger.info(f"✅ Created menu image: {response.data[0]['id']}")
        
        return {
            "success": True,
            "message": "Menu image created successfully",
            "image": response.data[0]
        }
        
    except HTTPException:
        raise
    except Exception as e:
        logger.error(f"❌ Error creating menu image: {str(e)}")
        raise HTTPException(status_code=500, detail=str(e))


@router.put("/images/{image_id}")
async def update_menu_image(image_id: str, data: Dict) -> Dict:
    """
    Update a menu image
    
    Args:
        image_id: UUID of the image
        data: Updated image data
    
    Returns:
        Updated image details
    """
    try:
        logger.info(f"📸 Updating menu image: {image_id}")
        
        # Check if image exists
        existing = supabase.table('menu_images') \
            .select('id') \
            .eq('id', image_id) \
            .single() \
            .execute()
        
        if not existing.data:
            raise HTTPException(status_code=404, detail="Image not found")
        
        response = supabase.table('menu_images') \
            .update(data) \
            .eq('id', image_id) \
            .execute()
        
        logger.info(f"✅ Updated menu image: {image_id}")
        
        return {
            "success": True,
            "message": "Menu image updated successfully",
            "image": response.data[0]
        }
        
    except HTTPException:
        raise
    except Exception as e:
        logger.error(f"❌ Error updating menu image: {str(e)}")
        raise HTTPException(status_code=500, detail=str(e))


@router.delete("/images/{image_id}")
async def delete_menu_image(image_id: str, soft_delete: bool = True) -> Dict:
    """
    Delete a menu image
    
    Args:
        image_id: UUID of the image
        soft_delete: If True, just set is_active=False (default: True)
    
    Returns:
        Success message
    """
    try:
        logger.info(f"📸 Deleting menu image: {image_id} (soft={soft_delete})")
        
        if soft_delete:
            # Soft delete - just deactivate
            response = supabase.table('menu_images') \
                .update({'is_active': False}) \
                .eq('id', image_id) \
                .execute()
            
            if not response.data:
                raise HTTPException(status_code=404, detail="Image not found")
            
            logger.info(f"✅ Deactivated menu image: {image_id}")
            message = "Menu image deactivated successfully"
        else:
            # Hard delete
            response = supabase.table('menu_images') \
                .delete() \
                .eq('id', image_id) \
                .execute()
            
            if not response.data:
                raise HTTPException(status_code=404, detail="Image not found")
            
            logger.info(f"✅ Deleted menu image: {image_id}")
            message = "Menu image deleted successfully"
        
        return {
            "success": True,
            "message": message
        }
        
    except HTTPException:
        raise
    except Exception as e:
        logger.error(f"❌ Error deleting menu image: {str(e)}")
        raise HTTPException(status_code=500, detail=str(e))


@router.post("/images/reorder")
async def reorder_menu_images(order_data: List[Dict]) -> Dict:
    """
    Reorder menu images
    
    Args:
        order_data: List of {id, display_order} objects
    
    Returns:
        Success message
    """
    try:
        logger.info(f"📸 Reordering {len(order_data)} menu images")
        
        for item in order_data:
            if 'id' not in item or 'display_order' not in item:
                raise HTTPException(
                    status_code=400,
                    detail="Each item must have 'id' and 'display_order'"
                )
            
            supabase.table('menu_images') \
                .update({'display_order': item['display_order']}) \
                .eq('id', item['id']) \
                .execute()
        
        logger.info(f"✅ Reordered menu images successfully")
        
        return {
            "success": True,
            "message": "Menu images reordered successfully"
        }
        
    except HTTPException:
        raise
    except Exception as e:
        logger.error(f"❌ Error reordering menu images: {str(e)}")
        raise HTTPException(status_code=500, detail=str(e))


@router.get("/items")
async def get_menu_items(category: Optional[str] = None) -> Dict:
    """
    Get all menu items
    
    Args:
        category: Filter by category (optional)
    
    Returns:
        List of menu items
    """
    try:
        logger.info(f"🍽️ Fetching menu items (category={category})")
        
        query = supabase.table('menu_items').select('*')
        
        if category:
            query = query.eq('category', category)
        
        response = query.order('code').execute()
        
        logger.info(f"✅ Found {len(response.data)} menu items")
        
        return {
            "success": True,
            "count": len(response.data),
            "items": response.data
        }
        
    except Exception as e:
        logger.error(f"❌ Error fetching menu items: {str(e)}")
        raise HTTPException(status_code=500, detail=str(e))

