"""
Credits Management API Routes
Handles credits/points tracking and management
"""

from fastapi import APIRouter, HTTPException
from typing import Dict, Optional
from pydantic import BaseModel
import logging
from supabase import create_client
from app.config import settings
from app.services.credits_manager import CreditsManager

router = APIRouter(prefix="/credits", tags=["credits"])
logger = logging.getLogger(__name__)

# Initialize Supabase client
supabase = create_client(settings.SUPABASE_URL, settings.SUPABASE_KEY)

# Initialize credits manager
credits_manager = CreditsManager(supabase)


class RefillRequest(BaseModel):
    """Request model for credits refill"""
    amount: int
    admin_id: Optional[str] = None
    description: Optional[str] = None


@router.get("/balance")
async def get_credits_balance() -> Dict:
    """
    Get current credits balance
    
    Returns:
        Current credits balance and information
    """
    try:
        logger.info("📊 Fetching credits balance")
        
        result = await credits_manager.get_credits_balance()
        
        if not result.get('success'):
            raise HTTPException(status_code=500, detail=result.get('error'))
        
        logger.info(f"✅ Credits balance: {result.get('remaining_credits')}/{result.get('total_credits')}")
        
        return result
        
    except HTTPException:
        raise
    except Exception as e:
        logger.error(f"❌ Error fetching credits balance: {str(e)}")
        raise HTTPException(status_code=500, detail=str(e))


@router.post("/refill")
async def refill_credits(request: RefillRequest) -> Dict:
    """
    Add credits to the system
    
    Args:
        request: Refill request with amount and optional admin_id/description
    
    Returns:
        Result of refill operation
    """
    try:
        logger.info(f"💰 Refilling credits: +{request.amount}")
        
        if request.amount <= 0:
            raise HTTPException(status_code=400, detail="Amount must be positive")
        
        result = await credits_manager.refill_credits(
            amount=request.amount,
            admin_id=request.admin_id,
            description=request.description
        )
        
        if not result.get('success'):
            raise HTTPException(status_code=500, detail=result.get('error'))
        
        logger.info(f"✅ Credits refilled successfully: {result.get('balance_after')}")
        
        return result
        
    except HTTPException:
        raise
    except Exception as e:
        logger.error(f"❌ Error refilling credits: {str(e)}")
        raise HTTPException(status_code=500, detail=str(e))


@router.get("/transactions")
async def get_credit_transactions(
    limit: int = 50,
    offset: int = 0
) -> Dict:
    """
    Get credit transaction history
    
    Args:
        limit: Number of transactions to return (default: 50)
        offset: Offset for pagination (default: 0)
    
    Returns:
        List of credit transactions
    """
    try:
        logger.info(f"📜 Fetching credit transactions (limit={limit}, offset={offset})")
        
        if limit < 1 or limit > 100:
            raise HTTPException(status_code=400, detail="Limit must be between 1 and 100")
        
        result = await credits_manager.get_credit_transactions(limit=limit, offset=offset)
        
        if not result.get('success'):
            raise HTTPException(status_code=500, detail=result.get('error'))
        
        logger.info(f"✅ Found {result.get('count')} transactions")
        
        return result
        
    except HTTPException:
        raise
    except Exception as e:
        logger.error(f"❌ Error fetching transactions: {str(e)}")
        raise HTTPException(status_code=500, detail=str(e))


@router.get("/stats")
async def get_credits_stats(days: int = 30) -> Dict:
    """
    Get credits usage statistics
    
    Args:
        days: Number of days to analyze (default: 30)
    
    Returns:
        Credits usage statistics and estimates
    """
    try:
        logger.info(f"📊 Fetching credits stats for last {days} days")
        
        if days < 1 or days > 365:
            raise HTTPException(status_code=400, detail="Days must be between 1 and 365")
        
        result = await credits_manager.get_credits_stats(days=days)
        
        if not result.get('success'):
            raise HTTPException(status_code=500, detail=result.get('error'))
        
        logger.info(f"✅ Stats generated successfully")
        
        return result
        
    except HTTPException:
        raise
    except Exception as e:
        logger.error(f"❌ Error fetching stats: {str(e)}")
        raise HTTPException(status_code=500, detail=str(e))


@router.get("/usage")
async def get_message_costs(
    limit: int = 50,
    offset: int = 0,
    message_type: Optional[str] = None
) -> Dict:
    """
    Get message costs history
    
    Args:
        limit: Number of records to return (default: 50)
        offset: Offset for pagination (default: 0)
        message_type: Filter by message type (optional)
    
    Returns:
        List of message costs
    """
    try:
        logger.info(f"💰 Fetching message costs (limit={limit}, offset={offset}, type={message_type})")
        
        if limit < 1 or limit > 100:
            raise HTTPException(status_code=400, detail="Limit must be between 1 and 100")
        
        query = supabase.table('message_costs').select('*')
        
        if message_type:
            query = query.eq('message_type', message_type)
        
        response = query.order('created_at', desc=True) \
            .range(offset, offset + limit - 1) \
            .execute()
        
        logger.info(f"✅ Found {len(response.data)} message cost records")
        
        return {
            "success": True,
            "count": len(response.data),
            "costs": response.data
        }
        
    except Exception as e:
        logger.error(f"❌ Error fetching message costs: {str(e)}")
        raise HTTPException(status_code=500, detail=str(e))

